from django.db import models
from django.db.models import Q
from django.utils.translation import ugettext_lazy as _

from plan.roles.querysets import RoleQueryset
from plan.staff.base import I18nModel


MAX_NAME_LEN = 255


class RoleScope(I18nModel):
    slug = models.CharField(
        max_length=MAX_NAME_LEN,
        unique=True,
    )
    name = models.CharField(
        max_length=MAX_NAME_LEN,
    )
    name_en = models.CharField(
        max_length=MAX_NAME_LEN,
    )

    protected = models.BooleanField(
        verbose_name=_('Защищенный скоуп, в котором пользователям нельзя заводить кастомные роли.'),
        default=False
    )

    utility_scope = models.BooleanField(
        verbose_name='Служебный скоуп',
        default=False
    )

    can_issue_at_duty_time = models.BooleanField(
        default=True,
        verbose_name=_('Роли этого скоупа можно выдавать на время дежурства'),
    )

    class Meta:
        app_label = 'roles'

    def __str__(self):
        return self.name


class Role(I18nModel):
    EXCLUSIVE_OWNER = 'product_head'
    DEPUTY_OWNER = 'product_deputy_head'
    RESPONSIBLE = 'responsible'
    DUTY = 'duty'
    RESPONSIBLE_FOR_DUTY = 'responsible_for_duty'
    SERVICE_FINANCIAL_RESOURCES_REQUESTER = 'financial_resources_requester'
    HEADS = {
        EXCLUSIVE_OWNER,
        DEPUTY_OWNER,
    }
    RESPONSIBLES = HEADS | {RESPONSIBLE}
    RESPONSIBLE_FOR_DUTY_OR_SERVICE = RESPONSIBLES | {RESPONSIBLE_FOR_DUTY}
    CAN_NOT_USE_FOR_DUTY = RESPONSIBLES
    EXCLUDE_FROM_DUTY = CAN_NOT_USE_FOR_DUTY | {RESPONSIBLE_FOR_DUTY}

    name = models.CharField(
        max_length=MAX_NAME_LEN,
        db_index=True,
    )
    name_en = models.CharField(
        max_length=MAX_NAME_LEN,
        db_index=True,
    )
    projected_role = models.ForeignKey('self', null=True, blank=True, default=None, related_name='+')
    service = models.ForeignKey(
        'services.Service', null=True, blank=True, db_index=True
    )

    scope = models.ForeignKey(
        RoleScope,
        related_name='roles',
        null=True, blank=True, default=None,
        on_delete=models.SET_NULL,
    )
    position = models.IntegerField(
        db_index=True,
        default=220,
    )
    weight = models.PositiveSmallIntegerField(
        default=0,
        verbose_name=_('Важность роли'),
    )
    code = models.CharField(max_length=32, default='other', db_index=True)

    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)
    is_exportable = models.BooleanField(
        default=True,
        verbose_name=_('Роль которая экспортируется в стафф'),
    )

    objects = RoleQueryset.as_manager()

    @classmethod
    def get_exclusive_owner(cls):
        return cls.objects.globalwide().get(code=cls.EXCLUSIVE_OWNER)

    @classmethod
    def get_deputy_owner(cls):
        return cls.objects.globalwide().get(code=cls.DEPUTY_OWNER)

    @classmethod
    def get_responsible(cls):
        return cls.objects.globalwide().get(code=cls.RESPONSIBLE)

    @property
    def is_owner(self):
        return self.code in self.HEADS

    @property
    def is_exclusive_owner(self):
        return self.code == self.EXCLUSIVE_OWNER

    def __str__(self):
        return self.name

    def __repr__(self):
        return 'Role %s' % self.id

    class Meta:
        app_label = 'roles'
        ordering = ('position', 'name')
        unique_together = (('name', 'service'), ('name_en', 'service'))


def merge_allowed_roles(roles_queryset, scopes_queryset, service=None) -> RoleQueryset:
    if service is None:
        service_q = Q(service=None)
    else:
        service_q = Q(service=service) | Q(service=None)
    roles = Role.objects.none()
    if roles_queryset:
        roles |= Role.objects.filter(id__in=roles_queryset.all())
    if scopes_queryset:
        roles |= Role.objects.filter(scope_id__in=scopes_queryset.all())
    return roles.filter(service_q).distinct()
