from django.db import models
from django.forms import models as forms_models
from django.utils import translation

from plan import exceptions
from plan.api.idm import actions
from plan.lib.views import base
from plan.roles import forms
from plan.roles import models as roles_models
from plan.staff.utils import get_i_field


class ScopesView(base.UniversalView):
    def get(self, request, *args, **kwargs):
        scopes_qs = roles_models.RoleScope.objects.get_queryset()
        result = []
        for scope in scopes_qs:
            name = get_i_field(
                forms_models.model_to_dict(scope), 'name'
            )
            obj = {'id': scope.id, 'name': name}
            result.append(obj)
        return result


class CreateView(base.UniversalView):
    def post(self, request, *args, **kwargs):
        raw_serializer = forms.RoleCreateSerializer(data=self.json_data)
        if not raw_serializer.is_valid():
            raise exceptions.FormConflictError(raw_serializer)

        name_data = raw_serializer.data.pop('name')
        raw_data = raw_serializer.data.copy()
        raw_data['name'] = name_data['ru']
        raw_data['name_en'] = name_data['en']

        role_form = forms.RoleCreateForm(raw_data)
        if role_form.is_valid():
            name = role_form.cleaned_data['name']
            name_en = role_form.cleaned_data['name_en']
            custom_role_qs = roles_models.Role.objects.filter(
                models.Q(service=role_form.cleaned_data['service']) |
                models.Q(service__isnull=True)
            )
            custom_role_qs = custom_role_qs.filter(
                models.Q(name__iexact=name) | models.Q(name_en__iexact=name_en)
            )
            if custom_role_qs.exists():
                raise exceptions.ConflictError(
                    error_code='VALIDATION_ERROR',
                    message=translation.ugettext_lazy('Role name exists'),
                )
            role_obj = role_form.save()

            actions.add_role(role_obj)

            result_name = get_i_field(
                forms_models.model_to_dict(role_obj), 'name'
            )
            result = {
                'id': role_obj.id,
                'code': role_obj.code,
                'name': result_name,
                'scope': {'id': role_obj.scope_id, 'name': role_obj.scope.name}
            }
            return result
        else:
            raise exceptions.FormConflictError(role_form)
