from django.contrib import admin
from django.core.management import call_command

from plan.common.base_admin import ViewModelAdmin
from plan.duty.tasks import send_notifications_problems
from plan.services import tasks, models as services_models


class ReadOnlyMixin(object):
    def get_readonly_fields(self, request, obj=None):
        return [field.name
                for field in self.model._meta.fields + self.model._meta.many_to_many]


class ServiceContactInlineAdmin(admin.TabularInline):
    model = services_models.ServiceContact
    extra = 0


@admin.register(services_models.ServiceContact)
class ServiceContactAdmin(ViewModelAdmin):
    list_display = ('type', 'content', 'title', 'comment')
    list_filter = ('type',)


@admin.register(services_models.ServiceSuspiciousReason)
class ServiceSuspiciousReasonAdmin(ViewModelAdmin):
    list_display = ('reason', 'context',)
    list_filter = ('reason',)


class ServiceTagInlineAdmin(admin.TabularInline):
    model = services_models.Service.tags.through
    extra = 0


@admin.register(services_models.ServiceTag)
class ServiceTagAdmin(ViewModelAdmin):
    list_display = ('slug', 'name', 'name_en', 'color')

    fieldsets = (
        (None, {
            'fields': (
                'slug',
                'name',
                'name_en',
                'color',
                'is_protected',
            )
        }),
        ('Описание', {
            'classes': ('collapse',),
            'fields': (
                ('description', 'description_en'),
            )
        }),
    )


class ServiceSuspiciousReasonInlineAdmin(admin.TabularInline):
    model = services_models.ServiceSuspiciousReason
    raw_id_fields = ['marked_by']
    extra = 0


class ServiceTypeFunctionInlineAdmin(admin.TabularInline):
    model = services_models.ServiceType.functions.through
    extra = 0


class ServiceTypeParentsInlineAdmin(admin.TabularInline):
    model = services_models.ServiceType.available_parents.through
    fk_name = 'from_servicetype'
    extra = 0


@admin.register(services_models.ServiceType)
class ServiceTypeAdmin(ViewModelAdmin):
    list_display = ('code', 'name', 'name_en')
    inlines = [
        ServiceTypeFunctionInlineAdmin,
        ServiceTypeParentsInlineAdmin,
    ]
    exclude = ('functions', 'available_parents', )


@admin.register(services_models.ServiceTypeFunction)
class ServiceTypeFunctionAdmin(ViewModelAdmin):
    list_display = ('code', 'name', 'name_en')


@admin.register(services_models.Service)
class ServiceAdmin(ViewModelAdmin):
    inlines = [
        ServiceContactInlineAdmin,
        ServiceTagInlineAdmin,
        ServiceSuspiciousReasonInlineAdmin
    ]
    list_display = ('name', 'slug', 'state', 'is_exportable', 'suspicious_date', 'is_base')
    list_filter = ('state', 'is_exportable', 'tags__name', 'readonly_state', 'suspicious_date', 'is_base')
    search_fields = ('name', 'slug', 'state', 'tags__name', 'tags__name_en')
    filter_horizontal = ('base_of', 'related_services')

    actions = [
        'close_services',
        'fix_readonly_hangups',
        'get_kpi',
        'make_exportable',
        'make_unexportable',
        'move_to_root',
        'notify_staff',
        'update_fields',
        'recalculate_shifts_default',
        'recalculate_shifts_priority',
        'deep_disable_membership_inheritance',
        'send_notifications_problems',
        'recalculate_gradient_fields'
    ]

    def move_to_root(self, request, queryset):
        for service in queryset:
            if not service.parent:
                continue

            tasks.move_to_root(service, request.person)

    move_to_root.short_description = 'Переместить в корень (сделать метасервисами)'

    def fix_readonly_hangups(self, request, queryset):
        from plan.maintenance.tasks import clear_readonly

        service_ids = list(queryset.values_list('id', flat=True))
        clear_readonly.delay(service_ids=service_ids)

    fix_readonly_hangups.short_description = 'Попробовать вывести из read-only'

    def notify_staff(self, request, queryset):
        for service_id in queryset.values_list('id', flat=True):
            tasks.notify_staff.delay(service_id)

    notify_staff.short_description = 'Отправить на Стафф'

    def get_kpi(self, request, queryset):
        for service_id in queryset.values_list('id', flat=True):
            tasks.update_services_kpi.delay(service_id)

    get_kpi.short_description = 'Пересчитать kpi'

    def close_services(self, request, queryset):
        for service_id in queryset.values_list('id', flat=True):
            tasks.close_service_admin.delay(service_id)

    close_services.short_description = 'Закрыть сервисы'

    def make_exportable(self, request, queryset):
        for service in queryset:
            service.get_descendants(include_self=True).update(is_exportable=True)

    make_exportable.short_description = 'Сделать сервисы и их детей экспортируемыми'

    def make_unexportable(self, request, queryset):
        for service in queryset:
            service.get_descendants(include_self=True).update(is_exportable=False)

    make_unexportable.short_description = 'Сделать сервисы и их детей неэкспортируемыми'

    def update_fields(self, request, queryset):
        for service_id in queryset.values_list('id', flat=True):
            tasks.update_service_fields.delay(service_id)

    update_fields.short_description = 'Пересчитать денормализованные поля'

    def recalculate_shifts_default(self, request, queryset):
        from plan.duty.tasks import default_recalculate_shifts_for_service
        for service_id in queryset.values_list('pk', flat=True):
            default_recalculate_shifts_for_service.delay(service_id)

    recalculate_shifts_default.short_description = 'Пересчитать дежурства (очередь DEFAULT)'

    def recalculate_shifts_priority(self, request, queryset):
        from plan.duty.tasks import priority_recalculate_shifts_for_service
        for service_id in queryset.values_list('pk', flat=True):
            priority_recalculate_shifts_for_service.delay(service_id)

    recalculate_shifts_priority.short_description = 'Пересчитать дежурства (очередь PRIORITY)'

    def deep_disable_membership_inheritance(self, request, queryset):
        for service_id in queryset.values_list('id', flat=True):
            call_command('deep_disable_membership_inheritance', service_id=service_id)

    deep_disable_membership_inheritance.short_description = 'Отключить наследование во всем поддереве'

    def recalculate_gradient_fields(self, request, queryset):
        for service_id in queryset.values_list('id', flat=True):
            tasks.calculate_gradient_fields.delay(service_id)

    recalculate_gradient_fields.short_description = 'Пересчитать градиентные поля (вся ветка с потомками)'

    def send_notifications_problems(self, request, queryset):
        for service_id in queryset.values_list('id', flat=True):
            send_notifications_problems.delay(service_id)

    send_notifications_problems.short_description = 'Отправить письмо с проблемами в дежурствах'

    fieldsets = (
        (None, {
            'fields': (
                ('name', 'name_en', 'is_important', 'is_base'),
                'state',
                'slug',
                'service_type',
                'owner',
                'parent',
                'readonly_state',
                'is_exportable',
                'suspicious_date',
                'suspicious_notification_date',
                'membership_inheritance',
                'sandbox_move_date',
                'flags',
                'functions',
            )
        }),
        ('Связи', {
            'classes': ('collapse',),
            'fields': (
                'base_of',
                'departments',
                'related_services',
            )
        }),
        ('Описание', {
            'classes': ('collapse',),
            'fields': (
                ('description', 'description_en'),
                ('activity', 'activity_en'),
            )
        }),
        ('Дополнительные атрибуты', {
            'classes': ('collapse',),
            'fields': (
                'url',
                'keywords',
                'is_external',
                'kpi_bugs_count',
                'kpi_release_count',
                'kpi_lsr_count',
            )
        })
    )

    class Media:
        js = (
            '//js.static.yandex.net/jquery/1.3.2/_jquery.js',
            '//center.yandex-team.ru/media/js/_js-autocomplete_staff.js',
        )


class AllStatesFilterMixin:
    def get_queryset(self, request):
        qs = self.model.all_states.get_queryset()
        ordering = self.get_ordering(request)
        if ordering:
            qs = qs.order_by(*ordering)
        return qs


@admin.register(services_models.ServiceMemberDepartment)
class ServiceMemberDepartmentAdmin(AllStatesFilterMixin, ViewModelAdmin):
    list_display = ('service', 'department', 'role', 'state')
    search_fields = ('service__slug', 'service__name', 'department__name', 'department__url', 'role__name')
    list_filter = ('state',)


@admin.register(services_models.ServiceMember)
class ServiceMemberAdmin(AllStatesFilterMixin, ViewModelAdmin):
    list_display = ('service', 'staff', 'role', 'state')
    search_fields = ('service__slug', 'service__name', 'staff__login', 'role__name')
    list_filter = ('state', )


@admin.register(services_models.ServiceMoveRequest)
class ServiceMoveRequestAdmin(ViewModelAdmin):
    list_display = ('service', 'destination')
    search_fields = ('service__slug', 'destination__slug')
    list_filter = ('state',)


@admin.register(services_models.ServiceDeleteRequest)
class ServiceDeleteRequestAdmin(ViewModelAdmin):
    list_display = ('service', 'state')
    search_fields = ('service__slug', )
    list_filter = ('state',)


@admin.register(services_models.ServiceCloseRequest)
class ServiceCloseRequestAdmin(ViewModelAdmin):
    list_display = ('service', 'state')
    search_fields = ('service__slug', )
    list_filter = ('state', )


@admin.register(services_models.ServiceCreateRequest)
class ServiceCreateRequestAdmin(ViewModelAdmin):
    list_display = ('service', 'move_to')
    search_fields = ('service__slug', 'move_to__slug')
    list_filter = ('state',)


@admin.register(services_models.ServiceNotification)
class ServiceNotificationAdmin(ViewModelAdmin):
    list_display = ('service', 'recipient', 'notification_id', 'sent_at')
    search_fields = ('service__slug', 'recipient__login')
    list_filter = ('notification_id', )
    date_hierarchy = 'sent_at'
