import itertools
import logging
import collections
from rest_framework import serializers, viewsets
from django.utils.translation import ugettext_lazy as _
from rest_framework.response import Response

from plan.contacts.helpers import get_url_helper
from plan.api import base, forms
from plan.api.mixins import DefaultFieldsMixin, TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from plan.api.filters import CustomModelMultipleChoiceFilter
from plan.api.fields import (
    IntegerInterfacedSerializerField,
    SlugInterfacedSerializerField,
    MappingField,
)
from plan.api.exceptions import NotFound, ValidationError
from plan.contacts.controllers import ContactsController
from plan.contacts.models import ContactType
from plan.exceptions import ContactError
from plan.history.mixins import HistoryMixin
from plan.services.models import Service, ServiceContact
from plan.swagger import SwaggerServices

log = logging.getLogger(__name__)


class ContactTitleField(serializers.CharField):
    def to_representation(self, value):
        return {
            'ru': value,
            'en': '',
        }

    def to_internal_value(self, value):
        if isinstance(value, dict):
            value = value.get('ru')
        return super().to_internal_value(value)


class ContactSerializer(base.ModelSerializer):
    service = base.CompactServiceSerializer()
    title = MappingField({'ru': 'title', 'en': 'title_en'})
    type = base.ContactTypeSerializer()
    url = serializers.SerializerMethodField()

    def get_url(self, obj):
        return get_url_helper(obj.type.validator).denormalize(
            content=obj.content,
        )

    class Meta:
        model = ServiceContact
        fields = (
            'id', 'type', 'title',
            'content', 'service',
            'created_at', 'modified_at',
            'url',
        )


class OldContactSerializer(ContactSerializer):
    title = serializers.CharField()
    type = serializers.SerializerMethodField()

    class Meta:
        model = ServiceContact
        fields = (
            'id',
            'title',
            'content',
            'comment',
            'url',
            'type',
        )

    def get_type(self, obj):
        return obj.type.code


def legacy_serialize_contacts(contacts_queryset):
    grouping = itertools.groupby(contacts_queryset.order_by('type__code'),
                                 lambda c: c.type.code)

    return {
        contact_type: [OldContactSerializer(contact).data for contact in contacts_list]
        for contact_type, contacts_list
        in grouping
    }


class CreateContactSerializer(base.ModelSerializer):
    type = IntegerInterfacedSerializerField(
        queryset=ContactType.objects.all(),
        serializer=base.ContactTypeSerializer,
    )
    title = ContactTitleField(required=True)
    title_en = serializers.CharField(required=False, default='')

    class Meta:
        model = ServiceContact
        fields = 'id', 'type', 'title', 'title_en', 'content', 'service',


class ContactReplaceSerializer(serializers.Serializer):
    content = serializers.CharField()
    title = serializers.CharField()
    title_en = serializers.CharField(default='')
    contact_type = SlugInterfacedSerializerField(
        queryset=ContactType.objects.all(),
        serializer=base.ContactTypeSerializer,
        slug_field='code',
    )


class ContactsReplaceSerializer(serializers.Serializer):
    contacts = ContactReplaceSerializer(many=True)
    service_id = serializers.CharField(required=False)

    def validate(self, attrs):
        is_valid = True
        errors = []
        for i, item in enumerate(attrs.get('contacts')):
            error = {}
            try:
                ContactsController(None).normalize(item)
            except ContactError:
                is_valid = False
                error = {'content': _('Неверное значение контакта')}
            errors.append(error)

        if not is_valid:
            raise ValidationError(
                message={
                    'ru': 'Неверное значение контакта',
                    'en': 'Invalid contact value'
                },
                extra={'contacts': errors},
            )
        return attrs


class ContactsFilter(base.ServiceDescendantsFilterSet):
    service = CustomModelMultipleChoiceFilter(
        field_name='service',
        queryset=Service.objects.all(),
    )
    contact_type = CustomModelMultipleChoiceFilter(
        field_name='type',
        queryset=ContactType.objects.all(),
    )
    type__code = CustomModelMultipleChoiceFilter(
        field_name='type__code',
        to_field_name='code',
        queryset=ContactType.objects.all(),
    )

    class Meta:
        model = ServiceContact
        form = forms.ServiceFilterForm
        fields = {
            'contact_type': ['exact', 'in'],
            'content': ['contains'],
            'service': ['exact', 'in'],
            'service__is_exportable': ['exact', 'in'],
            'service__slug': ['exact', 'in'],
            'title': ['exact', 'in', 'contains'],
        }


class ContactsView(TvmAccessMixin, base.OrderingMixin, viewsets.ModelViewSet):
    """
    Контакты сервисов
    """
    serializer_class = ContactSerializer
    create_serializer_class = CreateContactSerializer
    filter_class = ContactsFilter
    queryset = ServiceContact.objects.select_related(
        'service', 'type',
    )
    permission_classes = [TvmAuthenticated]
    _permissions_to_proceed = 'view_contacts'

    def get_serializer_class(self):
        if self.action == 'create':
            return self.create_serializer_class

        return self.serializer_class


class V4ContactsView(DefaultFieldsMixin, ContactsView):
    default_swagger_schema = SwaggerServices

    pagination_class = base.ABCCursorPagination
    ordering_fields = ('id',)

    default_fields = [
        'content',
        'id',

        'service.id',
        'service.slug',

        'type.code',
    ]


class ContactsReplaceView(HistoryMixin, viewsets.GenericViewSet):
    _permissions_to_proceed = 'view_contacts'
    http_method_names = ['post']

    def _get_params(self, contact, service):
        return {
            'type_id': contact['contact_type']['id'],
            'title': contact['title'],
            'title_en': contact['title_en'],
            'content': contact['content'],
            'service': service,
        }

    def _get_data(self, pk_or_slug, request):
        serializer = ContactsReplaceSerializer(data=request.data)
        if not serializer.is_valid():
            raise ValidationError(extra=serializer.errors)
        if not pk_or_slug:
            pk_or_slug = serializer.data.get('service_id')
        try:
            service = Service.objects.get_by_pk_or_slug(pk_or_slug)
        except Service.DoesNotExist:
            raise NotFound(detail=_('Сервис не найден'))

        return service, serializer

    def _get_contacts(self, service, serializer):
        new_contacts = collections.OrderedDict()
        for item in serializer.data['contacts']:
            key = (item['contact_type']['id'], item['title'], item['title_en'], item['content'])
            new_contacts[key] = item

        existing_contacts = {}
        for c in service.contacts.all():
            key = (c.type_id, c.title, c.title_en, c.content)
            existing_contacts[key] = c

        return new_contacts, existing_contacts

    def _add_contacts(self, service, new_contacts, existing_contacts):
        to_add = []

        for position, (key, contact_data) in enumerate(new_contacts.items()):
            exist = False
            if key in existing_contacts:
                contact = existing_contacts[key]
                exist = True
            else:
                params = self._get_params(contact_data, service)
                contact = ServiceContact(**params)
                to_add.append(contact)

            if contact.position != position:
                contact.position = position
                if exist:
                    contact.save(update_fields=['position'])

        if to_add:
            ServiceContact.objects.bulk_create(to_add)

    def post(self, request, pk_or_slug):
        service, serializer = self._get_data(pk_or_slug, request)

        new_contacts, existing_contacts = self._get_contacts(service, serializer)
        new_keys = set(new_contacts.keys())
        old_keys = set(existing_contacts.keys())
        to_delete = old_keys - new_keys

        if to_delete:
            ServiceContact.objects.filter(
                pk__in=[existing_contacts[i].id for i in to_delete]
            ).delete()

        self._add_contacts(
            service=service,
            new_contacts=new_contacts,
            existing_contacts=existing_contacts,
        )
        self.create_history_entry(service)
        return Response(data={'content': {}, 'error': {}})
