import logging

from django_filters import CharFilter, Filter
from django_fsm import TransitionNotAllowed

from rest_framework import serializers, viewsets
from rest_framework.decorators import action
from rest_framework.metadata import BaseMetadata
from rest_framework.response import Response

from plan.api.base import ModelSerializer
from plan.api.exceptions import BadRequest
from plan.api.filters import CustomModelMultipleChoiceFilter, PlanFilterSet
from plan.api.serializers import CompactServiceSerializer, CompactStaffSerializer
from plan.api.mixins import OrderingMixin
from plan.services import permissions
from plan.services.api.moves import ApproversSerializer
from plan.services.api.permissions import ServicePermissions
from plan.services.models import ServiceCreateRequest, ServiceMember, Service
from plan.staff.models import Staff
from plan.swagger import SwaggerFrontend


log = logging.getLogger(__name__)


class CreateRequestSerializer(ModelSerializer):
    service = CompactServiceSerializer()
    move_to = CompactServiceSerializer()
    requester = CompactStaffSerializer()
    approver_incoming = CompactStaffSerializer()

    actions = serializers.SerializerMethodField()
    state_display = serializers.SerializerMethodField()

    def get_state_display(self, instance):
        return {
            'en': instance.state.title(),
            'ru': instance.get_state_display()
        }

    def get_actions(self, instance):
        request = self.context.get('request')
        if request is None:
            return []

        person = request.person
        actions = set()

        if permissions.can_approve_create_request(instance.service, person):
            actions |= set(['approve', 'reject'])

        return sorted(list(actions))

    class Meta:
        model = ServiceCreateRequest
        fields = (
            'id',
            'created_at',
            'service',
            'move_to',
            'state',
            'state_display',
            'requester',
            'approver_incoming',
            'actions',
        )


class PermissionsMetadata(BaseMetadata):
    def determine_metadata(self, request, view):
        view_permissions = []

        try:
            create_request = view.get_object()
            if permissions.can_approve_create_request(create_request.service, request.person):
                view_permissions.extend(['can_approve', 'can_cancel'])

        except AssertionError:  # выскочит если вызывают там, где нет инстанса
            pass

        return {'permissions': view_permissions}


class OnlyMineFilter(Filter):
    def with_descendants(self):
        with_descendants = self.parent.data.get('with_descendants')

        if with_descendants in ('1', 'True', 'true'):
            return True

        return False

    def filter(self, qs, value):
        """
        only_mine:
            если True, то показываем только те запросы, которые может подтвердить реквестер
            иначе: все
        with_descendants:
            дополнительный фильтр для only_mine,
            если True, то показвываем вместе с дочерними
            иначе: только прямые
        """

        only_mine = False
        if value in ('1', 'True', 'true', True):
            only_mine = True

        if not only_mine:
            return qs

        person = self.parent.request.person
        services = person.get_own_responsible_services(with_descendants=self.with_descendants())
        create_requests = ServiceCreateRequest.objects.approvable(services).values_list('id', flat=True)
        return qs.filter(pk__in=create_requests)


class CreateRequestFilter(PlanFilterSet):
    service = CustomModelMultipleChoiceFilter(
        queryset=Service.objects.all(),
        distinct=True,
    )
    move_to = CustomModelMultipleChoiceFilter(
        queryset=Service.objects.all(),
        distinct=True,
    )
    kind = CharFilter(method='filter_create_request_kind')
    requester = CustomModelMultipleChoiceFilter(
        field_name='requester__login',
        queryset=Staff.objects.active(),
        to_field_name='login',
        distinct=True,
    )
    only_mine = OnlyMineFilter()

    def filter_create_request_kind(self, queryset, name, value):
        if value == 'active':
            queryset = queryset.active()
        elif value == 'inactive':
            queryset = queryset.inactive()
        return queryset

    class Meta:
        model = ServiceCreateRequest
        fields = {
            'id': ['exact', 'in'],
            'state': ['exact', 'in'],
            'service': ['exact', 'in'],
            'move_to': ['exact', 'in'],
        }


class CreateRequestsView(OrderingMixin, viewsets.ModelViewSet):
    default_swagger_schema = SwaggerFrontend

    serializer_class = CreateRequestSerializer
    search_fields = ()
    http_method_names = ['get', 'post', 'options']
    queryset = ServiceCreateRequest.objects.select_related(
        'service',
        'move_to',
        'approver_incoming',
    )
    permission_classes = [ServicePermissions]
    metadata_class = PermissionsMetadata
    filter_class = CreateRequestFilter
    ordering_fields = ()

    @action(methods=['post'], detail=True)
    def reject(self, request, pk=None):
        create_request = self.get_object()
        try:
            create_request.reject(rejector=self.request.person)
        except TransitionNotAllowed:
            raise BadRequest(message={
                'ru': 'Решение по этому запросу уже было принято',
                'en': 'This request was already processed',
            })

        return Response(status=204)

    @action(methods=['post'], detail=True)
    def approve(self, request, pk=None):
        create_request = self.get_object()
        try:
            create_request.approve(approver=self.request.person)
        except TransitionNotAllowed:
            raise BadRequest(message={
                'ru': 'Решение по этому запросу уже было принято',
                'en': 'This request was already processed',
            })

        return Response(status=204)

    @staticmethod
    def _get_approvers(service):
        recommended = []
        for approver in service.members.responsibles():
            if approver.staff not in recommended:
                recommended.append(approver.staff)

        other = []
        potential_approvers = (
            ServiceMember.objects
            .filter(service__in=service.get_ancestors())
            .responsibles()
            .order_by('service__level')
        )
        for approver in potential_approvers:
            if approver.staff not in other and approver.staff not in recommended:
                other.append(approver.staff)

        return {'recommended': recommended, 'other': other}

    @action(methods=['get'], detail=True)
    def approvers(self, request, pk=None):
        create_request = self.get_object()

        move_to_approvers_serializer = ApproversSerializer(self._get_approvers(create_request.move_to))

        return Response(
            {
                'parent_approvers': move_to_approvers_serializer.data
            }
        )
