import logging

from django.core.validators import MinValueValidator
from django.conf import settings
from rest_framework import serializers, viewsets, status
from rest_framework.decorators import action
from rest_framework.generics import get_object_or_404
from rest_framework.response import Response

from plan.api import base, validators
from plan.api.exceptions import Conflict
from plan.api.filters import CustomModelMultipleChoiceFilter
from plan.api.idm import actions
from plan.api.mixins import DefaultFieldsMixin, TvmAccessMixin
from plan.api.serializers import BasePlanModelSerializer
from plan.idm import exceptions as idm_exceptions
from plan.roles.models import Role
from plan.services.api.permissions import ServiceDepartmentMemberPermissions
from plan.services.models import Service, ServiceMember, ServiceMemberDepartment
from plan.services.exceptions import ServiceReadonlyError
from plan.services.state import SERVICEMEMBER_STATE
from plan.staff.models import Department, Staff
from plan.services.helpers import get_expires_at
from plan.swagger import SwaggerServices, SwaggerFrontend

log = logging.getLogger(__name__)


class DepartmentMemberSerializer(base.ModelSerializer):
    department = base.CompactDepartmentSerializer()
    service = base.CompactServiceSerializer()
    role = base.CompactRoleSerializer()

    class Meta:
        model = ServiceMemberDepartment
        fields = (
            'id', 'department', 'service',
            'role', 'created_at', 'modified_at',
        )


class FrontendDepartmentMemberSerializer(DepartmentMemberSerializer):
    members_count = serializers.IntegerField(read_only=True)

    class Meta:
        model = ServiceMemberDepartment
        fields = DepartmentMemberSerializer.Meta.fields + ('members_count', )


class CreateDepartmentMemberSerializer(BasePlanModelSerializer):
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.alive(),
        required=True,
    )
    comment = serializers.CharField(required=False, default='')
    silent = serializers.BooleanField(required=False, default=False)
    deprive_after = serializers.IntegerField(
        required=False,
        validators=[MinValueValidator(1)]
    )
    deprive_at = serializers.DateField(required=False)

    def to_internal_value(self, data):
        updated_data = data.copy()
        if 'department' not in data and 'staff_department' in data:
            department = get_object_or_404(Department.objects.exclude(staff_id=0), staff_id=data['staff_department'])
            updated_data['department'] = department.id
        return super(CreateDepartmentMemberSerializer, self).to_internal_value(updated_data)

    class Meta:
        model = ServiceMemberDepartment
        fields = (
            'service', 'department', 'role',
            'deprive_after', 'deprive_at',
            'comment', 'silent',
        )
        validators = [
            validators.MutuallyExclusiveFieldsValidator(('deprive_at', 'deprive_after'))
        ]


class DepartmentMemberFilter(base.ServiceDescendantsFilterSet):
    department = CustomModelMultipleChoiceFilter(
        field_name='department',
        queryset=Department.objects.all(),
        distinct=False,
    )

    class Meta:
        model = ServiceMemberDepartment
        fields = {
            'service': ['exact', 'in'],
            'service__slug': ['exact', 'in'],
            'department': ['exact', 'in'],
        }


class DepartmentsView(base.OrderingMixin, viewsets.ModelViewSet):
    """
    Групповые роли в сервисах
    """
    serializer_class = DepartmentMemberSerializer
    filterset_class = DepartmentMemberFilter
    http_method_names = ['get', 'post', 'delete']
    queryset = ServiceMemberDepartment.objects.all()
    _permissions_to_proceed = 'view_own_services'

    def get_serializer_class(self):
        if self.action == 'create':
            return CreateDepartmentMemberSerializer

        return DepartmentMemberSerializer

    @staticmethod
    def check_active_role(service: Service, department: Department, role: Role) -> bool:
        return ServiceMemberDepartment.objects.filter(
            service=service,
            department=department,
            role=role,
            state=SERVICEMEMBER_STATE.ACTIVE
        ).exists()

    def create(self, request, *args, **kwargs):
        """
        Заявка на новую групповую роль
        """
        requester = request.user.staff
        serializer = self.get_serializer(data=request.data)

        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data

        if data['service'].readonly_state is not None:
            raise ServiceReadonlyError(data['service'])

        if self.check_active_role(service=data['service'], department=data['department'], role=data['role']):
            raise Conflict(
                detail='Role already exists',
                message={
                    'en': f'Group \"{data["department"].name_en}\" already has this role',
                    'ru': f'У группы \"{data["department"].name}\" уже есть такая роль',
                }
            )

        if (result := self._request_membership(requester, data)) is not None:
            expires_at = get_expires_at(data)
            self._create_membership(data, result['id'], expires_at=expires_at)

        headers = self.get_success_headers(serializer.data)
        return Response(status=status.HTTP_201_CREATED, headers=headers)

    def _create_membership(self, data, idm_role_id, expires_at):
        service = data['service']
        department = data['department']
        role = data['role']
        member_department, created = ServiceMemberDepartment.all_states.get_or_create(
            service=service, role=role, department=department,
        )
        member_department.request(idm_role_id, expires_at=expires_at)

        self._create_members(member_department, service, department, role, expires_at=expires_at)

    def _create_members(self, member_department, service, department, role, expires_at):
        departments_id_list = department.get_descendants(include_self=True).values_list('id', flat=True)
        staff_id_list = set(
            Staff.objects.filter(is_dismissed=False, department_id__in=departments_id_list).values_list('id', flat=True)
        )
        existing_members = ServiceMember.all_states.filter(
            service=service, from_department=member_department, role=role, staff_id__in=staff_id_list)
        existing_members.exclude(state=SERVICEMEMBER_STATE.ACTIVE).update(
            state=SERVICEMEMBER_STATE.REQUESTED, expires_at=expires_at,
        )

        existing_staff_id_list = set(existing_members.values_list('staff_id', flat=True))
        missing_staff_id_list = staff_id_list - existing_staff_id_list

        missing_members = []
        for staff_id in missing_staff_id_list:
            missing_members.append(
                ServiceMember(
                    service=service, staff_id=staff_id,
                    role=role, from_department=member_department,
                    expires_at=expires_at,
                )
            )
        if missing_members:
            ServiceMember.objects.bulk_create(missing_members)

    def _request_membership(self, requester, data):
        try:
            result = actions.request_membership(
                data['service'],
                data['department'],
                data['role'],
                requester=requester,
                deprive_after=data.get('deprive_after'),
                deprive_at=data.get('deprive_at'),
                comment=data['comment'],
                silent=data['silent'],
                timeout=settings.IDM_POST_ROLES_TIMEOUT,
                retry=settings.ABC_IDM_FROM_API_RETRY,
            )

        except idm_exceptions.Conflict as e:
            if self.check_active_role(service=data['service'], department=data['department'], role=data['role']):
                return None
            raise Conflict(
                message=e.message,
                detail=e.detail,
                extra=e.extra,
            )

        return result

    def destroy(self, request, *args, **kwargs):
        """
        Отзыв групповой роли
        """
        obj = self.get_object()
        requester = self.request.user.staff

        log.info('%s asked to remove department membership %s from %r', requester.login, obj, obj.service)

        self._deprive_role(requester, obj)
        self._deprive_membership(obj)

        return Response(status=status.HTTP_204_NO_CONTENT)

    def _deprive_role(self, requester, obj):
        actions.deprive_role(
            member=obj,
            requester=requester,
            timeout=settings.IDM_DELETE_ROLES_TIMEOUT,
            retry=settings.ABC_IDM_FROM_API_RETRY,
        )

    def _deprive_membership(self, obj):
        obj.members.set_depriving_state()
        obj.set_depriving_state()

    @action(methods=['post'], detail=True)
    def rerequest(self, request, pk=None):
        """
        Перезапрос устаревающей групповой роли
        """
        obj = self.get_object()
        requester = self.request.user.staff

        actions.rerequest_role(
            service_member=obj,
            requester=requester,
            timeout=settings.IDM_POST_ROLES_TIMEOUT,
            retry=settings.ABC_IDM_FROM_API_RETRY,
        )
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(methods=['post'], detail=False)
    def approve(self, request):
        """
        Подтверждение групповой роли
        """
        idm_role_id = request.data['idm_role']

        actions.approve_role(
            role_id=idm_role_id,
            requester=request.user.staff,
            timeout=settings.IDM_POST_ROLES_TIMEOUT,
            retry=settings.ABC_IDM_FROM_API_RETRY,
        )
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(methods=['post'], detail=False)
    def decline(self, request):
        """
        Отклонение групповой роли
        """
        idm_role_id = request.data['idm_role']

        actions.decline_role(
            role_id=idm_role_id,
            requester=request.user.staff,
            timeout=settings.IDM_POST_ROLES_TIMEOUT,
            retry=settings.ABC_IDM_FROM_API_RETRY,
        )
        return Response(status=status.HTTP_204_NO_CONTENT)


class V4DepartmentsView(DefaultFieldsMixin, DepartmentsView, TvmAccessMixin):
    """
    **Роли** для подразделения в сервисе
    """
    default_swagger_schema = SwaggerServices

    permission_classes = [ServiceDepartmentMemberPermissions]
    default_fields = [
        'id',
        'members_count',

        'department.url',

        'role.code',
        'role.scope.slug',

        'service.id',
        'service.slug',
    ]


class FrontendDepartmentsView(V4DepartmentsView):
    default_swagger_schema = SwaggerFrontend

    def get_queryset(self):
        queryset = self.queryset

        return queryset

    def get_serializer_class(self):
        if self.action == 'create':
            return CreateDepartmentMemberSerializer

        return FrontendDepartmentMemberSerializer
