from collections import defaultdict

import django_filters
import logging
import six

from io import BytesIO

from django import forms
from django.conf import settings
from django.forms.forms import Form
from django.http.request import QueryDict
from django.shortcuts import get_object_or_404
from django.utils.functional import cached_property

from rest_framework import serializers, fields, viewsets, mixins
from rest_framework.response import Response
from rest_framework.generics import get_object_or_404 as drf_get_object_or_404
from rest_framework.decorators import action

from plan.services.constants.action import MEMBER_FIELDS, DEFAULT_ACTIONS
from plan.api.base import ABCLargeCursorPagination
from plan.api.exceptions import BadRequest, TooManyServices
from plan.api.fields import MappingField
from plan.api.mixins import FieldsMappingMixin, TvmAccessMixin, OrderingMixin
from plan.common.utils.collection.mapping import dict_to_querydict
from plan.common.utils.xls import make_attachment_response
from plan.oebs.api.serializers import OEBSAgreementSerializer
from plan.oebs.models import OEBSAgreement
from plan.services.api.contacts import V4ContactsView, ContactsFilter, ContactsReplaceView
from plan.services.api.members import V4MembersView, MemberFilter
from plan.services.api.permissions import ServicePermissions
from plan.services.api.responsibles import V4ResponsiblesView, ResponsibleFilter
from plan.services.api.services import (
    ServicesView,
    ServiceFilter,
    ListServiceSerializer,
    ServiceTagSerializer,
    DetailServiceSerializer,
    V4CreateServiceSerializer,
)
from plan.services.constants.api_frontend import (
    MATCHED_LABEL,
    ROOT_KEY,
    FRONTEND_DEEP_FILTERS,
    DEEP_FILTER_MODE_LABEL,
    SHALLOW_FILTER_MODE_LABEL,
    FrontendServiceSerializerFields,
    FrontendTagsSerializerField,
)
from plan.services.models import Service, ServiceTag
from plan.contacts.models import ContactType
from plan.services.utils.xls import build_xlsx_workbook
from plan.staff.constants import LANG
from plan.suspicion.api.serializers import ServiceTrafficStatusSerializer, IssueGroupSerializer
from plan.suspicion.constants import FrontendTrafficSerializerFields
from plan.suspicion.models import ServiceTrafficStatus
from plan.api.filters import CustomModelChoiceFilter
from plan.services.permissions import (
    is_service_responsible,
    can_close_service,
    can_delete_service,
)
from plan.api.serializers import ContactTypeSerializer
from plan.swagger import SwaggerFrontend

ZERO_VALUE = object()
logger = logging.getLogger(__name__)


def is_deep_mode(request):
    return any(key in FRONTEND_DEEP_FILTERS for key in request.GET.keys())


def parent_list(value):
    parents = []
    for parent in value.split(','):
        parent = parent.strip()
        if not parent.isdigit():
            raise BadRequest(message={
                'ru': 'неправильный параметр фильтра parents',
                'en': 'invalid parameter for parents filter',
            })
        parents.append(int(parent))
    return parents


class MaybeZeroField(forms.ModelChoiceField):
    zero_values = ['0', 0]

    def to_python(self, value):
        if value in self.zero_values:
            return ZERO_VALUE
        return super(MaybeZeroField, self).to_python(value)


class MaybeZeroModelChoiceFilter(CustomModelChoiceFilter):
    field_class = MaybeZeroField

    def filter(self, qs, value):
        if value not in (ZERO_VALUE, None):
            qs = qs.children_of(value.pk)
        return qs


class FrontendServiceFilter(ServiceFilter):
    parents = django_filters.CharFilter(method='filter_parents')
    owner = django_filters.CharFilter(method='filter_by_owner')
    root = MaybeZeroModelChoiceFilter(
        queryset=Service.objects.all(),
        to_field_name='id',
    )

    class Meta(ServiceFilter.Meta):
        form = Form
        fields = {
            'has_external_members': ['exact', 'in'],
            'id': ['exact', 'in', 'lt', 'gt'],
            'is_exportable': ['exact', 'in'],
            'is_suspicious': ['exact', 'in'],
            'member': ['exact'],
            'name': ['exact', 'contains'],
            'name_en': ['exact', 'contains'],
            'readonly_state': ['exact', 'in'],
            'slug': ['exact', 'contains', 'in'],
            'state': ['exact', 'in'],
            'tags': ['exact'],
            'tags__id': ['in'],
            'tags__slug': ['exact', 'in'],
            'type': ['exact', 'in'],
            'membership_inheritance': ['exact'],
        }

    def filter_parents(self, queryset, name, value):
        if value and not is_deep_mode(self.request):
            queryset = queryset.by_parent_list(parent_list(value))
        return queryset

    def filter_by_owner(self, queryset, name, value):
        if value:
            queryset = queryset.filter(owner__login=value)
        return queryset


class FrontendTrafficSerializer(ServiceTrafficStatusSerializer):
    default_swagger_schema = SwaggerFrontend

    group = fields.SerializerMethodField(read_only=True)

    def get_group(self, traffic):
        group = {
            key_of_group: traffic[key_of_traffic]
            for key_of_group, key_of_traffic in FrontendTrafficSerializerFields.GROUP_FIELDS.items()
        }
        return IssueGroupSerializer(group).data


class FrontendOwnerSerializer(FieldsMappingMixin, serializers.Serializer):
    login = fields.CharField(source='owner__login')
    name = serializers.SerializerMethodField()

    fields_mapping_ = {
        'login': ('login',),
        'name': ('first_name', 'last_name', 'first_name_en', 'last_name_en'),
    }

    class Meta:
        fields = [
            'login',
            'name',
        ]

    def get_name(self, obj):
        return {
            'ru': ('%s %s' % (obj['owner__first_name'], obj['owner__last_name'])).strip(),
            'en': ('%s %s' % (obj['owner__first_name_en'], obj['owner__last_name_en'])).strip(),
        }


class FrontendServiceTypeSerializer(FieldsMappingMixin, serializers.Serializer):
    code = fields.CharField(source='service_type__code')
    id = fields.IntegerField(source='service_type__id')
    name = serializers.SerializerMethodField()

    fields_mapping_ = {
        'code': ('code',),
        'name': ('name_en', 'name', ),
    }

    class Meta:
        fields = [
            'id',
            'code',
            'name',
        ]

    def get_name(self, obj):
        return {
            'ru': obj['service_type__name'],
            'en': obj['service_type__name_en'],
        }


class FrontendServiceListSerializer(ListServiceSerializer):
    traffic_light = fields.SerializerMethodField(read_only=True)
    matched = fields.BooleanField(read_only=True)
    owner = FrontendOwnerSerializer(source='*')
    parent = fields.SerializerMethodField(read_only=True)
    tags = fields.SerializerMethodField(read_only=True)
    type = FrontendServiceTypeSerializer(read_only=True, source='*')

    # ToDo: Поправить, когда будем делать https://st.yandex-team.ru/ABC-8007, и убрать SerializerMethodField
    fields_mapping_ = {'matched': [], 'traffic_light': [], 'parent': [], 'tags': [], 'type': []}

    def __init__(self, *args, **kwargs):
        self._traffic_lights_source = kwargs.pop('traffic_lights', None)
        self._tags_source = kwargs.pop('tags', None)
        super(FrontendServiceListSerializer, self).__init__(*args, **kwargs)

    def get_traffic_light(self, service):
        traffics = sorted(self._traffic_lights_source[service['id']], key=lambda traffic: traffic['issue_group__code'])
        return FrontendTrafficSerializer(traffics, many=True).data

    def get_tags(self, service):
        tags = self._tags_source[service['id']]
        return ServiceTagSerializer(tags, many=True).data

    def get_parent(self, service):
        parent_id = service['parent_id']
        if parent_id:
            return {'id': parent_id}

    class Meta(ListServiceSerializer.Meta):
        fields = FrontendServiceSerializerFields.META_FIELDS + (
            'type', 'membership_inheritance',
        )


class FrontendServiceDetailSerializer(DetailServiceSerializer):
    oebs_agreement = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()
    available_states = serializers.SerializerMethodField()
    flags = serializers.JSONField()
    functions = serializers.JSONField(read_only=True)

    fields_mapping_ = {
        'issue': ('id',),
        'human_readonly_state': ('id',),
        'oebs_agreement': ('id',),
        'actions': ('id',),
        'available_states': ('id', ),
    }

    class Meta:
        model = Service
        fields = DetailServiceSerializer.Meta.fields + (
            'human_readonly_state',
            'oebs_agreement',
            'use_for_hardware',
            'use_for_hr',
            'use_for_procurement',
            'use_for_revenue',
            'use_for_group_only',
            'actions',
            'available_states',
            'is_important',
            'flags',
            'functions',
        )

    @staticmethod
    def get_oebs_agreement(service):
        oebs_agreement = OEBSAgreement.objects.filter(service=service).last()
        return OEBSAgreementSerializer(oebs_agreement).data if oebs_agreement else None

    def get_available_states(self, obj):
        person = self.context['request'].person
        unavailable_states = {obj.state}
        if not can_delete_service(obj, person):
            unavailable_states.add(Service.states.DELETED)
        if not can_close_service(obj, person):
            unavailable_states.add(Service.states.CLOSED)
        if obj.state not in Service.states.NORMAL_STATES and obj.has_change_state_execution():
            unavailable_states.update(Service.states.NORMAL_STATES)

        return [
            {
                'id': state,
                'verbose': str(Service.states[state].name)
            }
            for state in Service.states
            if state not in unavailable_states
        ]

    def get_actions(self, obj):
        person = self.context['request'].person
        all_possible_actions = [
            {
                action: {'verb': verb}
            }
            for action, verb in DEFAULT_ACTIONS.items()
        ]

        is_responsible = is_service_responsible(obj, person)

        if is_responsible:
            all_possible_actions.extend([
                {'member_approve': {'verb': 'Подтвердить участие'}},
                {'member_decline': {'verb': 'Отклонить участие'}},
                {'department_approve': {'verb': 'Отклонить участие'}},
                {'department_decline': {'verb': 'Отклонить подразделение'}},
                {'chown_approve': {'verb': 'Подтвердить изменение руководителя'}},
                {'delete': {'verb': 'Удалить сервис'}},
            ])

        if obj.readonly_state is not None:
            for possible_action in list(all_possible_actions):
                if list(possible_action.keys())[0] in MEMBER_FIELDS:
                    all_possible_actions.remove(possible_action)

        # редирект на страницу создания проекта.
        # Но для фронта это action, поэтому такой костыль
        fake_actions = []
        if obj.state != 'deleted':
            fake_actions.extend([
                {'request_resource': {'verb': 'Запросить ресурс'}},
            ])

            if is_responsible:
                fake_actions.append({
                    'provide_resource': {'verb': 'Предоставить ресурс'},
                })

        return all_possible_actions + fake_actions


class FrontendServiceSerializer(V4CreateServiceSerializer):
    description = MappingField(
        {
            'ru': 'description',
            'en': 'description_en',
        },
        required=True,
    )


class FrontendServicesView(ServicesView):
    default_swagger_schema = SwaggerFrontend

    filter_class = FrontendServiceFilter
    pagination_class = ABCLargeCursorPagination
    list_serializer = FrontendServiceListSerializer
    detail_serializer = FrontendServiceDetailSerializer
    create_serializer = FrontendServiceSerializer
    _permissions_to_proceed = 'view_own_services'

    def get_object(self):
        """
        Копия оригинального метода, отличие - в поддержке получения данных
        и по слагу сервиса тоже
        """
        queryset = self.filter_queryset(self.get_queryset())
        lookup_url_kwarg = self.lookup_url_kwarg or self.lookup_field

        assert lookup_url_kwarg in self.kwargs, (
            'Expected view %s to be called with a URL keyword argument '
            'named "%s". Fix your URL conf, or set the `.lookup_field` '
            'attribute on the view correctly.' %
            (self.__class__.__name__, lookup_url_kwarg)
        )
        filter_value = self.kwargs[lookup_url_kwarg]
        lookup_field = self.lookup_field

        if not (isinstance(filter_value, int) or filter_value.isdigit()):
            # предполагаем тогда, что это слаг
            lookup_field = 'slug'

        filter_kwargs = {lookup_field: filter_value}
        obj = drf_get_object_or_404(queryset, **filter_kwargs)

        self.check_object_permissions(self.request, obj)

        return obj

    @property
    def queryset(self):
        qs = super().queryset
        if self.get_serializer_class() == self.list_serializer:
            qs = qs.alive()
        return qs

    @cached_property
    def ordering(self):
        user_language = getattr(self.request, 'LANGUAGE_CODE', self.request.person.staff.lang_ui)
        result_field = 'name_en'
        if user_language == LANG.RU:
            result_field = 'name'
        return 'level', result_field, 'id'

    def get_fields_to_ordering(self, service):
        fields = []
        for field in self.ordering:
            value = service[field]
            if isinstance(value, six.string_types):
                fields.append(value.lower())
            else:
                fields.append(value)
        return fields

    def max_service_count(self):
        return settings.MAX_SERVICES_COUNT_PER_FRONTEND_REQUEST

    def querysets_to_matched_values_and_pk_set(self, querysets_with_flags):
        result_services = []
        used_ids = set()
        total_length = 0
        for queryset, _ in querysets_with_flags:
            total_length += queryset.order_by().count()
            if total_length > self.max_service_count():
                raise TooManyServices()
        for queryset, matched in querysets_with_flags:
            for service in queryset.values(*FrontendServiceSerializerFields.FIELD_LIST_TO_SELECT):
                service_id = service['id']
                if service_id not in used_ids:
                    service[MATCHED_LABEL] = matched
                    result_services.append(service)
                    used_ids.add(service_id)
        return (result_services, used_ids)

    def services_with_acceptable_children(self):
        """
            Метод используется с "узкими" фильтрами.
            Возвращает выборку из сервисов:
            а) Сервисы подошли по всем фильтрам, matched проставляется True
            б) Сервисы подходят по фильтру parents, не подходят по остальным,
               но имеют детей, которые подходят под фильтры. matched проставляется False
            :return: (Список сервисов, сет из первичных ключей)
        """

        params = {key: val for key, val in self.request.GET.items()}
        parents = params.pop('parents', None)
        queryset = self.get_queryset()
        filtered_services = self.filter_queryset(queryset=queryset)
        result_querysets = [(filtered_services, True)]
        if parents:
            params = dict_to_querydict(params)
            no_parents_filtered_services = self.filter_class(params, queryset=queryset, request=self.request).qs
            filtered_only_by_parents = queryset.by_parent_list(parent_list(parents))
            parents_pks = Service._closure_model.objects.filter(
                child__in=no_parents_filtered_services,
                parent__in=filtered_only_by_parents,
            ).values('parent')
            # Включаем парентов из предыдущего запроса,
            # тех, что пофильтровали изначально исключаем в querysets_to_matched_values_and_pk_set
            not_matched = queryset.filter(pk__in=parents_pks)
            result_querysets.append((not_matched, False))
        return self.querysets_to_matched_values_and_pk_set(result_querysets)

    def services_with_parents(self, root_pk):
        """
            Используется с "широкими" фильтрами.
            Возвращает выборку из сервисов:
            а) Сервисы подошли по всем фильтрам, matched проставляется True
            б) Сервисы не подходящие по фильтрам, но являющиеся предками сервисов из пункта (а)
            :return: (Список сервисов, сет из первичных ключей)
        """
        filtered_services = self.filter_queryset(self.get_queryset())
        parents = filtered_services.get_parents()
        if root_pk not in MaybeZeroField.zero_values:
            parents = parents.children_of(root_pk)
        result_querysets = [
            (filtered_services, True),
            (parents, False),
        ]
        return self.querysets_to_matched_values_and_pk_set(result_querysets)

    def list(self, request, *args, **kwargs):
        user_can_use_filters = self.request.user.has_perm('internal_roles.can_filter_and_click')
        if not user_can_use_filters:
            # Если у пользователя нет доступа к фильтрам, то затираем все его query-параметры
            # всегда фильтруем только по сервисам, в которых он участник
            self._view_only_own_services(request)

        root_pk = request.GET.get(ROOT_KEY, '0')
        if is_deep_mode(request):
            filter_mode = DEEP_FILTER_MODE_LABEL
            services_list, ids = self.services_with_parents(root_pk)
        else:
            filter_mode = SHALLOW_FILTER_MODE_LABEL
            services_list, ids = self.services_with_acceptable_children()

        if root_pk not in MaybeZeroField.zero_values:
            # Сдвигаем на root.level + 1, что бы непосредственные дети сервиса имели уровень 0
            # Т.к. этот код используется для страницы вложенных сервисов
            # Где дети сервиса отображаются как лежащие в корне
            service_level_shift = get_object_or_404(Service, pk=root_pk).level + 1
            for service in services_list:
                service['level'] -= service_level_shift
        service_id_to_traffics = {id: [] for id in ids}
        traffic_lights = (
            ServiceTrafficStatus.objects.of_services_ids(ids)
            .values(*FrontendTrafficSerializerFields.FIELD_LIST_TO_SELECT)
        )
        for traffic in traffic_lights:
            service_id_to_traffics[traffic['service_id']].append(traffic)
        tags = (
            ServiceTag.objects.filter(service__id__in=ids)
            .values(*FrontendTagsSerializerField.FIELDS_LIST_TO_SELECT)
        )
        service_tag_map = defaultdict(list)
        for tag in tags:
            service_tag_map[tag['service__id']].append(tag)
        services_list.sort(key=self.get_fields_to_ordering)
        serializer = self.get_serializer(
            services_list,
            many=True,
            traffic_lights=service_id_to_traffics,
            tags=service_tag_map,
        )

        return Response({'results': serializer.data, 'filter_mode': filter_mode})

    def _view_only_own_services(self, request):
        """Удаляет из запроса все query-параметры и добавляет фильтрацию member=request.user"""
        query_params = QueryDict(mutable=True, encoding=request.GET.encoding)
        query_params['member'] = self.request.user.staff.login
        query_params['fields'] = self.request._request.GET['fields']
        query_params._mutable = self.request._request.GET._mutable
        self.request._request.GET = query_params


class FrontendContactsFilter(ContactsFilter):
    class Meta:
        model = ContactsFilter.Meta.model
        fields = ContactsFilter.Meta.fields


class FrontendContactsView(V4ContactsView, ContactsReplaceView):
    default_swagger_schema = SwaggerFrontend

    ordering_fields = ('id', 'position', )
    ordering = ('position', )
    filter_class = FrontendContactsFilter
    http_method_names = ['get', 'post', 'put', 'patch', 'delete', 'options']

    @action(methods=['put'], url_path='replace', url_name='replace', detail=False)
    def replace_contacts(self, request):
        return super().post(request, None)


class FrontendResponsibleFilter(ResponsibleFilter):
    class Meta:
        model = ResponsibleFilter.Meta.model
        fields = ResponsibleFilter.Meta.fields


class FrontendResponsibleView(V4ResponsiblesView):
    default_swagger_schema = SwaggerFrontend

    filter_class = FrontendResponsibleFilter


class FrontendMemberFilter(MemberFilter):
    class Meta:
        model = MemberFilter.Meta.model
        fields = MemberFilter.Meta.fields


class FrontendMembersView(V4MembersView):
    is_frontend = True
    default_swagger_schema = SwaggerFrontend

    filterset_class = FrontendMemberFilter


class FrontendServiceExportView(TvmAccessMixin, viewsets.ModelViewSet):
    """
    Экспорт должен быть доступен только для полной роли,
    тк сейчас пермишен can_export есть только у неё.
    """
    default_swagger_schema = SwaggerFrontend

    filter_class = FrontendServiceFilter
    search_fields = ('slug', 'name', 'name_en')
    http_method_names = ['get']
    queryset = Service.objects.all()
    permission_classes = [ServicePermissions]
    _permissions_to_proceed = 'can_export'

    def _check_filter_params(self, get_params):
        """
        Проверим можно ли делать такой запрос, не слишком
        ли он тяжелый
        """
        if 'parents' in get_params:
            raise BadRequest(message={
                'ru': 'Экспорт не поддерживает параметр parents',
                'en': 'Export does not support the parent filter',
            })

        if 'skip_check_filter' in get_params:
            return

        root_param = int(get_params.get('root', 0))

        tags__id__in = get_params.get('tags__id__in')

        filter_fields = (
            'search', 'owner', 'member', 'department',
            'has_external_members', 'is_suspicious',
        )

        if (
            not tags__id__in and
            not root_param and
            not (any(field in get_params for field in filter_fields))
        ):
            raise BadRequest(message={
                'ru': 'Слишком широкое условие поиска',
                'en': 'The search term is too broad',
            })

    def list(self, request):
        self._check_filter_params(request.GET)

        queryset = self.filter_queryset(self.get_queryset()).select_related('owner', 'service_type', ).prefetch_related('tags')

        root_param = request.GET.get('root', 0)
        root = Service.objects.filter(id=root_param).first()

        buffer = BytesIO()
        build_xlsx_workbook(queryset, root, buffer)
        return make_attachment_response(
            data=buffer.getvalue(),
            filename='services.xlsx',
        )


class FrontendContactsTypeView(TvmAccessMixin, mixins.ListModelMixin, OrderingMixin, viewsets.GenericViewSet):
    default_swagger_schema = SwaggerFrontend
    queryset = ContactType.objects.all()
    serializer_class = ContactTypeSerializer
    pagination_class = None
