from django.conf import settings
from rest_framework import serializers, viewsets

from plan.api import base
from plan.api.exceptions import ServiceTagsDuplicatingError
from plan.api.permissions import TvmAuthenticated
from plan.api.mixins import DefaultFieldsMixin, SelectOnlyFieldsMixin, TvmAccessMixin, OrderingMixin
from plan.services.models import Service
from plan.swagger import SwaggerServices


class GradientSerializer(base.ModelSerializer):
    type = serializers.SerializerMethodField()

    class Meta:
        model = Service
        fields = ('id', 'type', 'valuestream', 'umbrella')

    def get_type(self, service):
        try:
            gradient_type = service.gradient_type()
        except ServiceTagsDuplicatingError:
            return None

        return gradient_type

    def to_representation(self, instance):
        result = super(GradientSerializer, self).to_representation(instance)

        # для valuestream храним valuestream, но в ручке отдаём None
        # если структура поломана и у valuestream оказался зонтик, то тоже отдаём его как None
        if result['type'] == settings.GRADIENT_VS:
            result['valuestream'] = None
            result['umbrella'] = None

        # для зонтика храним зонтик, но в ручке отдаём None
        if result['type'] == settings.GRADIENT_UMB:
            result['umbrella'] = None

        return result


class V4GradientView(DefaultFieldsMixin,
                     viewsets.ReadOnlyModelViewSet,
                     TvmAccessMixin,
                     SelectOnlyFieldsMixin,
                     OrderingMixin):
    """
    Справочник valuestream'ов
    """
    default_swagger_schema = SwaggerServices

    pagination_class = base.ABCCursorPagination
    permission_classes = [TvmAuthenticated]
    queryset = Service.objects.gradient().prefetch_related('tags')
    serializer_class = GradientSerializer
    _permissions_to_proceed = 'can_view'
    ordering_fields = ('valuestream', 'id',)
    default_fields = [
        'id',
        'type',
        'valuestream',
        'umbrella',
    ]
