import logging

from django.core.validators import MinValueValidator
from django.db.models import Subquery, OuterRef, Min
from django.utils.translation import ugettext_lazy as _
from django.conf import settings

from rest_framework import serializers, viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response

import django_filters

from plan.services.helpers import get_expires_at
from plan.api import base, validators
from plan.api.base import ABCPagination, ABCCursorPagination
from plan.api.filters import (
    CharInFilter,
    IntegerListFilter,
    CustomModelMultipleChoiceFilter,
)
from plan.api.forms import ServiceFilterForm
from plan.api.exceptions import Conflict, ValidationError
from plan.api.idm import actions
from plan.api.mixins import DefaultFieldsMixin, SelectOnlyFieldsMixin, TvmAccessMixin
from plan.hr import helpers as hr_helpers
from plan.idm import exceptions as idm_exceptions
from plan.roles.models import Role, RoleScope
from plan.services.api.permissions import ServiceMemberPermissions
from plan.services.models import Service, ServiceMember
from plan.services.api.departments import DepartmentMemberSerializer
from plan.services.exceptions import ServiceReadonlyError
from plan.staff.models import Staff
from plan.api.serializers import BasePlanModelSerializer
from plan.services.state import SERVICEMEMBER_STATE
from plan.swagger import SwaggerServices

log = logging.getLogger(__name__)

MEMBERS_RELATED_FIELDS = [
    'staff',
    'service',
    'role',
    'role__scope',
    'role__service',
    'from_department',
    'from_department__department',
    'from_department__service',
    'from_department__role',
    'from_department__role__scope',
]


class MemberSerializer(base.ModelSerializer):
    staff = base.CompactStaffSerializer()
    service = base.CompactServiceSerializer()
    role = base.CompactRoleSerializer()
    department_member = DepartmentMemberSerializer(source='from_department')

    class Meta:
        model = ServiceMember
        fields = (
            'created_at',
            'department_member',
            'id',
            'modified_at',
            'role',
            'service',
            'staff',
        )


class V3MemberSerializer(MemberSerializer):
    staff = None
    person = base.CompactStaffSerializer(source='staff')

    class Meta(MemberSerializer.Meta):
        fields = ('id', 'person', 'service', 'role', 'created_at', 'modified_at', 'department_member')


class CreateMemberSerializer(BasePlanModelSerializer):
    person = serializers.SlugRelatedField(
        queryset=Staff.objects.all(),
        required=True,
        slug_field='login',
        source='staff',
    )
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.alive(),
        required=True,
    )
    comment = serializers.CharField(required=False, default='')
    silent = serializers.BooleanField(required=False, default=False)
    deprive_after = serializers.IntegerField(
        required=False,
        validators=[MinValueValidator(1)]
    )
    deprive_at = serializers.DateField(required=False)

    class Meta:
        model = ServiceMember
        fields = (
            'service', 'person', 'role',
            'deprive_after', 'deprive_at',
            'comment', 'silent',
        )
        validators = [
            validators.MutuallyExclusiveFieldsValidator(('deprive_at', 'deprive_after'))
        ]


class MemberFilter(base.ServiceDescendantsFilterSet):
    service__in = IntegerListFilter(
        field_name='service',
        lookup_expr='in',
    )

    staff = django_filters.CharFilter(field_name='staff__login')

    person = django_filters.CharFilter(field_name='staff__login')
    person__in = CharInFilter(field_name='staff__login', lookup_expr='in')

    person__login = django_filters.CharFilter(field_name='staff__login')
    person__login__in = CharInFilter(field_name='staff__login', lookup_expr='in')

    person__uid = django_filters.CharFilter(field_name='staff__uid')
    person__uid__in = CharInFilter(field_name='staff__uid', lookup_expr='in')

    role = CustomModelMultipleChoiceFilter(
        field_name='role',
        queryset=Role.objects.all(),
        distinct=False,
    )

    # у role__scope весёлое поведение:
    # exact не умеет работать с id, а in не умеет в slug
    # оставляем поведение как есть, тк кто-то может пользоваться
    # вместо этого добавляем два новых фильтра
    role__scope = CustomModelMultipleChoiceFilter(
        field_name='role__scope__slug',
        queryset=RoleScope.objects.all(),
        to_field_name='slug',
        distinct=False,
    )

    role__code = CustomModelMultipleChoiceFilter(
        field_name='role__code',
        queryset=Role.objects.all(),
        to_field_name='code',
        distinct=False,
    )
    is_exportable = base.BooleanFilter(
        field_name='role__is_exportable',
        lookup_expr='exact',
    )

    service__state = CustomModelMultipleChoiceFilter(
        field_name='service__state',
        queryset=Service.objects.all(),
        to_field_name='state',
        distinct=False,
    )
    service__with_descendants = base.ServiceWithDescendantsFilter(
        field_name='service',
        queryset=Service.objects.all(),
        distinct=False,
        only_active=True,
        only_id=True,
    )
    service__with_descendants__slug = base.ServiceWithDescendantsFilter(
        field_name='service__slug',
        queryset=Service.objects.all(),
        to_field_name='slug',
        distinct=False,
        only_active=True,
    )

    service__with__tags__exclude = django_filters.CharFilter(
        field_name='service__tags__slug',
        exclude=True,
    )

    service__with__tags__exclude__in = CharInFilter(
        field_name='service__tags__slug',
        exclude=True,
        lookup_expr='in',
    )

    is_robot = base.BooleanFilter(
        field_name='staff__is_robot',
        lookup_expr='exact',
    )

    unique = base.BooleanFilter(
        field_name='staff',
        method='filter_unique',
    )

    def filter_unique(self, queryset, name, value):
        if value in ('1', 'True', 'true'):
            queryset = queryset.filter(pk__in=Subquery(
                ServiceMember.objects
                    .filter(service=OuterRef('service'), staff=OuterRef('staff'))
                    .values('staff_id')
                    .annotate(min_id=Min('id'))
                    .values_list('min_id', flat=True)
            ))

        return queryset

    class Meta:
        model = ServiceMember
        form = ServiceFilterForm
        fields = {
            'id': ['gt', 'lt', 'exact', 'in'],
            'created_at': ['gt', 'lt'],
            'modified_at': ['gt', 'lt'],
            'role': ['exact', 'in'],
            'role__code': ['exact', 'in'],
            'role__scope__id': ['exact', 'in'],
            'role__scope__slug': ['exact', 'in'],
            'role__scope': ['exact', 'in'],

            'service': ['exact', 'in'],
            'service__is_exportable': ['exact', 'in'],
            'service__slug': ['exact', 'in'],
            'service__state': ['exact', 'in'],
            'is_exportable': ['exact'],
        }


class MembersView(TvmAccessMixin, SelectOnlyFieldsMixin, base.OrderingMixin, viewsets.ModelViewSet):
    """
    Персональные роли в сервисах
    """
    is_frontend = False
    serializer_class = MemberSerializer
    filterset_class = MemberFilter
    http_method_names = ['get', 'post', 'delete']
    permission_classes = [ServiceMemberPermissions]
    pagination_class = ABCPagination
    _queryset = ServiceMember.objects.team()
    _permissions_to_proceed = 'view_own_services'

    @property
    def queryset(self):
        queryset = self._queryset
        if not self.is_frontend:
            queryset = queryset.exclude(staff__is_frozen=True)

        if self.request.user and not self.request.user.has_perm('internal_roles.view_all_services'):
            queryset = self.get_restricted_queryset()

        if self.action == 'list':
            return queryset.filter(service__state__in=Service.states.ALIVE_STATES)
        return queryset

    def get_restricted_queryset(self):
        requester = None
        if self.request.user and hasattr(self.request.user, 'staff'):
            requester = self.request.user.staff
        return self._queryset.filter(service__in=Service.objects.own_only(requester))

    def get_serializer_class(self):
        if self.action == 'create':
            return CreateMemberSerializer

        return self.serializer_class

    @staticmethod
    def check_active_role(service: Service, staff: Staff, role: Role) -> bool:
        return ServiceMember.objects.filter(
            service=service,
            staff=staff,
            role=role,
            from_department=None,
            state=SERVICEMEMBER_STATE.ACTIVE
        ).exists()

    def create(self, request, *args, **kwargs):
        """
        Заявка на новую персональную роль
        """
        requester = request.user.staff
        serializer = self.get_serializer(data=request.data)

        if not serializer.is_valid():
            raise ValidationError(extra=serializer.errors)

        data = serializer.validated_data

        if data['service'].readonly_state is not None:
            raise ServiceReadonlyError(data['service'])

        if self.check_active_role(service=data['service'], staff=data['staff'], role=data['role']):
            raise Conflict(
                detail='Role already exists',
                message={
                    'en': f'User \"{data["staff"].get_full_name()}\" already has this role',
                    'ru': f'У пользователя \"{data["staff"].get_full_name()}\" уже есть такая роль',
                },
            )

        if (role_data := self._request_membership(requester, data)) is not None:
            expires_at = get_expires_at(data)
            self._create_membership(data, role_data['id'], expires_at=expires_at)

        headers = self.get_success_headers(serializer.data)
        return Response(status=status.HTTP_201_CREATED, headers=headers)

    def _create_membership(self, data, idm_role_id, expires_at):
        membership, created = ServiceMember.all_states.get_or_create(
            service=data['service'], staff=data['staff'], role=data['role'],
            resource=None, from_department=None)
        membership.request(idm_role_id, autorequested=False, expires_at=expires_at)

    def _request_membership(self, requester, data):
        try:
            role_data = actions.request_membership(
                data['service'],
                data['staff'],
                data['role'],
                requester=requester,
                deprive_after=data.get('deprive_after'),
                deprive_at=data.get('deprive_at'),
                comment=data['comment'],
                silent=data['silent'],
                timeout=settings.IDM_POST_ROLES_TIMEOUT,
                retry=settings.ABC_IDM_FROM_API_RETRY,
            )

        except idm_exceptions.Conflict as e:
            if self.check_active_role(service=data['service'], staff=data['staff'], role=data['role']):
                return None
            raise Conflict(
                message=e.message,
                detail=e.detail,
                extra=e.extra,
            )

        return role_data

    def destroy(self, request, *args, **kwargs):
        """
        Отзыв персональной роли
        """

        obj = self.get_object()
        requester = self.request.user.staff

        # система aware, состав групповых ролей отслеживает сама
        if obj.from_department:
            raise ValidationError(detail=_('Cannot deprive sub-role'))

        if obj.state == 'depriving':
            raise ValidationError(detail=_('Роль уже отзывается'))

        # руководителя отзывать нельзя
        if obj.role.code == Role.EXCLUSIVE_OWNER:
            raise Conflict(detail=_('Роль руководителя отзывать нельзя. Назначьте нового.'))

        log.info('%s asked to delete membership %s', requester.login, obj)

        hr_helpers.create_history_object(obj, requester, obj.part_rate, None)

        self._deprive_role(requester, obj)
        self._deprive_membership(obj)

        return Response(status=status.HTTP_204_NO_CONTENT)

    def _deprive_role(self, requester, obj):
        actions.deprive_role(
            member=obj,
            requester=requester,
            timeout=settings.IDM_DELETE_ROLES_TIMEOUT,
            retry=settings.ABC_IDM_FROM_API_RETRY,
        )

    def _deprive_membership(self, obj):
        obj.set_depriving_state()

    @action(methods=['post'], detail=True)
    def rerequest(self, request, pk=None):
        """
        Перезапрос устаревающей персональной роли
        """
        obj = self.get_object()
        requester = self.request.user.staff

        actions.rerequest_role(
            service_member=obj,
            requester=requester,
            timeout=settings.IDM_POST_ROLES_TIMEOUT,
            retry=settings.ABC_IDM_FROM_API_RETRY,
        )
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(methods=['post'], detail=False)
    def approve(self, request):
        """
        Подтверждение заказанной персональной роли
        """
        idm_role_id = request.data['idm_role']

        actions.approve_role(
            role_id=idm_role_id,
            requester=request.user.staff,
            timeout=settings.IDM_POST_ROLES_TIMEOUT,
            retry=settings.ABC_IDM_FROM_API_RETRY,
        )
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(methods=['post'], detail=False)
    def decline(self, request):
        """
        Отклонение заказанной персональной роли
        """
        idm_role_id = request.data['idm_role']

        actions.decline_role(
            role_id=idm_role_id,
            requester=request.user.staff,
            timeout=settings.IDM_POST_ROLES_TIMEOUT,
            retry=settings.ABC_IDM_FROM_API_RETRY,
        )
        return Response(status=status.HTTP_204_NO_CONTENT)


class V3MembersView(MembersView):
    """
    Персональные роли в сервисах для API v3
    """
    serializer_class = V3MemberSerializer


class V4MembersView(DefaultFieldsMixin, V3MembersView):
    """
    Персональные роли в сервисах
    """
    default_swagger_schema = SwaggerServices

    pagination_class = ABCCursorPagination
    default_fields = [
        'id',

        'department_member.department.url',

        'person.login',
        'person.id',
        'person.uid',

        'role.code',
        'role.id',
        'role.scope.slug',

        'service.id',
        'service.slug',
    ]
