import logging

import django_filters

from django.conf import settings
from django_fsm import TransitionNotAllowed

from rest_framework import serializers, viewsets
from rest_framework.decorators import action
from rest_framework.metadata import BaseMetadata
from rest_framework.response import Response

from plan.api.mixins import TvmAccessMixin
from plan.api.base import ModelSerializer
from plan.api.filters import CustomModelMultipleChoiceFilter, PlanFilterSet
from plan.api.exceptions import BadRequest, ValidationError
from plan.api.serializers import CompactServiceSerializer, CompactStaffSerializer
from plan.api.mixins import OrderingMixin
from plan.services import permissions
from plan.services.api.permissions import ServicePermissions
from plan.services.api.validators import (
    FryParadoxValidator, SameParentValidator,
    service_is_active, RestrictMoveToJunk,
    CheckOEBSRestrictionsForMovingService,
    CheckBillingPointRestrictionsForMovingService,
    CheckAllowedParentType,
)
from plan.services.models import Service, ServiceMoveRequest, ServiceMember
from plan.services.tasks import notify_move_request
from plan.staff.models import Staff
from plan.swagger import SwaggerServices, SwaggerFrontend


log = logging.getLogger(__name__)


class ApproversSerializer(serializers.Serializer):
    recommended = CompactStaffSerializer(many=True)
    other = CompactStaffSerializer(many=True)


class CreateMoveRequestSerializer(serializers.Serializer):
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.all(),
        required=True,
        validators=[service_is_active]
    )
    destination = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.all(),
        required=True,
        validators=[service_is_active]
    )

    class Meta:
        validators = [
            FryParadoxValidator(),
            SameParentValidator(),
            RestrictMoveToJunk(),
            CheckOEBSRestrictionsForMovingService(),
            CheckBillingPointRestrictionsForMovingService(),
            CheckAllowedParentType(),
        ]


class MoveRequestSerializer(ModelSerializer):
    service = CompactServiceSerializer()
    destination = CompactServiceSerializer()
    requester = CompactStaffSerializer()
    approver_incoming = CompactStaffSerializer()
    approver_outgoing = CompactStaffSerializer()

    actions = serializers.SerializerMethodField()
    state_display = serializers.SerializerMethodField()

    def get_state_display(self, instance):
        return {
            'en': instance.state.title(),
            'ru': instance.get_state_display()
        }

    def get_actions(self, instance):
        request = self.context.get('request')
        if request is None:
            return []

        person = request.person
        actions = set()

        if permissions.can_approve_move_request(instance.service, person):
            actions.add('approve')

        if permissions.can_cancel_move_request(instance.service, person):
            actions.add('reject')

        return sorted(list(actions))

    class Meta:
        model = ServiceMoveRequest
        fields = (
            'id',
            'created_at',
            'service',
            'destination',
            'state',
            'state_display',
            'requester',
            'approver_incoming',
            'approver_outgoing',
            'actions'
        )


class MoveRequestFilter(PlanFilterSet):
    service = CustomModelMultipleChoiceFilter(
        queryset=Service.objects.all(),
        distinct=True,
    )
    destination = CustomModelMultipleChoiceFilter(
        queryset=Service.objects.all(),
        distinct=True,
    )
    kind = django_filters.CharFilter(method='filter_move_request_kind')
    requester = CustomModelMultipleChoiceFilter(
        field_name='requester__login',
        queryset=Staff.objects.active(),
        to_field_name='login',
        distinct=True,
    )

    def filter_move_request_kind(self, queryset, name, value):
        if value == 'active':
            queryset = queryset.active()
        elif value == 'inactive':
            queryset = queryset.dead()
        return queryset

    class Meta:
        model = ServiceMoveRequest
        fields = {
            'id': ['exact', 'in'],
            'state': ['exact', 'in'],
            'service': ['exact', 'in'],
            'destination': ['exact', 'in'],
        }


class PermissionsMetadata(BaseMetadata):
    def determine_metadata(self, request, view):
        view_permissions = []

        try:
            move_request = view.get_object()
            if permissions.can_approve_move_request(move_request.service, request.person):
                view_permissions.append('can_approve')
            if permissions.can_cancel_move_request(move_request.service, request.person):
                view_permissions.append('can_cancel')

        except AssertionError:  # выскочит если вызывают там, где нет инстанса
            pass

        return {'permissions': view_permissions}


class MovesView(OrderingMixin, TvmAccessMixin, viewsets.ModelViewSet):
    serializer_class = MoveRequestSerializer
    filter_class = MoveRequestFilter
    search_fields = ()
    http_method_names = ['get', 'post', 'options']
    queryset = ServiceMoveRequest.objects.select_related(
        'service',
        'destination',
        'approver_incoming',
        'approver_outgoing'
    )
    permission_classes = [ServicePermissions]
    metadata_class = PermissionsMetadata
    ordering_fields = ()

    def create(self, request):
        """
        Создание запроса на перемещение сервиса.
        """
        serializer = CreateMoveRequestSerializer(data=request.data, context={'request': request})
        if serializer.is_valid():
            service = serializer.validated_data['service']
            destination = serializer.validated_data['destination']
            move_request = ServiceMoveRequest.request(
                service=service,
                destination=destination,
                requester=request.person
            )
            notify_move_request.apply_async(
                args=[move_request.id, request.person.login],
                countdown=settings.ABC_DEFAULT_COUNTDOWN,
            )
            return Response(MoveRequestSerializer(move_request).data)
        else:
            raise ValidationError(extra=serializer.errors)

    @action(methods=['post'], detail=True)
    def reject(self, request, pk=None):
        """
        Отклонение заявки на перемещения сервиса.
        """
        move_request = self.get_object()
        try:
            move_request.reject(rejector=self.request.person)
        except TransitionNotAllowed:
            raise BadRequest(message={
                'ru': 'Решение по этому запросу уже было принято',
                'en': 'This request was already processed',
            })

        return Response(status=204)

    @action(methods=['post'], detail=True)
    def approve(self, request, pk=None):
        """
        Подтверждение заявки на перемещения сервиса.
        """
        move_request = self.get_object()
        try:
            move_request.approve(approver=self.request.person)
        except TransitionNotAllowed:
            raise BadRequest(message={
                'ru': 'Решение по этому запросу уже было принято',
                'en': 'This request was already processed',
            })

        return Response(status=204)

    @staticmethod
    def _get_approvers(service):
        recommended = []
        for approver in service.members.responsibles():
            if approver.staff not in recommended:
                recommended.append(approver.staff)

        other = []
        potential_approvers = (
            ServiceMember.objects
            .filter(service__in=service.get_ancestors())
            .responsibles()
            .order_by('service__level')
        )
        for approver in potential_approvers:
            if approver.staff not in other and approver.staff not in recommended:
                other.append(approver.staff)

        return {'recommended': recommended, 'other': other}

    @action(methods=['get'], detail=True)
    def approvers(self, request, pk=None):
        """
        Получение списка подтверждающих для заявки.
        """
        move_request = self.get_object()

        service_approvers_serializer = ApproversSerializer(self._get_approvers(move_request.service))
        destination_approvers_serializer = ApproversSerializer(self._get_approvers(move_request.destination))

        return Response(
            {
                'service': service_approvers_serializer.data,
                'destination': destination_approvers_serializer.data
            }
        )


class V4MovesView(MovesView):
    """
    Запросы на перемещение сервиса
    """
    default_swagger_schema = SwaggerServices


class FrontendMovesView(MovesView):
    default_swagger_schema = SwaggerFrontend
