from collections import defaultdict

from django.conf import settings
from django.db.models import Count, F
from rest_framework import viewsets
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from plan.api.serializers import CompactServiceSerializer
from plan.services.models import Service
from plan.staff.constants import DEPARTMENT_ROLES
from plan.staff.models import Staff, Department, DepartmentStaff
from plan.swagger import SwaggerFrontend


class ParentView(viewsets.GenericViewSet):
    default_swagger_schema = SwaggerFrontend

    _permissions_to_proceed = 'can_edit'
    permission_classes = [IsAuthenticated]
    serializer_class = CompactServiceSerializer
    queryset = Service.objects.active().filter(is_exportable=True)

    def get_response_services_pks(self, owner):
        department_pks_to_level = {
            pk: level
            for pk, level in
            Department._closure_model.objects.filter(child=owner.department).values_list('parent', 'depth')
        }

        staff_pks_to_level = {
            ds.staff_id: department_pks_to_level.get(ds.department_id)
            for ds in DepartmentStaff.objects.filter(
                department_id__in=department_pks_to_level.keys(), role=DEPARTMENT_ROLES.CHIEF
            )
        }
        if owner.id not in staff_pks_to_level:
            staff_pks_to_level[owner.id] = -1

        base_non_leaf_services_pks = (
            Service.objects
            .filter(is_base=True, children__is_base=True)
            .values_list('pk', flat=True)
        )

        owned_services = (
            Service._closure_model.objects
            .exclude(parent_id__in=base_non_leaf_services_pks)
            .filter(
                parent__owner_id__in=staff_pks_to_level.keys(),
                child__owner=F('parent__owner'),
                depth__lte=1
            )
            .values('parent', 'parent__owner_id')
            .annotate(Count('parent'))
            .order_by('-parent__count')
        )

        levels_to_services_pks = defaultdict(list)

        for owned_service in owned_services:
            level = staff_pks_to_level[owned_service['parent__owner_id']]
            levels_to_services_pks[level].append(owned_service['parent'])

        if len(levels_to_services_pks.keys()) == 0:
            response_services_pks = []
        else:
            level = min(levels_to_services_pks.keys())
            limit = settings.PARENT_HINT_LIMIT
            response_services_pks = levels_to_services_pks[level][0:limit]
        return response_services_pks

    def list(self, request, *args, **kwargs):
        owner_login = request.query_params.get('owner', '')
        owner = get_object_or_404(Staff.objects.select_related('department'), login=owner_login)
        response_services_pks = self.get_response_services_pks(owner)

        response_services = list(self.queryset.filter(pk__in=response_services_pks))
        response_services.sort(key=lambda s: response_services_pks.index(s.id))

        serializer = self.get_serializer(response_services, many=True)
        return Response({'results': serializer.data})
