from plan.api.exceptions import PermissionDenied
from plan.api.permissions import TvmAuthenticated
from plan.duty.permissions import is_watcher_tvm_user
from plan.services import models, permissions


class ServicePermissions(TvmAuthenticated):
    def has_object_permission(self, request, view, obj):
        if isinstance(obj, models.Service):
            service = obj
        elif hasattr(obj, 'service'):
            service = obj.service
        else:
            return False

        if view.action in ('update', 'destroy'):
            return (permissions.is_team_member(service, request.person) or
                    permissions.is_service_responsible(service, request.person))
        else:
            return True


class ServiceMemberPermissions(TvmAuthenticated):
    def has_object_permission(self, request, view, obj):
        if view.action == 'destroy':
            return permissions.can_remove_member(obj, request.person)

        elif view.action in ('approve', 'decline'):
            return permissions.is_service_responsible(obj, request.person)

        else:
            return True


class ServiceDepartmentMemberPermissions(TvmAuthenticated):
    def has_object_permission(self, request, view, obj):
        if view.action in ('approve', 'decline', 'destroy'):
            return permissions.is_service_responsible(obj.service, request.person)

        else:
            return True


def can_edit_flags(request, old_flags, new_flags):
    ticket = getattr(request, 'tvm_service_ticket', None)
    if ticket is not None and is_watcher_tvm_user(ticket):
        return
    if old_flags != new_flags:
        raise PermissionDenied(
            message={
                'ru': "Редактирование флагов запрещено",
                'en': "Editing flags is prohibited",
            }
        )
