import logging

import django_filters

from django.conf import settings
from rest_framework import serializers, viewsets

from plan.api import base, forms
from plan.api.exceptions import PermissionDenied, ValidationError
from plan.api.filters import CharInFilter
from plan.api.mixins import DefaultFieldsMixin, TvmAccessMixin
from plan.api.idm import actions
from plan.api.permissions import TvmAuthenticated
from plan.api.serializers import BasePlanModelSerializer
from plan.roles.models import Role
from plan.services import permissions
from plan.services.models import Service, ServiceMember
from plan.staff.models import Staff
from plan.swagger import SwaggerServices

log = logging.getLogger(__name__)


class BaseResponsibleSerializer(base.ModelSerializer):
    service = base.CompactServiceSerializer()
    is_editable = serializers.SerializerMethodField()

    def get_is_editable(self, obj):
        return obj.role.code == Role.RESPONSIBLE


class V3ResponsibleSerializer(BaseResponsibleSerializer):
    person = base.CompactStaffSerializer(source='staff')

    class Meta:
        model = ServiceMember
        fields = ('id', 'service', 'person', 'is_editable')


class ResponsibleSerializer(BaseResponsibleSerializer):
    staff = base.CompactStaffSerializer()

    class Meta:
        model = ServiceMember
        fields = ('id', 'service', 'staff', 'is_editable')


class CreateResponsibleSerializer(BasePlanModelSerializer):
    staff = serializers.SlugRelatedField(
        queryset=Staff.objects.all(),
        slug_field='login',
    )
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.all(),
    )

    class Meta:
        model = ServiceMember
        fields = ('service', 'staff')

    def validate(self, attrs):
        person = self.context['request'].person
        if not settings.CROWDTEST:
            can_manage_responsibles = any((
                permissions.is_service_responsible(attrs['service'], person),
                permissions.is_service_support(person)
            ))
            if not can_manage_responsibles:
                raise PermissionDenied(
                    message={
                        'ru': 'Вам нельзя добавлять управляющих этому сервису',
                        'en': 'You are not allowed to add responsible to this service'
                    }
                )

        present = ServiceMember.objects.non_head_responsibles().filter(
            service=attrs['service'],
            staff=attrs['staff'],
        )
        if present.exists():
            raise ValidationError(
                message={
                    'ru': 'Этот сотрудник уже является управляющим сервиса',
                    'en': 'This employee already responsible in this service'
                }
            )

        return attrs


class ResponsibleFilter(base.ServiceDescendantsFilterSet):
    staff = django_filters.CharFilter(field_name='staff__login')

    person = django_filters.CharFilter(field_name='staff__login')
    person__in = CharInFilter(field_name='staff__login', lookup_expr='in')

    person__login = django_filters.CharFilter(field_name='staff__login')
    person__login__in = CharInFilter(field_name='staff__login', lookup_expr='in')

    person__uid = django_filters.CharFilter(field_name='staff__uid')
    person__uid__in = CharInFilter(field_name='staff__uid', lookup_expr='in')

    class Meta:
        model = ServiceMember
        form = forms.ServiceFilterForm
        fields = {
            'service': ['exact', 'in'],
            'service__is_exportable': ['exact', 'in'],
            'service__slug': ['exact', 'in'],
        }


class ResponsiblesBaseView(TvmAccessMixin, base.OrderingMixin, viewsets.ModelViewSet):
    """
    Ответственные за сервисы
    """
    permission_classes = [TvmAuthenticated]
    serializer_class = ResponsibleSerializer
    filter_class = ResponsibleFilter
    http_method_names = ['get', 'post', 'delete']
    queryset = ServiceMember.objects.select_related('service', 'staff').responsibles()
    _permissions_to_proceed = 'view_own_services'

    def get_queryset(self):
        user_can_view_all_services = self.request.user.has_perm('internal_roles.view_all_services')

        if not user_can_view_all_services:
            return self.queryset.filter(service__team=self.request.user.staff)
        return self.queryset


class ResponsiblesView(ResponsiblesBaseView):
    """
    Ответственные за сервисы
    """

    def get_serializer_class(self):
        if self.action == 'create':
            return CreateResponsibleSerializer

        return ResponsibleSerializer

    def perform_create(self, serializer):
        data = serializer.validated_data
        log.info('Add responsible %s to service %s from interface', data['staff'].login, data['service'].slug)

        requester = self.request.person.staff

        if permissions.is_service_support(requester):
            actions.request_membership(
                data['service'],
                data['staff'],
                Role.get_responsible(),
                comment=(
                    'Роль запрошена сотрудником службы поддержки {} и '
                    'автоподтверждена Конрадом.'
                    .format(requester.login)
                )
            )
        else:
            actions.request_membership(
                data['service'],
                data['staff'],
                Role.get_responsible(),
                requester=requester
            )

        if settings.CROWDTEST:
            ServiceMember.objects.get_or_create(
                state=ServiceMember.states.ACTIVE,
                service=data['service'],
                role=Role.get_responsible(),
                staff=data['staff'],
            )

    def perform_destroy(self, obj):
        if obj.role.code != Role.RESPONSIBLE:
            # TODO: так как мы убрали отдельную модель управляющих, кажется нужно пересмотреть здесь логику
            # руклей и замов меняют через основную ручку /services/members/
            raise PermissionDenied(
                message={
                    'ru': 'Нельзя убрать руководителя или заместителя руководителя из управляющих сервиса',
                    'en': 'You cannot remove a head or deputy from services responsibles'
                }
            )

        person = self.request.person
        can_manage_responsibles = any((
            permissions.is_service_responsible(obj.service, person),
            permissions.is_service_support(person)
        ))
        if not can_manage_responsibles:
            raise PermissionDenied(
                message={
                    'ru': 'Вам нельзя убирать управляющих из этого сервиса',
                    'en': 'You are not allowed to delete responsibles from this service'
                }
            )

        log.info('Remove responsible %s to service %s from interface (by <%s>)',
                 obj.staff.login, obj.service.slug, person.login)

        if permissions.is_service_support(person):
            actions.deprive_role(obj, comment=(
                'Роль отозвана сотрудником службы поддержки {} и автоподтверждена Конрадом.'.format(person.login)
            ))
        else:
            actions.deprive_role(obj, requester=person.staff)

        if settings.CROWDTEST:
            obj.deprive()


class V3ResponsiblesView(ResponsiblesBaseView):
    """
    Ответственные за сервисы
    """
    serializer_class = V3ResponsibleSerializer
    http_method_names = ['get']


class V4ResponsiblesView(DefaultFieldsMixin, V3ResponsiblesView):
    """
    Управляющие сервисом.
    Список людей, имеющих права на управление командой, ресурсами и прочими орг вопросами.
    """
    default_swagger_schema = SwaggerServices

    default_fields = [
        'id',

        'person.login',
        'person.uid',

        'service.id',
        'service.slug',
    ]
