import logging

import django.core.exceptions
import urllib.parse
import waffle

from django.conf import settings
from django.core.urlresolvers import reverse
from django.http import QueryDict, HttpResponseRedirect
from django.forms import Form as DjangoForm
from django.utils import timezone
from django.db.models import Count
from drf_yasg.utils import swagger_auto_schema

import rest_framework.exceptions

from rest_framework import serializers, viewsets
from rest_framework.decorators import action
from rest_framework.metadata import BaseMetadata
from rest_framework.response import Response
from rest_framework.validators import UniqueValidator
from rest_framework import mixins

from plan.idm.exceptions import TimeoutError
from plan.idm.constants import ACTIVE_STATES, INACTIVE_STATES
from plan.api.idm import actions
from plan.api.base import ABCPagination
from plan.api.mixins import NoPaginationListMixin
from plan.services.state import SERVICEMEMBER_STATE
from plan.api import base
from plan.api.exceptions import BadRequest, Conflict, PermissionDenied, FailedDependency
from plan.api.fields import MappingField, IntegerInterfacedSerializerField, LocaleServiceNameField
from plan.api.filters import PlanFilterSet, CustomModelChoiceFilter
from plan.puncher.rules import PuncherClient

from plan.api.mixins import DefaultFieldsMixin, FieldsByPermissionsCutterMixin, SelectOnlyFieldsMixin, TvmAccessMixin
from plan.api.serializers import CompactServiceSerializer
from plan.api.validators import MappingFieldUniqueValidator, ServiceNameValidator
from plan.common.internal_roles import get_internal_roles
from plan.common.utils import timezone as utils
from plan.duty.models import Shift
from plan.duty.api.filters import ShiftBaseFilterSet, CustomModelMultipleChoiceFilter
from plan.duty.api.serializers import ShiftBaseSerializer
from plan.history.mixins import HistoryMixin
from plan.holidays.utils import not_workday
from plan.idm.roles import get_requested_roles_for_service
from plan.oebs.constants import OEBS_FLAGS
from plan.oebs.models import OEBSAgreement
from plan.oebs.utils import is_oebs_related
from plan.roles.models import Role
from plan.services import permissions, tasks
from plan.services.api import validators
from plan.services.api.tags import ServiceTagSerializer as BaseServiceTagSerializer
from plan.services.api.permissions import (
    ServicePermissions,
    TvmAuthenticated,
    can_edit_flags,
)
from plan.services.models import (
    Service,
    ServiceCreateRequest,
    ServiceDeleteRequest,
    ServiceCloseRequest,
    ServiceTag,
    ServiceType,
    ServiceMember,
    ServiceSuspiciousReason,
    ServiceNotification,
)
from plan.services.tasks import process_suspicious_service, calculate_gradient_fields, update_service_review_policy
from plan.signals import reopen_service
from plan.staff.models import Department, Staff
from plan.suspicion.models import ServiceTrafficStatus
from plan.suspicion.api.serializers import (
    ServiceTrafficStatusSerializer,
    MainServiceIssueSerializer,
    IssueGroupWithIssuesSerializer,
    grouped_service_issues, main_service_issue,
)
from plan.swagger import SwaggerServices, SwaggerFrontend
from plan.resources.exceptions import SupplierError

log = logging.getLogger(__name__)


class IdmRoleSerializer(serializers.Serializer):
    person = base.CompactStaffSerializer()
    department = base.CompactDepartmentSerializer()
    role = base.CompactRoleSerializer()
    service = base.CompactServiceSerializer()

    idm_id = serializers.IntegerField()
    state = serializers.CharField()


class ServiceTagSerializer(BaseServiceTagSerializer):

    class Meta(BaseServiceTagSerializer.Meta):
        fields = ('id', 'name', 'color', 'description', 'slug')


class ServiceTypeSerializer(base.ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = ServiceType
        fields = ('id', 'code', 'name')


class ServiceSuspiciousReasonSerializer(base.ModelSerializer):
    class Meta:
        model = ServiceSuspiciousReason
        fields = ('id', 'created_at', 'updated_at', 'reason', 'human_text')


class ListServiceSerializer(base.ModelSerializer):
    ancestors = serializers.SerializerMethodField()

    name = MappingField({
        'ru': 'name',
        'en': 'name_en',
    })
    description = MappingField({
        'ru': 'description',
        'en': 'description_en',
    })
    activity = MappingField({
        'ru': 'activity',
        'en': 'activity_en',
    })
    owner = base.CompactStaffSerializer(queryset=Staff.objects.all())
    parent = base.CompactServiceSerializer(queryset=Service.objects.all())
    tags = IntegerInterfacedSerializerField(
        queryset=ServiceTag.objects.all(),
        serializer=ServiceTagSerializer,
        many=True,
    )
    type = ServiceTypeSerializer(source='service_type', queryset=ServiceType.objects.all())
    related_services = IntegerInterfacedSerializerField(
        queryset=Service.objects.all(),
        serializer=base.CompactServiceSerializer,
        many=True,
    )
    state_display = serializers.SerializerMethodField()
    state_display_i18n = serializers.SerializerMethodField()
    departments = IntegerInterfacedSerializerField(
        queryset=Department.objects.all(),
        serializer=base.CompactDepartmentSerializer,
        many=True,
    )
    kpi = MappingField({
        'bugs_count': 'kpi_bugs_count',
        'releases_count': 'kpi_release_count',
        'lsr_count': 'kpi_lsr_count',
    })
    traffic_light = ServiceTrafficStatusSerializer(
        many=True,
        read_only=True,
        source='traffic_statuses',
        queryset=ServiceTrafficStatus.objects.order_by('issue_group__code'),
    )
    fields_mapping_ = {
        'name': ('name', 'name_en'),
        'description': ('description', 'description_en'),
        'activity': ('activity', 'activity_en'),
        'state_display': ('state',),
        'state_display_i18n': ('state',),
        'kpi': ('kpi_bugs_count', 'kpi_release_count', 'kpi_lsr_count'),
        'is_suspicious': ('suspicious_date',),
    }

    def __init__(self, *args, **kwargs):
        super(ListServiceSerializer, self).__init__(*args, **kwargs)
        request = self.context['request']
        if hasattr(request, 'person'):
            self.can_view_tags = request.person.user.has_perm('internal_roles.view_tags')
        else:
            self.can_view_tags = True

    class Meta:
        model = Service
        fields = (
            'activity',
            'ancestors',
            'children_count',
            'created_at',
            'departments',
            'descendants_count',
            'description',
            'id',
            'is_exportable',
            'is_suspicious',
            'sandbox_move_date',
            'keywords',
            'kpi',
            'modified_at',
            'name',
            'owner',
            'parent',
            'membership_inheritance',
            'path',
            'readonly_state',
            'slug',
            'state',
            'state_display',
            'state_display_i18n',
            'tags',
            'type',
            'related_services',
            'unique_immediate_members_count',
            'unique_immediate_external_members_count',
            'unique_immediate_robots_count',
            'has_forced_suspicious_reason',
            'has_external_members',
            'traffic_light',
        )

    def get_state_display(self, obj):
        return {
            # TODO: сделать переводы значений
            'ru': obj.get_state_display(),
            'en': obj.get_state_display(),
        }

    def get_state_display_i18n(self, obj):
        return obj.get_state_display()

    def get_ancestors(self, obj):
        for ancestor in obj.ancestors:
            if not self.can_view_tags:
                ancestor.pop('tags', None)
        return obj.ancestors


class DetailServiceSerializer(ListServiceSerializer):
    issue = serializers.SerializerMethodField()
    fields_mapping_ = {'issue': ('id',)}

    class Meta:
        model = Service
        fields = ListServiceSerializer.Meta.fields + ('issue',)

    @staticmethod
    def get_issue(service):
        main_issue = main_service_issue(service.id)
        if main_issue:
            return MainServiceIssueSerializer(main_issue).data


class CreateServiceSerializer(base.ModelSerializer):
    slug = serializers.CharField(
        validators=[
            validators.ServiceSlugValidator(),
            UniqueValidator(
                queryset=Service.objects.all(),
                message={
                    'ru': 'Сервис с таким слагом уже существует',
                    'en': 'Service with this slug already exists',
                }
            ),
        ],
    )
    name = LocaleServiceNameField({
        'ru': 'name',
        'en': 'name_en',
    })
    owner = serializers.SlugRelatedField(
        queryset=Staff.objects.all(),
        slug_field='login',
        required=True,
    )
    parent = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.all(),
        required=False,
    )
    type = serializers.SlugRelatedField(
        source='service_type',
        slug_field='code',
        queryset=ServiceType.objects.all(),
        default=lambda: ServiceType.get_default_type()
    )
    id = serializers.IntegerField(read_only=True)

    tags = IntegerInterfacedSerializerField(
        queryset=ServiceTag.objects.all(),
        serializer=ServiceTagSerializer,
        many=True,
        required=False,
    )

    class Meta:
        model = Service
        fields = ('id', 'slug', 'name', 'owner', 'parent', 'type', 'tags', )
        validators = [
            validators.CheckAllowedParentType(),
        ]


class V4CreateServiceSerializer(CreateServiceSerializer):
    description = MappingField(
        {
            'ru': 'description',
            'en': 'description_en',
        },
        required=False,
    )

    class Meta:
        model = Service
        fields = CreateServiceSerializer.Meta.fields + ('description', )
        validators = [
            validators.CheckAllowedParentType(),
        ]


class UpdateServiceSerializer(base.ModelSerializer):
    tags = IntegerInterfacedSerializerField(
        queryset=ServiceTag.objects.all(),
        serializer=ServiceTagSerializer,
        many=True,
    )
    activity = MappingField(
        {
            'ru': 'activity',
            'en': 'activity_en',
        },
        required=False,
    )
    description = MappingField(
        {
            'ru': 'description',
            'en': 'description_en',
        },
        required=False,
    )
    state = serializers.ChoiceField(
        choices=Service.states,
        validators=[
            validators.CanDelete(),
            validators.CannotRestore(),
            validators.AncestorsMustBeActive(),
            validators.CannotCloseWithImportantResources(),
            validators.CannotChangeStateWithOEBSAgreement(),
        ],
    )
    name = MappingField(
        {'ru': 'name', 'en': 'name_en'},
        validators=[
            MappingFieldUniqueValidator(queryset=Service.objects.all()),
            ServiceNameValidator()
        ]
    )
    departments = IntegerInterfacedSerializerField(
        queryset=Department.objects.all(),
        serializer=base.CompactDepartmentSerializer,
        many=True,
    )
    type = serializers.SlugRelatedField(
        source='service_type',
        slug_field='code',
        queryset=ServiceType.objects.all(),
        required=False,
    )

    related_services = IntegerInterfacedSerializerField(
        queryset=Service.objects.all(),
        serializer=base.CompactServiceSerializer,
        many=True
    )

    membership_inheritance = serializers.BooleanField(
        help_text="Признак, указывающий будет ли сервис наследоваться в вышестоящие сервисы. "
                  "Если эту галку снять, то участники сервиса и его дети больше не будут отдаваться "
                  "в выдачи родительских сервисов в staff-api"
    )

    class Meta:
        model = Service
        fields = (
            'url',
            'tags',
            'activity',
            'description',
            'state',
            'parent',
            'name',
            'departments',
            'type',
            'flags',
            'related_services',
            'membership_inheritance',
            'use_for_hardware',
            'use_for_hr',
            'use_for_revenue',
            'use_for_procurement',
            'use_for_group_only',
        )
        validators = [
            validators.ServiceIsReadonly(),
            validators.ChangeOEBSFlags(),
        ]

    def _get_oebs_flags(self, instance, validated_data):
        updated_flags = {}
        for flag in OEBS_FLAGS:
            flag_value = validated_data.pop(flag, None)
            if (
                    flag_value is not None and
                    flag_value != getattr(instance, flag)
            ):
                updated_flags[flag] = flag_value

        return updated_flags

    def update(self, instance: Service, validated_data: dict):
        # редактируем стейт
        changing_state = renaming = False
        state = validated_data.pop('state', instance.state)
        requester = getattr(self.context['request'], 'person', None)

        if state != instance.state:
            if state == Service.states.DELETED and instance.has_moving_descendants():
                raise Conflict(
                    code='Cannot delete service with moving descendants',
                    title={
                        'ru': 'Нельзя удалить сервис, если он сам или его потомки находятся в состоянии перемещения',
                        'en': 'Cannot delete moving service or service with moving descendants'
                    },
                    message={
                        'ru': 'Нельзя удалить сервис, если он сам или его потомки находятся в состоянии перемещения',
                        'en': 'Cannot delete moving service or service with moving descendants'
                    }
                )
            if False and state not in Service.states.ACTIVE_STATES:
                try:
                    unclosed_services = instance.is_closable_from_d_side()
                except SupplierError as err:
                    log.info('HTTP error in get_unclosable_services: %s', err)
                else:
                    if unclosed_services:
                        raise Conflict(
                            code='cannot_close_service_with_issued_quotas',
                            message={
                                'ru': 'Нельзя закрыть или удалить сервис, у которого есть выданные квоты',
                                'en': 'Cannot close or delete a service that has issued quotas',
                            },
                        )
            # все действия приводящие к записи в базу мы сделаем после коммита
            # изменений самого сервиса
            changing_state = True
            if state in Service.states.ACTIVE_STATES:
                if instance.state == Service.states.CLOSED:
                    reopen_service.send(sender=instance.__class__, service=instance)
                instance.state = state

        name = validated_data.get('name', instance.name)
        name_en = validated_data.get('name_en', instance.name_en)
        if name != instance.name or name_en != instance.name_en:
            log.info('Changing readonly_state(RENAMING) on %s', instance.slug)

            oebs_related = is_oebs_related(instance, with_descendants=False)
            if oebs_related:
                old_name = instance.name
                old_name_en = instance.name_en
                # поменяем имя после согласования
                validated_data.pop('name_en')
                validated_data.pop('name')
            instance.readonly_state = Service.RENAMING
            renaming = True

        if validated_data.get('related_services'):
            pks = [x.pk for x in validated_data['related_services']]
            service_type = validated_data.get('service_type') or instance.service_type
            if Service.objects.filter(pk__in=pks, service_type=service_type).exists():
                raise BadRequest('Invalid related services')

        new_membership_inheritance = validated_data.get('membership_inheritance', instance.membership_inheritance)

        if (
            new_membership_inheritance != instance.membership_inheritance
        ):
            if not permissions.can_edit_membership_inheritance(instance, requester):
                raise PermissionDenied(
                    message={
                        'ru': "У вас недостаточно прав для изменения настройки наследования прав.",
                        'en': "You don't have sufficient permissions to update field 'membership_inheritance'."
                    }
                )
            if new_membership_inheritance and not requester.staff.user.is_superuser:
                raise PermissionDenied(
                    message={
                        'ru': "Включение наследования прав отключено, если оно вам необходимо - обратитесь в поддержку",
                        'en': "Enabling member_inheritance is disabled, if you need it, please, contact support",
                    }
                )
            services_to_notify_about = []
            for service in instance.get_ancestors(include_self=False).order_by('-level'):
                services_to_notify_about.append(service.id)
                if not service.membership_inheritance:
                    break

            for service_id in services_to_notify_about:
                tasks.notify_staff.delay_on_commit(service_id)

        tags = validated_data.get('tags')
        need_gradient_calculate = False
        new_review_policy = None
        if tags is not None:
            old_tags = instance.tags.all()
            validators.verify_tags_permissions(requester, old_tags, tags)
            if settings.RESTRICT_OEBS_TAGS:
                validators.validate_oebs_tags(instance, tags)
            new_gradient_tag = [tag.slug for tag in tags if tag.slug in settings.GRADIENT_TAGS]
            old_gradient_tags = {tag.slug for tag in old_tags if tag.slug in settings.GRADIENT_TAGS}
            validators.gradient_tags_validate(instance, requester, new_gradient_tag, old_gradient_tags)
            if set(new_gradient_tag) != old_gradient_tags:
                need_gradient_calculate = True

            enable_review_required = settings.REVIEW_REQUIRED_TAG_SLUGS.intersection({tag.slug for tag in tags})
            if enable_review_required ^ settings.REVIEW_REQUIRED_TAG_SLUGS.intersection({tag.slug for tag in old_tags}):
                new_review_policy = bool(enable_review_required)

        flags = validated_data.get('flags', False)
        if flags is not False:
            can_edit_flags(self.context['request'], instance.flags, flags)

        updated_flags = self._get_oebs_flags(
            instance=instance,
            validated_data=validated_data,
        )

        new_type = validated_data.get('service_type', False)
        if new_type is not False:
            if not new_type and instance.service_type_id:
                # пытаются убрать тип
                raise BadRequest(message={
                    'ru': 'Убирать тип у сервиса запрещено',
                    'en': 'It is forbidden to clean the type of service',
                })
            if new_type.id != instance.service_type_id:
                # меняют тип
                if instance.service_type.code != ServiceType.UNDEFINED:
                    raise BadRequest(message={
                        'ru': 'Изменение типа возможно только в неопределенных сервисов',
                        'en': 'Changing the type is only possible in undefined services',
                    })
                if not new_type.available_parents.filter(
                    code=instance.parent.service_type.code
                ).exists():
                    raise BadRequest(message={
                        'ru': 'Тип родителя данного сервиса является недопустимым для указанного',
                        'en': 'The parent type of this service is invalid for the specified',
                    })

        try:
            # super() метод редактирует все оставшиеся поля и коммитится в базу
            instance: Service = super(UpdateServiceSerializer, self).update(instance, validated_data)

        except django.core.exceptions.ValidationError as e:
            raise rest_framework.exceptions.ValidationError(getattr(e, 'message', str(e)), e.code)

        # пост-обработка с генерацией всяких тасков
        if changing_state:
            if state == Service.states.DELETED:
                ServiceDeleteRequest.request(instance, requester)

            elif state == Service.states.CLOSED:
                ServiceCloseRequest.request(instance, requester)

            if state not in Service.states.ACTIVE_STATES:
                tasks.drop_requests.delay(instance.id)

        if renaming:
            if is_oebs_related(instance, with_descendants=False):
                OEBSAgreement.objects.rename_service(
                    service=instance, requester=requester.staff,
                    name=name, name_en=name_en,
                    old_name=old_name,
                    old_name_en=old_name_en,
                )
            else:
                tasks.rename_service.apply_async(
                    args=(instance.id,),
                    countdown=settings.ABC_DEFAULT_COUNTDOWN,
                )

        if need_gradient_calculate:
            calculate_gradient_fields.delay(instance.id)

        if updated_flags:
            OEBSAgreement.objects.change_flags(
                service=instance,
                requester=self.context['request'].person.staff,
                updated_flags=updated_flags,
            )

        if new_review_policy is not None:
            update_service_review_policy.delay(service_slug=instance.slug, review_required=new_review_policy)

        tasks.notify_staff.delay(instance.id)

        return instance


class ServiceFilterForm(DjangoForm):

    def clean_is_exportable(self):
        if self.cleaned_data['is_exportable'] is None and 'is_exportable__in' not in self.data:
            self.cleaned_data['is_exportable'] = True
        return self.cleaned_data['is_exportable']


class ServiceFilter(PlanFilterSet):
    parent__with_descendants = base.ServiceWithDescendantsFilter(
        field_name='parent',
        queryset=Service.objects.all(),
        only_active=False,
    )
    parent__with_descendants__slug = base.ServiceWithDescendantsFilter(
        field_name='parent__slug',
        queryset=Service.objects.all(),
        to_field_name='slug',
        only_active=False,
    )
    tags = CustomModelMultipleChoiceFilter(
        field_name='tags',
        queryset=ServiceTag.objects.all(),
        distinct=True,
    )
    member = CustomModelMultipleChoiceFilter(
        field_name='members__staff__login',
        queryset=Staff.objects.active(),
        to_field_name='login',
        distinct=True,
        method='filter_by_member'
    )
    responsible = CustomModelChoiceFilter(
        queryset=Staff.objects.active(),
        to_field_name='login',
        method='filter_by_responsible'
    )
    department = CustomModelMultipleChoiceFilter(
        field_name='members__staff__department',
        queryset=Department.objects.all(),
        distinct=True,
        method='filter_by_department'
    )
    type = CustomModelMultipleChoiceFilter(
        field_name='service_type__code',
        queryset=ServiceType.objects.all(),
        to_field_name='code',
        distinct=True
    )
    is_suspicious = base.BooleanFilter(field_name='suspicious_date__isnull', exclude=True)

    def filter_by_department(self, queryset, name, value):
        if value == settings.NULL_VALUE:
            return queryset.none()

        if value:
            queryset = queryset.filter(
                members__staff__department__in=value,
                members__state__in=[SERVICEMEMBER_STATE.ACTIVE, SERVICEMEMBER_STATE.DEPRIVING],
            ).distinct()
        return queryset

    def filter_by_member(self, queryset, name, value):
        if value == settings.NULL_VALUE:
            return queryset.none()

        if value:
            queryset = queryset.filter(
                members__staff__in=value,
                members__state__in=[SERVICEMEMBER_STATE.ACTIVE, SERVICEMEMBER_STATE.DEPRIVING],
            ).distinct()
        return queryset

    def filter_by_responsible(self, queryset, name, value):
        if value == settings.NULL_VALUE:
            return queryset.none()

        if value:
            queryset = queryset.filter(
                members__staff__login=value.login,
                members__role__code__in=Role.RESPONSIBLES,
                members__state__in=[SERVICEMEMBER_STATE.ACTIVE, SERVICEMEMBER_STATE.DEPRIVING],
            ).distinct()
        return queryset

    class Meta:
        model = Service
        form = ServiceFilterForm
        fields = {
            'has_external_members': ['exact', 'in'],
            'id': ['exact', 'in', 'lt', 'gt'],
            'is_exportable': ['exact', 'in'],
            'is_suspicious': ['exact', 'in'],
            'keywords': ['exact', 'contains'],
            'member': ['exact'],
            'name': ['exact', 'contains'],
            'name_en': ['exact', 'contains'],
            'parent': ['exact', 'in', 'isnull'],
            'parent__slug': ['exact', 'in'],
            'readonly_state': ['exact', 'in'],
            'slug': ['exact', 'contains', 'in'],
            'state': ['exact', 'in'],
            'tags': ['exact'],
            'tags__slug': ['exact', 'in'],
            'type': ['exact', 'in']
        }


class ServicesMetadata(BaseMetadata):
    def determine_metadata(self, request, view):
        view_permissions = set()

        view_permissions.add('can_edit_description')
        view_permissions.add('can_edit_activity')

        if permissions.is_service_support(request.person):
            view_permissions.add('can_manage_responsible')
            view_permissions.add('can_edit_contacts')

        if settings.CROWDTEST:
            view_permissions.add('can_manage_responsible')

        try:
            service = view.get_object()

            if permissions.is_team_member(service, request.person):
                view_permissions.add('can_edit_tags')
                view_permissions.add('is_service_member')
                view_permissions.add('can_edit_contacts')

            if permissions.is_service_responsible(service, request.person):
                view_permissions.add('can_manage_responsible')
                view_permissions.add('can_fix_issues')
                view_permissions.add('can_edit_oebs_flags')

            if service.readonly_state is None:
                view_permissions.add('can_request_move')

            if permissions.can_mark_as_not_suspicious(service, request.person):
                view_permissions.add('can_mark_as_not_suspicious')

            if permissions.can_add_forced_suspicious_reason(service, request.person):
                view_permissions.add('can_add_forced_suspicious_reason')

            if permissions.can_remove_forced_suspicious_reason(service, request.person):
                view_permissions.add('can_remove_forced_suspicious_reason')

            if permissions.can_edit_membership_inheritance(service, request.person):
                view_permissions.add('can_edit_membership_inheritance')

            if permissions.can_edit_duty_settings(service, request.person):
                view_permissions.add('can_approve_shifts')
                view_permissions.add('can_edit_duty_settings')

        except AssertionError:  # выскочит если вызывают там, где нет инстанса
            pass

        return {
            'permissions': sorted(view_permissions),
            'options': {'type': ServiceTypeSerializer(ServiceType.objects.all(), many=True).data}
        }


class ServicesView(HistoryMixin, FieldsByPermissionsCutterMixin,
                   TvmAccessMixin, SelectOnlyFieldsMixin,
                   base.OrderingMixin, viewsets.ModelViewSet):
    filter_class = ServiceFilter
    search_fields = ('slug', 'name', 'name_en')
    http_method_names = ['get', 'post', 'patch', 'options']
    _queryset = Service.objects.all()
    permission_classes = [ServicePermissions]
    metadata_class = ServicesMetadata
    TVM_ALLOWED_METHODS = {'GET', 'PATCH'}
    ordering_fields = (
        'id',
        'name',
        'name_en',
        'kpi_bugs_count',
        'kpi_release_count',
        'kpi_lsr_count',
        'departments',
        'related_services'
    )
    detail_serializer = DetailServiceSerializer
    update_serializer = UpdateServiceSerializer
    create_serializer = CreateServiceSerializer
    list_serializer = ListServiceSerializer
    _permissions_to_proceed = 'view_own_services'

    _EMPTY = 'empty'
    PERMISSION_CODENAME_MAPPER = {
        'view_own_services': {_EMPTY, },
        'view_team': {
            'unique_immediate_members_count',
            'unique_immediate_robots_count',
            'unique_immediate_external_members_count',
            'has_external_members',
        },
        'view_hierarchy': {'ancestors', 'descendants_count', 'children_count', },
        'can_view': {
            'readonly_state', 'id', 'slug',
            'name', 'level', 'matched', 'parent',
            'owner', 'human_readonly_state',
            'use_for_hardware', 'use_for_hr', 'use_for_group_only',
            'use_for_procurement', 'use_for_revenue', 'flags', 'functions',
        },
        'view_department': {'departments', },
        'view_all_services': {_EMPTY, },
        'view_duty': {_EMPTY, },
        'view_kpi': {'kpi', },
        'view_tags': {'tags', },
        'view_details': {
            'is_exportable', 'is_suspicious',
            'state', 'state_display', 'state_display_i18n',
            'created_at', 'sandbox_move_date', 'keywords',
            'modified_at', 'membership_inheritance', 'path',
            'type', 'related_services', 'has_forced_suspicious_reason',
            'issue', 'oebs_agreement', 'actions', 'available_states',
            'is_important',
        },
        'view_activity': {'activity', },
        'view_description': {'description', },
        'can_export': {_EMPTY, },
        'can_edit': {_EMPTY, },
        'view_hardware': {_EMPTY, },
        'view_resources': {_EMPTY, },
        'view_traffic_light': {'traffic_light', },
    }

    @property
    def queryset(self):
        user_can_view_all_services = self.request.user.has_perm('internal_roles.view_all_services')

        queryset = self._queryset
        if not user_can_view_all_services:
            queryset = self.get_restricted_queryset()

        return queryset

    def get_restricted_queryset(self):
        """Возвращает только сервисы, в которых состоит пользователь и их предков"""
        requester = None
        if self.request.user and hasattr(self.request.user, 'staff'):
            requester = self.request.user.staff
        return self._queryset.own_only(requester)

    def get_serializer_class(self):
        if self.action == 'create':
            return self.create_serializer

        if self.action in ('update', 'partial_update'):
            return self.update_serializer

        if self.action == 'retrieve':
            return self.detail_serializer

        return self.list_serializer

    def perform_create(self, serializer):
        """
            Создание сервиса
        """
        parent = serializer.validated_data.get('parent')
        requester = self.request.user.staff

        if not Service.can_staff_place_service_here(requester, parent, False):
            raise PermissionDenied(
                code='incorrect_parent',
                detail='incorrect service parent',
                title={
                    'ru': 'Неправильный родитель сервиса',
                    'en': 'Incorrect service parent',
                },
                message={
                    'ru': 'Вы не можете сделать сервис потомком "{}"'.format(parent.name),
                    'en': 'You can\'t make service a descendant of "{}"'.format(parent.name_en)
                }
            )

        tags = serializer.validated_data.get('tags')

        if tags is not None:
            validators.verify_tags_permissions(requester, [], tags)
            if settings.RESTRICT_OEBS_TAGS:
                validators.validate_oebs_tags(None, tags)
            old_gradient_tags = set()
            new_gradient_tag = [tag.slug for tag in tags if tag.slug in settings.GRADIENT_TAGS]
            validators.gradient_tags_validate(None, requester, new_gradient_tag, old_gradient_tags, parent)

        is_exportable = False
        initial_parent = Service.objects.get(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG)

        can_be_incoming_approver = permissions.is_service_responsible(
            person=self.request.person,
            service=parent,
        )
        if can_be_incoming_approver:
            is_exportable = True
            initial_parent = parent

        service = serializer.save(
            parent=initial_parent,
            readonly_state='creating',
            readonly_start_time=timezone.now(),
            is_exportable=is_exportable,
        )

        log.info('Service was created by %s with id=%s', requester.login, service.id)

        create_request = ServiceCreateRequest.request(
            service=service,
            requester=self.request.person,
            move_to=parent,
        )
        tasks.register_service.apply_async(
            args=(create_request.id,),
            countdown=settings.ABC_DEFAULT_COUNTDOWN
        )
        self.create_history_entry(serializer.instance)

    @action(methods=['get'], detail=True)
    def requested_roles(self, request, pk=None):
        service = self.get_object()
        roles = get_requested_roles_for_service(service)
        serializer = IdmRoleSerializer(roles, many=True)
        return Response(serializer.data)

    @action(methods=['get'], detail=True)
    def suggest_parent(self, request, pk=None):
        service = self.get_object()

        if not service.get_ancestors().filter(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG).exists():
            return Response([])

        try:
            services_qs = service.members.owners().select_related('staff__department')
            department = services_qs.latest('created_at').staff.department
        except ServiceMember.DoesNotExist:
            return Response([])

        if department is None:
            return Response([])

        possible_parents = (
            Service
            .objects
            .filter(departments__in=department.get_ancestors(include_self=True))
            .order_by('-level')[:5]
        )

        serializer = CompactServiceSerializer(possible_parents, many=True)
        return Response(serializer.data)

    @action(methods=['get'], detail=True)
    def suspicious_reasons(self, request, pk=None):
        service = self.get_object()
        serializer = ServiceSuspiciousReasonSerializer(service.suspicious_reasons.filter(marked_by=None), many=True)
        return Response(serializer.data)

    def old_on_duty(self, request):
        service = self.get_object()
        now = timezone.now()
        queryset = (
            Shift.objects
            .filter(
                schedule__service=service,
                start_datetime__lte=now,
                end_datetime__gt=now,
                state__in=[Shift.STARTED, Shift.SCHEDULED],
            ).select_related('schedule', 'replace_for')
        )

        if not_workday(utils.today()):
            queryset = queryset.filter(schedule__only_workdays=False)

        filtered_shifts = ShiftBaseFilterSet(request.query_params, queryset=queryset, request=request).qs
        current_shifts = filtered_shifts.remove_replaced_shifts()

        serializer = ShiftBaseSerializer(current_shifts, many=True)
        return Response(serializer.data)

    @action(methods=['get'], detail=True)
    def on_duty(self, request, pk=None):
        # При включенном гранулярном доступе в request.query_params так же приходят, например,
        # разрешенные для роли поля, которые имеют отношения к ручкам services и никак не коррелируют с ручками duty
        # Поэтому кажется проще параметры для фильтрации взять из урла и добавить туда ещё сервис
        query = urllib.parse.urlparse(request.build_absolute_uri()).query
        querydict = QueryDict(query, mutable=True)
        querydict['service'] = int(pk)
        updated_query = querydict.urlencode(safe='/')

        if waffle.switch_is_active('redirect_on_duty'):
            return HttpResponseRedirect(reverse('api-v4:duty-on-duty-list') + '?' + updated_query)

        return self.old_on_duty(request)

    @action(methods=['post'], detail=True)
    def add_forced_suspicious_reason(self, request, pk=None):
        service = self.get_object()
        if service.suspicious_reasons.filter(reason=ServiceSuspiciousReason.FORCED, marked_by=None).exists():
            raise BadRequest({
                'ru': 'Сервис уже помечен как подозрительный',
                'en': 'Service wal already marked as suspicious',
            })

        if not permissions.can_add_forced_suspicious_reason(service, request.person):
            raise PermissionDenied()

        service.suspicious_reasons.create(
            reason=ServiceSuspiciousReason.FORCED, context={'parent_owner': request.person.staff.login}
        )

        service.has_forced_suspicious_reason = True
        if not service.is_suspicious:
            ServiceNotification.objects.get_or_create(
                service=service,
                notification_id=ServiceNotification.CHANGES_SUSPICION_DIGEST,
                sent_at=None,
                recipient=None,
                defaults={'old_suspicious_date': service.suspicious_date}
            )
            service.suspicious_date = timezone.now().date()
            service.save(update_fields=['has_forced_suspicious_reason', 'suspicious_date'])
            process_suspicious_service.apply_async(args=[service.id])
        else:
            service.save(update_fields=['has_forced_suspicious_reason'])

        return Response()

    @action(methods=['post'], detail=True)
    def remove_forced_suspicious_reason(self, request, pk=None):
        service = self.get_object()
        if not service.suspicious_reasons.filter(reason=ServiceSuspiciousReason.FORCED, marked_by=None).exists():
            raise BadRequest({
                'ru': 'Сервис не помечен как подозрительный',
                'en': 'Service is not marked as suspicious',
            })

        if not permissions.can_remove_forced_suspicious_reason(service, request.person):
            raise PermissionDenied()

        service.suspicious_reasons.filter(reason=ServiceSuspiciousReason.FORCED).update(marked_by=request.person.staff)

        if not service.suspicious_reasons.filter(marked_by=None).exists():
            service.remove_suspicion()
            process_suspicious_service.apply_async(args=[service.id])
        else:
            service.has_forced_suspicious_reason = False
            service.save(update_fields=['has_forced_suspicious_reason'])

        return Response()

    @action(methods=['post'], detail=True)
    def mark_as_not_suspicious(self, request, pk=None):
        text = request.data.get('text')
        if not text:
            raise BadRequest(message={'ru': '"text" должен быть передан', 'en': '"text" should be provided'})

        service = self.get_object()
        if not service.is_suspicious:
            raise BadRequest(message={'ru': 'Сервис не является подозрительным', 'en': 'Service is not suspicious'})

        if not permissions.can_mark_as_not_suspicious(service, request.person):
            raise PermissionDenied()

        service.suspicious_reasons.exclude(
            reason__in=ServiceSuspiciousReason.UNREMOVABLE_REASONS
        ).update(marked_by=request.person.staff, marking_text=text)

        if not service.suspicious_reasons.filter(
                reason__in=ServiceSuspiciousReason.UNREMOVABLE_REASONS,
                marked_by=None
        ).exists():
            service.remove_suspicion()
            process_suspicious_service.apply_async(args=[service.id])

        return Response()

    @action(methods=['get'], detail=True)
    def issuegroups(self, request, pk=None):
        user_perms = get_internal_roles(request.user)
        if 'can_edit' not in user_perms:
            raise PermissionDenied()
        groups = grouped_service_issues(pk)
        serializer = IssueGroupWithIssuesSerializer(groups, context={'service_id': pk}, many=True)
        return Response(serializer.data)


class V4ServicesView(ServicesView, DefaultFieldsMixin):
    default_swagger_schema = SwaggerServices

    create_serializer = V4CreateServiceSerializer
    pagination_class = base.ABCCursorPagination
    ordering_fields = ('id',)

    default_fields = [
        'id',
        'readonly_state',
        'slug',
        'state',
    ]

    def dispatch(self, request, *args, **kwargs):
        self.check_fields(request)
        return super(V4ServicesView, self).dispatch(request, *args, **kwargs)

    def list(self, *args, **kwargs):
        """
        По умолчанию возвращает список только экспортируемых сервисов.
        """
        return super(V4ServicesView, self).list(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['post'], detail=True)
    def add_forced_suspicious_reason(self, *args, **kwargs):
        return super(V4ServicesView, self).add_forced_suspicious_reason(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['get'], detail=True)
    def issuegroups(self, *args, **kwargs):
        return super(V4ServicesView, self).issuegroups(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['post'], detail=True)
    def mark_as_not_suspicious(self, *args, **kwargs):
        return super(V4ServicesView, self).mark_as_not_suspicious(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['get'], detail=True)
    def on_duty(self, *args, **kwargs):
        return super(V4ServicesView, self).on_duty(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['post'], detail=True)
    def remove_forced_suspicious_reason(self, *args, **kwargs):
        return super(V4ServicesView, self).remove_forced_suspicious_reason(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['get'], detail=True)
    def requested_roles(self, *args, **kwargs):
        return super(V4ServicesView, self).requested_roles(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['get'], detail=True)
    def suggest_parent(self, *args, **kwargs):
        return super(V4ServicesView, self).suggest_parent(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['get'], detail=True)
    def suspicious_reasons(self, *args, **kwargs):
        return super(V4ServicesView, self).suspicious_reasons(*args, **kwargs)


class ScopeCounterSerializer(serializers.Serializer):
    scope_slug = serializers.CharField(source='role__scope__slug')
    members_count = serializers.IntegerField()


class ScopeCounterView(TvmAccessMixin, NoPaginationListMixin, mixins.ListModelMixin, viewsets.GenericViewSet):
    """
    Счетчики количества людей в скоупах сервиса
    """
    default_swagger_schema = SwaggerFrontend

    _permissions_to_proceed = 'can_view'
    serializer_class = ScopeCounterSerializer
    permission_classes = [TvmAuthenticated]

    def get_queryset(self):
        return ServiceMember.objects.team().values('role__scope__slug').annotate(members_count=Count('id'))

    def filter_queryset(self, queryset):
        service_id = self.request.query_params.get('service_id')
        if not service_id:
            raise BadRequest(message={
                'ru': 'Не передан service_id',
                'en': 'Service_id is not provided',
            })
        min_count = self.request.query_params.get('min_count', settings.HUGE_SCOPE_CUTOFF)
        return queryset.filter(service_id=service_id, members_count__gte=min_count)


class FrontendPuncherRuleSerializer(serializers.Serializer):
    scope = serializers.CharField()
    destination = serializers.CharField()
    destination_link = serializers.URLField()
    locations = serializers.ListField()
    ports = serializers.ListField()
    created = serializers.CharField()
    until = serializers.CharField()
    link = serializers.URLField()
    comment = serializers.CharField()


class FrontendPuncherRulesSerializer(serializers.Serializer):
    next = serializers.URLField()
    count = serializers.IntegerField()
    results = FrontendPuncherRuleSerializer(many=True)


class FrontendPuncherRules(TvmAccessMixin, mixins.ListModelMixin, viewsets.GenericViewSet):
    pagination_class = None
    serializer_class = FrontendPuncherRulesSerializer

    def get_serializer(self, *args, **kwargs):
        kwargs.pop('many')
        return super().get_serializer(*args, **kwargs)

    def format_rules(self, puncher_response, service) -> list:
        rules = []
        for rule in puncher_response:
            for destination in rule['destinations']:
                for source in rule['sources']:
                    svc_key = f'_svc_{service.slug}_'
                    if (
                        source['type'] in ('servicerole', 'service', ) and
                        (
                            f'_svc_{service.slug}@' in source['machine_name'] or
                            svc_key in source['machine_name']
                        )
                    ):
                        scope = None
                        if source['type'] == 'servicerole':
                            scope = source['machine_name'].strip('@').split(svc_key)[-1]
                        rules.append(
                            {
                                'scope': scope,
                                'ports': rule['ports'],
                                'locations': rule['locations'],
                                'until': rule['until'],
                                'destination': destination['machine_name'],
                                'destination_link': destination.get('url'),
                                'created': rule['added'],
                                'comment': rule['comment'],
                                'link': f'https://puncher.yandex-team.ru/?id={rule["id"]}',
                            }
                        )
        return rules

    def get_queryset(self):
        cursor_id = self.request.GET.get('cursor_id')
        service_id = self.request.GET.get('service_id')
        if not service_id:
            raise BadRequest(message={
                'ru': 'Необходимо указать service_id в запросе',
                'en': 'You must specify the service_id in the request',
            })
        service = Service.objects.filter(pk=service_id).first()
        if not service:
            raise BadRequest(message={
                'ru': 'Указанного сервиса не существует',
                'en': 'The specified service does not exist',
            })

        client = PuncherClient()
        puncher_response, count, next_cursor_id = client.get_rules(service_id=service_id, cursor_id=cursor_id)

        next_link = None
        if next_cursor_id:
            next_link = ABCPagination.replace_query_param(
                self.request.build_absolute_uri(),
                'cursor_id',
                next_cursor_id,
            )

        return {
            'count': count,
            'next': next_link,
            'results': self.format_rules(
                puncher_response=puncher_response,
                service=service
            ),
        }

    def filter_queryset(self, queryset):
        return queryset


class FrontendIDMRoleSerializer(serializers.Serializer):
    system = serializers.CharField()
    system_slug = serializers.CharField()
    scope = serializers.CharField()
    until = serializers.CharField()
    link = serializers.CharField()
    role = serializers.CharField()
    fields_data = serializers.JSONField()
    human_state = serializers.CharField()
    state = serializers.CharField()
    updated = serializers.CharField()
    granted = serializers.CharField()
    created = serializers.CharField()
    with_inheritance = serializers.BooleanField()
    with_robots = serializers.BooleanField()
    with_external = serializers.BooleanField()


class FrontendIDMRolesSerializer(serializers.Serializer):
    next = serializers.URLField()
    count = serializers.IntegerField()
    results = FrontendIDMRoleSerializer(many=True)


class FrontendIDMRoles(TvmAccessMixin, mixins.ListModelMixin, viewsets.GenericViewSet):
    pagination_class = None
    serializer_class = FrontendIDMRolesSerializer

    def get_serializer(self, *args, **kwargs):
        kwargs.pop('many')
        return super().get_serializer(*args, **kwargs)

    def format_roles(self, roles, service) -> list:
        result = []
        for role in roles:
            link = f'{settings.IDM_HOST}/group/{role["group"]["id"]}/roles#f-status=all,sort-by=-updated,role={role["id"]}'
            scope = None
            if role['group']['slug'] != f'svc_{service.slug}':
                scope = role['group']['slug'].split(f'_svc_{service.slug}_')[-1]

            result.append(
                {
                    'system': role['system']['name'],
                    'system_slug': role['system']['slug'],
                    'scope': scope,
                    'until': role['expire_at'],
                    'link': link,
                    'role': role['human_short'],
                    'fields_data': role['fields_data'],
                    'human_state': role['human_state'],
                    'state': role['state'],
                    'updated': role['updated'],
                    'granted': role['granted_at'],
                    'created': role['added'],
                    'with_inheritance': role['with_inheritance'],
                    'with_robots': role['with_robots'],
                    'with_external': role['with_external'],
                }
            )
        return result

    def get_queryset(self):
        cursor_id = int(self.request.GET.get('cursor_id', 0))
        page_size = int(self.request.GET.get('page_size', 20))
        service_id = self.request.GET.get('service_id')
        if not service_id:
            raise BadRequest(message={
                'ru': 'Необходимо указать service_id в запросе',
                'en': 'You must specify the service_id in the request',
            })
        service = Service.objects.filter(pk=service_id).first()
        if not service:
            raise BadRequest(message={
                'ru': 'Указанного сервиса не существует',
                'en': 'The specified service does not exist',
            })

        try:
            roles = actions.get_roles_for_service(
                service=service,
                offset=cursor_id,
                limit=page_size,
                states=ACTIVE_STATES+INACTIVE_STATES,
                requester=self.request.person,
                timeout=settings.IDM_GET_ROLES_TIMEOUT,
                retry=settings.ABC_IDM_FROM_API_RETRY,
            )
        except TimeoutError:
            raise FailedDependency()

        next_link = None
        if roles['total_count'] > roles['next_offset']:
            next_link = ABCPagination.replace_query_param(
                self.request.build_absolute_uri(),
                'cursor_id',
                roles['next_offset'],
            )

        return {
            'count': roles['total_count'],
            'next': next_link,
            'results': self.format_roles(
                roles=roles['objects'],
                service=service
            ),
        }

    def filter_queryset(self, queryset):
        return queryset
