from pytils.translit import translify
from django.template.defaultfilters import slugify

from rest_framework import viewsets
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from plan.api.exceptions import BadRequest
from plan.duty.models import Schedule
from plan.roles.models import RoleScope
from plan.services.models import Service, SERVICE_SLUG_PATTERN
from plan.swagger import SwaggerFrontend

CHECK_FOR_SLUG_LEN = 25
MAX_SLUG_LEN = 50
SERVICE_SLUG_TYPE = 'service'
SCHEDULE_SLUG_TYPE = 'schedule'
SLUG_TYPES = {
    SERVICE_SLUG_TYPE,
    SCHEDULE_SLUG_TYPE,
}


def is_unique(slug, existing_slugs, role_scope_slugs):
    return slug not in existing_slugs and ('_' not in slug or slug.rsplit('_')[-1] not in role_scope_slugs)


def make_unique_slug(initial_slug, slug_type=SERVICE_SLUG_TYPE, service_id=None):
    role_scope_slugs = set()
    existing_slugs = set()

    if slug_type == SERVICE_SLUG_TYPE:
        existing_slugs = set(
            Service.objects.filter(slug__startswith=initial_slug[:CHECK_FOR_SLUG_LEN]).values_list('slug', flat=True)
        )
        role_scope_slugs = set(RoleScope.objects.values_list('slug', flat=True)) if '_' in initial_slug else set()

    elif slug_type == SCHEDULE_SLUG_TYPE:
        service_slug = Service.objects.get(pk=service_id).slug
        if not initial_slug.startswith(f'{service_slug}_'):
            initial_slug = f'{service_slug}_{initial_slug}'
        existing_slugs = set(
            Schedule.objects.filter(slug__startswith=initial_slug[:CHECK_FOR_SLUG_LEN], service_id=service_id).values_list('slug', flat=True)
        )

    i = 1
    suffix = ''
    while not is_unique(initial_slug[:MAX_SLUG_LEN - len(suffix)] + suffix, existing_slugs, role_scope_slugs):
        suffix = str(i)
        i += 1

    return initial_slug[:MAX_SLUG_LEN - len(suffix)] + suffix


class SlugMixin:
    def get_request_data(self, request):
        service_id = request.query_params.get('service', '')
        slug_type = request.query_params.get('type', SERVICE_SLUG_TYPE)

        if slug_type != SERVICE_SLUG_TYPE and not service_id:
            raise BadRequest(message={
                'ru': 'Сервис не был передан в параметрах',
                'en': 'No service provided in request',
            })
        if slug_type not in SLUG_TYPES:
            raise BadRequest(message={
                'ru': 'Неизвестный тип слага',
                'en': 'Unknown slug type',
            })
        return slug_type, service_id


class MakeSlugView(SlugMixin, viewsets.ViewSet):
    default_swagger_schema = SwaggerFrontend

    _permissions_to_proceed = 'can_edit'
    permission_classes = [IsAuthenticated]

    def get_response_data(self, request):
        name = request.query_params.get('name', '')
        initial_slug = slugify(translify(name)).replace('-', '_')
        slug_type, service_id = self.get_request_data(request)
        return {
            'slug': make_unique_slug(
                initial_slug,
                slug_type=slug_type,
                service_id=service_id,
            )
        }

    def list(self, request):
        return Response(self.get_response_data(request))


class MakeSlugFrontendApiView(MakeSlugView):
    def list(self, request):
        return Response(
            {
                'content': self.get_response_data(request),
                'error': {},
            }
        )


class ValidateSlugView(SlugMixin, viewsets.ViewSet):
    default_swagger_schema = SwaggerFrontend

    _permissions_to_proceed = 'can_edit'
    permission_classes = [IsAuthenticated]

    def list(self, request):
        slug = request.query_params.get('slug', '')
        slug_type, service_id = self.get_request_data(request)
        if not SERVICE_SLUG_PATTERN.match(slug):
            raise BadRequest(message={
                'ru': 'Неверный слаг. '
                      'Слаг должен состоять из латинских букв в нижнем регистре, цифр, знаков подчеркивания или дефиса.',
                'en': 'Slug is invalid. '
                      'Slug should contain only lower alphanumeric, underscore and hyphen characters.',
            })
        if len(slug) > MAX_SLUG_LEN:
            raise BadRequest(message={
                'ru': f'Неверный слаг. '
                      f'Длина слага должна быть не более {MAX_SLUG_LEN} символов.',
                'en': f'Slug is invalid. '
                      f'Slug length should not exceed {MAX_SLUG_LEN} characters.',
            })
        if slug.isdigit():
            raise BadRequest(message={
                'ru': 'Неверный слаг. '
                      'Слаг не может состоять из одних цифр.',
                'en': 'Slug is invalid. '
                      'Slug can\'t contain only digits.',
            })

        unique_slug = make_unique_slug(
            slug,
            slug_type=slug_type,
            service_id=service_id,
        )

        if unique_slug == slug:
            data = {
                'valid': True,
                'alternative': None
            }
        else:
            data = {
                'valid': False,
                'alternative': unique_slug
            }

        return Response(data)
