import logging

from django.conf import settings
from rest_framework import viewsets
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from plan.api import base
from plan.api.fields import MappingField
from plan.services.api import validators
from plan.services.models import Service, ServiceTag
from plan.swagger import SwaggerFrontend

log = logging.getLogger(__name__)


class ServiceTagSerializer(base.ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    description = MappingField({'ru': 'description', 'en': 'description_en'})
    fields_mapping_ = {
        'name': ('name', 'name_en'),
        'description': ('description', 'description_en'),
    }

    class Meta:
        model = ServiceTag
        fields = (
            'color',
            'description',
            'id',
            'name',
        )


class ServiceTagsView(base.OrderingMixin, viewsets.ModelViewSet):
    serializer_class = ServiceTagSerializer
    http_method_names = ['get']
    queryset = ServiceTag.objects.all()


class ValidateTagsView(viewsets.ViewSet):
    """
    Валидируем переданные теги при создании сервиса на градиентные и OEBS-ные соответствия.
    Параметры:
        * parent - родитель нового сервиса, для которого валидируем теги;
        * tags - список кодов тегов;
    """
    default_swagger_schema = SwaggerFrontend

    permission_classes = [IsAuthenticated]
    _permissions_to_proceed = 'can_edit'

    def list(self, request):
        params = {k: v for k, v in request.query_params.lists()}

        parent = None
        if params.get('parent'):
            parent = Service.objects.filter(pk__in=params.get('parent',)).first()
        tags = ServiceTag.objects.filter(slug__in=params.get('tags', []))

        if tags is not None:
            requester = self.request.user.staff
            validators.verify_tags_permissions(requester, [], tags)
            new_gradient_tag = [tag.slug for tag in tags if tag.slug in settings.GRADIENT_TAGS]
            old_gradient_tags = set()

            validators.gradient_tags_validate(None, requester, new_gradient_tag, old_gradient_tags, parent)
            if settings.RESTRICT_OEBS_TAGS:
                validators.validate_oebs_tags(None, tags)

        return Response()
