from django.conf import settings
from django.db.models.expressions import RawSQL

from rest_framework import serializers, viewsets
from rest_framework.generics import get_object_or_404
from rest_framework.response import Response

from plan.api.mixins import TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from plan.api.serializers import BasePlanModelSerializer
from plan.roles.models import Role, RoleScope
from plan.services.models import ServiceMember, Service
from plan.services.permissions import can_remove_member_bulk
from plan.staff.models import Staff, Department
from plan.swagger import SwaggerFrontend


class PersonSerializer(BasePlanModelSerializer):
    firstName = serializers.CharField(source='i_first_name')
    lastName = serializers.CharField(source='i_last_name')
    isDismissed = serializers.BooleanField(source='is_dismissed')
    fullName = serializers.SerializerMethodField()

    def __init__(self, *args, **kwargs):
        self.prefetched_data = kwargs.pop('prefetched_data', {})
        super(PersonSerializer, self).__init__(*args, **kwargs)

    def get_fullName(self, obj):
        return obj.get_full_name()

    class Meta:
        model = Staff
        fields = [
            'id',
            'login',
            'firstName',
            'lastName',
            'fullName',
            'isDismissed',
            'is_robot',
            'affiliation',
            'is_frozen',
        ]
        read_only_fields = fields


class PersonWithABCExtSerializer(PersonSerializer):
    abc_ext = serializers.SerializerMethodField()

    def get_abc_ext(self, obj):
        return obj.get_abc_ext()

    class Meta:
        model = Staff
        fields = PersonSerializer.Meta.fields + ['abc_ext']
        read_only_fields = fields


class PersonWithRoleSerializer(PersonWithABCExtSerializer):
    default_service_role = serializers.SerializerMethodField()

    @classmethod
    def get_role_spec(cls, default_role_for=None):
        role_field = 'default_{}_role'.format(default_role_for)
        getter_name = 'get_{}'.format(role_field)
        if not hasattr(cls, getter_name):
            raise ValueError("No such default role")
        return role_field

    def get_default_service_role(self, obj):
        default_service_roles = self.prefetched_data['default_service_roles']
        return RoleSerializer(default_service_roles[obj.id]).data

    class Meta:
        model = Staff
        fields = PersonWithABCExtSerializer.Meta.fields + ['default_service_role']
        read_only_fields = fields


class RoleScopeSerializer(BasePlanModelSerializer):
    id = serializers.CharField(source='slug')
    name = serializers.CharField(source='i_name')
    scope_id = serializers.IntegerField(source='id')

    class Meta:
        model = RoleScope
        fields = [
            'id',
            'name',
            'scope_id'
        ]
        read_only_fields = fields


class RoleSerializer(BasePlanModelSerializer):
    service = serializers.IntegerField(source='service_id')
    scope = RoleScopeSerializer()
    name = serializers.CharField(source='i_name')

    class Meta:
        model = Role
        fields = [
            'id',
            'code',
            'name',
            'scope',
            'service',
            'is_exportable',
        ]
        read_only_fields = fields


class DepartmentSerializer(BasePlanModelSerializer):
    name = serializers.CharField(source='i_name')

    class Meta:
        model = Department
        fields = [
            'id',
            'name',
        ]
        read_only_fields = fields


class ServiceMemberSerializer(BasePlanModelSerializer):
    person = PersonSerializer(source='staff')
    role = RoleSerializer()
    state = serializers.SerializerMethodField()
    serviceMemberDepartmentId = serializers.IntegerField(source='from_department_id')
    fromDepartment = DepartmentSerializer(source='from_department.department', required=False)
    role_url = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()

    def get_role_url(self, obj):
        role_path = f'abc/services{obj.service.path}*/{obj.role.id}'
        ownership = 'group' if obj.from_department else 'personal'
        role_filter = f'f-role={role_path},f-user={obj.staff.login},f-ownership={ownership}'
        return settings.IDM_ROLE_FILTER_URL.format(filter=role_filter)

    def get_state(self, obj):
        return 'approved'

    def get_actions(self, obj):
        can_remove_member = self.context.get('can_remove_member', {obj: False})
        actions = []
        is_service_alive = obj.service.state in Service.states.ALIVE_STATES
        if is_service_alive and can_remove_member[obj]:
            actions.append('member_remove')
        return actions

    class Meta:
        model = ServiceMember
        fields = [
            'id',
            'person',
            'role',
            'serviceMemberDepartmentId',
            'fromDepartment',
            'state',
            'actions',
            'role_url',
            'expires_at',
        ]
        read_only_fields = fields


class TeamView(TvmAccessMixin, viewsets.ViewSet):
    default_swagger_schema = SwaggerFrontend

    permission_classes = [TvmAuthenticated]
    _permissions_to_proceed = 'view_team'

    def _get_max_scope_members(self):
        max_scope_members = self.request.query_params.get('max_scope_members')

        if max_scope_members:
            try:
                return int(max_scope_members)
            except ValueError:
                pass

    def _get_scope(self):
        return self.request.query_params.get('scope__slug')

    def retrieve(self, request, pk=None):
        service = get_object_or_404(Service, pk=pk)
        queryset = (
            service.members
            .team()
            .select_related('role', 'role__scope', 'from_department', 'from_department__department', 'staff')
        )
        scope = self._get_scope()
        if scope:
            queryset = queryset.filter(role__scope__slug=scope)

        max_scope_members = self._get_max_scope_members()

        if max_scope_members is not None:
            queryset = queryset.filter(
                id__in=RawSQL(
                    ("SELECT window_table.id FROM (SELECT m.id, m.staff_id, r.scope_id, "
                     "row_number() OVER (PARTITION BY r.scope_id ORDER BY s.is_robot, s.first_name, s.last_name, r.id) "
                     "FROM services_servicemember m "
                     "inner join roles_role r on r.id = m.role_id {}"
                     "inner join intranet_staff s on s.id = m.staff_id "
                     "where m.service_id = %s and m.state in (%s, %s) and NOT r.code = %s {}) "
                     "window_table WHERE window_table.row_number <= %s ").format(
                        '' if not scope else 'inner join roles_rolescope rs on rs.id = r.scope_id ',
                        '' if not scope else 'and rs.slug = \'{}\' '.format(scope),
                    ),
                    (
                        service.id, ServiceMember.states.ACTIVE,
                        ServiceMember.states.DEPRIVING,
                        Role.RESPONSIBLE,
                        max_scope_members,

                    )
                )
            )
        order_fields = (
            'staff__is_robot', 'staff__first_name',
            'staff__last_name', 'role_id',
        )

        queryset = queryset.order_by(*order_fields)

        can_remove_member = can_remove_member_bulk(queryset, request.person)
        serializer = ServiceMemberSerializer(
            queryset,
            context={
                'can_remove_member': can_remove_member,
            },
            read_only=True,
            many=True,
        )
        return Response(serializer.data)
