import logging
from collections import defaultdict

from django.db.models import F
from django.conf import settings as django_settings
from plan.api.mixins import TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from rest_framework import viewsets, serializers, mixins

from plan import settings
from plan.api.serializers import ModelSerializer
from plan.api.idm import actions

from plan.idm.constants import ACTIVE_STATES, STATES, INACTIVE_TEAM_STATES, INACTIVE_STATES
from plan.idm import exceptions
from plan.roles.models import Role

from plan.services.models import Service, ServiceMember, ServiceMemberDepartment
from plan.staff.models import Staff, Department
from plan.swagger import SwaggerFrontend


log = logging.getLogger(__name__)

STATE_ACTIONS = {
    'requested': ('approve', 'decline', 'rerequest',),
    'awaiting': ('approve', 'decline',),
    'depriving_validation': ('remove',),
    'declined': ('rerequest',),
    'failed': ('rerequest',),
    'need_request': ('rerequest',),
    'deprived': ('rerequest',),
}

PERMISSION_ACTIONS = {
    'can_be_approved': ('approve', 'decline', ),
    'can_be_deprived': ('remove',),
    'can_be_rerequested': ('rerequest',),
}


class CompactRoleSerializer(ModelSerializer):
    scope = serializers.SerializerMethodField()
    service = serializers.IntegerField(source='service_id')
    name = serializers.CharField(source='i_name')

    class Meta:
        model = Role
        fields = [
            'id',
            'name',
            'code',
            'scope',
            'service'
        ]

    def get_scope(self, obj):
        if obj.scope:
            return {
                'id': obj.scope.slug,
                'name': obj.scope.i_name,
            }


class CompactStaffSerializer(ModelSerializer):
    firstName = serializers.CharField(source='i_first_name')
    lastName = serializers.CharField(source='i_last_name')
    fullName = serializers.SerializerMethodField(method_name='get_full_name')
    isDismissed = serializers.BooleanField(source='is_dismissed')

    class Meta:
        model = Staff
        fields = [
            'id',
            'login',
            'firstName',
            'lastName',
            'isDismissed',
            'fullName',
            'is_robot',
            'affiliation',
        ]

    def get_full_name(self, obj):
        return obj.get_full_name()


class RoleSerializer(serializers.Serializer):
    id = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()
    expires = serializers.SerializerMethodField()
    raw_state = serializers.SerializerMethodField()
    role_url = serializers.SerializerMethodField()
    state = serializers.SerializerMethodField()
    approvers = serializers.SerializerMethodField()

    def __init__(self,
                 visible_actions,
                 service,
                 roles,
                 staffs,
                 person_members,
                 department_members,
                 *args,
                 **kwargs,
                 ):
        super(RoleSerializer, self).__init__(*args, **kwargs)

        self.visible_actions = visible_actions
        self.service = service
        self.roles = roles
        self.staffs = staffs
        self.person_members = person_members
        self.department_members = department_members

    def get_id(self, role):
        if role['user']:
            login = role['user']['username']
            role_pk = int(role['node']['data']['role'])
            sm = self.person_members.get(login).get(role_pk) if self.person_members.get(login) else None
            return sm.id if sm else None
        else:
            staff_id = role['group']['id']
            role_pk = int(role['node']['data']['role'])
            smd = self.department_members.get(staff_id).get(role_pk) if self.department_members.get(staff_id) else None
            return smd.id if smd else None

    def get_actions(self, role):
        if role.get('permissions'):
            available_actions = []
            for permission in role['permissions']:
                available_actions += PERMISSION_ACTIONS.get(permission, [])
        else:
            possible_actions = STATE_ACTIONS.get(role['state'], [])
            available_actions = [action for action in possible_actions if action in self.visible_actions]

        prefix = 'member_' if role.get('user') else 'department_'
        return [prefix + action for action in available_actions]

    def get_expires(self, role):
        expires = role.get('expire_at')

        if role['state'] in ACTIVE_STATES and expires:
            return expires
        else:
            return None

    def get_raw_state(self, role):
        return {
            'value': role.get('state'),
            'title': str(STATES[role.get('state')])
        }

    def get_role_url(self, role):
        role_filter = 'role={id}'.format(id=role['id'])
        return settings.IDM_ROLE_FILTER_URL.format(filter=role_filter)

    def get_approvers(self, role):
        approvers = role.get('require_approve_from')
        result = []
        if approvers:
            result = CompactStaffSerializer(
                [self.staffs[approver['username']] for approver in approvers[0]], many=True
            ).data
        return result

    def get_state(self, role):
        return None

    def get_role(self, role):
        role_pk = int(role['node']['data']['role'])
        return CompactRoleSerializer(self.roles[role_pk]).data


class PersonRoleSerializer(RoleSerializer):
    fromDepartment = serializers.SerializerMethodField(method_name='get_from_department')
    serviceMemberDepartmentId = serializers.SerializerMethodField(method_name='get_from_department_id')
    person = serializers.SerializerMethodField()
    role = serializers.SerializerMethodField()

    def get_id(self, role):
        return role['id']

    def get_from_department(self, role):
        if role.get('group'):
            return {
                'id': role['group']['department']['id'],
                'name': role['group']['department']['name']
            }
        else:
            return None

    def get_from_department_id(self, role):
        if role.get('group'):
            return role['id']
        else:
            return None

    def get_person(self, role):
        return CompactStaffSerializer(self.staffs[role['user']['username']]).data

    def get_state(self, role):
        return 'waiting_approval'


class DepartmentRoleSerializer(RoleSerializer):
    membersCount = serializers.SerializerMethodField(method_name='get_members_count')
    name = serializers.SerializerMethodField()
    serviceMemberDepartmentId = serializers.SerializerMethodField(method_name='get_real_id')
    role = serializers.SerializerMethodField()
    isApproved = serializers.SerializerMethodField(method_name='get_is_approved')

    def get_id(self, role):
        return role['group']['department']['id']

    def get_name(self, role):
        return role['group']['department']['name']

    def get_members_count(self, role):
        return role['group']['department_member_count']

    def get_real_id(self, role):
        return role['id']

    def get_state(self, role):
        return 'new'

    def get_is_approved(self, role):
        return False


class TeamStatusesSerializer(ModelSerializer):
    def expand_groups(self, roles):
        # Выгружаем все департаменты и родительские взаимоотношения из closuretree
        department_staff_ids = [
            role['group']['id']
            for role in roles
            if role['group'] and role['state'] in INACTIVE_STATES + ['rerequested']
        ]
        departments = {
            d.staff_id: {
                'id': d.id,
                'name': d.i_name,
            }
            for d in (
                Department.objects
                .filter(staff_id__in=department_staff_ids)
            )
        }
        department_closure = (
            Department._closure_model.objects
            .filter(parent__in=[d['id'] for d in departments.values()])
            .exclude(parent=F('child'))
            .values_list('parent', 'child')
        )

        # Строим по каждому департаменту список вложенных
        all_departments_ids = {d['id'] for d in departments.values()}
        department_roots = defaultdict(list)
        for parent, child in department_closure:
            all_departments_ids.add(parent)
            all_departments_ids.add(child)
            department_roots[parent].append(child)

        # Строим по каждому департаменту список прямых членств
        members = Staff.objects.filter(department__in=all_departments_ids).active()
        department_members = defaultdict(list)
        for member in members:
            department_members[member.department_id].append(member)

        # По словарю прямых членств и словарю вложенных строим для каждого департамента полный список членств
        department_nested_members = defaultdict(list)
        for root_id, children in department_roots.items():
            department_nested_members[root_id] = []
            for child_id in children:
                department_nested_members[root_id] += department_members.get(child_id, [])
        for department_id, members in department_members.items():
            department_nested_members[department_id] += members

        expanded_roles = []
        for role in roles:
            expanded_roles.append(role)

            if not role.get('group') or not role['state'] in INACTIVE_TEAM_STATES:
                continue

            current_department = departments[role['group']['id']]
            current_members = department_nested_members.get(current_department['id'], [])

            role['group']['department'] = current_department
            role['group']['department_member_count'] = len(current_members)

            for member in current_members:
                expanded_roles.append({
                    'id': role['id'],
                    'node': {
                        'data': {
                            'role': int(role['node']['data']['role'])
                        }
                    },
                    'expire_at': role.get('expire_at'),
                    'user': {
                        'username': member.login
                    },
                    'group': role['group'],
                    'require_approve_from': role.get('require_approve_from'),
                    'state': role['state'],
                    'permissions': role.get('permissions')
                })

        return expanded_roles

    def get_role_objects(self, roles):
        role_pks = [int(role['node']['data']['role']) for role in roles]
        return {r.pk: r for r in Role.objects.filter(pk__in=role_pks).select_related('scope')}

    def get_staffs(self, roles):
        person_usernames = [role['user']['username'] for role in roles if role.get('user')]
        for role in roles:
            approvers = role.get('require_approve_from')
            if approvers:
                person_usernames += [approver['username'] for approver in approvers[0]]
        return {s.login: s for s in Staff.objects.filter(login__in=person_usernames)}

    def get_visible_actions(self, requester, service):
        visible_actions = set()
        if requester.user.is_superuser:
            visible_actions = {'remove'}
        if requester.is_responsible(service):
            visible_actions = {'approve', 'decline', 'remove', 'rerequest'}
        return visible_actions

    def get_person_members(self, service):
        person_members = defaultdict(dict)
        service_members = (
            ServiceMember.objects
            .filter(service=service)
            .select_related('staff', 'role')
        )
        for sm in service_members:
            person_members[sm.staff.login][sm.role.pk] = sm
        return person_members

    def get_department_members(self, service):
        department_members = defaultdict(dict)
        service_member_departments = (
            ServiceMemberDepartment.objects
            .filter(service=service)
            .select_related('department', 'role')
        )
        for smd in service_member_departments:
            department_members[smd.department.staff_id][smd.role.pk] = smd
        return department_members

    def to_representation(self, service):
        result = {
            'inactive': {
                'persons': [],
                'departments': [],
            },
            'active': {
                'persons': [],
                'departments': [],
            }
        }

        if not service.is_alive or service.readonly_state == Service.CREATING:
            return result

        requester = self.context['request'].person
        statuses = [status for status in INACTIVE_STATES + ACTIVE_STATES if status != 'granted']
        try:
            roles = actions.get_roles(
                service=service,
                requester=requester,
                state=statuses,
                full=False,
                timeout=django_settings.IDM_GET_ROLES_TIMEOUT,
                retry=django_settings.ABC_IDM_FROM_API_RETRY,
            )
        except exceptions.TimeoutError:
            log.warning(
                'Got timeout exception from idm in team_statuses view'
            )
            return result
        except exceptions.BadRequest:
            return result

        roles = self.expand_groups(roles)

        staffs = self.get_staffs(roles)
        role_objects = self.get_role_objects(roles)
        visible_actions = self.get_visible_actions(requester, service)
        person_members = self.get_person_members(service)
        department_members = self.get_department_members(service)

        role_serializer = RoleSerializer(
            visible_actions, service, role_objects, staffs, person_members, department_members
        )
        person_role_serializer = PersonRoleSerializer(
            visible_actions, service, role_objects, staffs, person_members, department_members
        )
        department_role_serializer = DepartmentRoleSerializer(
            visible_actions, service, role_objects, staffs, person_members, department_members
        )

        active_roles = [role for role in roles if role['state'] in ACTIVE_STATES]
        active_persons = [
            role_serializer.to_representation(role)
            for role in active_roles
            if role.get('user') and role_serializer.get_id(role)
        ]
        active_departments = [
            role_serializer.to_representation(role)
            for role in active_roles
            if not role.get('user') and role_serializer.get_id(role)
        ]
        inactive_roles = [role for role in roles if role['state'] in INACTIVE_TEAM_STATES]
        inactive_persons = [
            person_role_serializer.to_representation(role) for role in inactive_roles if role.get('user')
        ]
        inactive_departments = [
            department_role_serializer.to_representation(role) for role in inactive_roles if not role.get('user')
        ]

        return {
            'active': {
                'persons': active_persons,
                'departments': active_departments
            },
            'inactive': {
                'persons': inactive_persons,
                'departments': inactive_departments
            }
        }

    class Meta:
        model = Service
        # drf-yasg ругается на отсуствие fields, поэтому делаю "заплатку",
        # в to_representation все равно переопределяем результат
        fields = ['id', ]
        read_only_fields = fields


class TeamStatusesView(TvmAccessMixin, viewsets.GenericViewSet, mixins.RetrieveModelMixin):
    default_swagger_schema = SwaggerFrontend

    _permissions_to_proceed = 'view_own_services'
    permission_classes = [TvmAuthenticated]
    serializer_class = TeamStatusesSerializer
    queryset = Service.objects.all()
