import waffle

from django.utils.translation import ugettext_lazy as _
from django.conf import settings
from django.contrib.auth import get_user_model
from django.db.models import Q

from rest_framework.serializers import ValidationError

from plan.api.exceptions import (
    PermissionDenied,
    ServiceTagsDuplicatingError,
    GradientStructureError,
    ValidationError as PlanValidationError,
)
from plan.resources.models import ServiceResource
from plan.services import exceptions, permissions
from plan.services.constants.permissions import SERVICE_TAG_APP, SERVICE_TAG_MODEL, CHANGE_SERVICE_TAG_CODENAME
from plan.services.models import Service, SERVICE_SLUG_PATTERN, Department
from plan.services.api.slug import MAX_SLUG_LEN, make_unique_slug
from plan.oebs.constants import (
    OEBS_FLAGS,
    OEBS_PROCUREMENT_FLAG,
    OEBS_HARDWARE_FLAG,
    OEBS_HR_FLAG,
    OEBS_REVENUE_FLAG,
    OEBS_GROUP_ONLY_FLAG,
)
from plan.oebs.models import OEBSAgreement
from plan.oebs.utils import is_oebs_related


def _can_manage_oebs(requester):
    if (
        settings.OEBS_MDM_RESTRICTED
        and Department.objects.get(pk=requester.department_id).url != settings.OEBS_MDM_DEPARTMENT
        and not requester.user.is_superuser
    ):
        return False

    return True


class ServiceValidator(object):

    def set_context(self, serializer):
        if getattr(serializer, 'parent', None) is not None:
            serializer = serializer.parent

        self.requester = getattr(serializer.context['request'], 'person', None)

        if isinstance(getattr(serializer, 'instance', None), Service):
            self.service = serializer.instance

        elif hasattr(serializer.instance, 'service'):
            self.service = serializer.instance.service


class ServiceSlugValidator(object):
    def __call__(self, value, *args, **kwargs):
        if not value or not SERVICE_SLUG_PATTERN.match(value):
            raise ValidationError(
                _("Код сервиса может состоять только из латинских букв, цифр, знаков подчеркивания и дефисов.")
            )

        if len(value) > MAX_SLUG_LEN:
            ValidationError(
                _(f"Длина слага должна быть не более {MAX_SLUG_LEN} символов.")
            )

        unique_slug = make_unique_slug(value)
        if unique_slug != value:
            raise ValidationError(_('Слаг не уникален'))

        if value.isdigit():
            raise ValidationError(_('Код не может состоять из одних цифр.'))


class ServiceIsReadonly(ServiceValidator):

    def __call__(self, attrs, *args, **kwargs):
        if self.service.readonly_state:
            raise exceptions.ServiceReadonlyError(self.service)


class CanDelete(ServiceValidator):
    def __call__(self, value, *args, **kwargs):
        person = self.requester
        can_delete = permissions.is_service_responsible(self.service, person)
        if value == Service.states.DELETED and not can_delete:
            raise exceptions.CannotDeleteService()


class CannotRestore(ServiceValidator):
    def __call__(self, value, *args, **kwargs):
        if self.service.is_deleted and value != Service.states.DELETED:
            raise exceptions.CannotRestoreService()


class AncestorsMustBeActive(ServiceValidator):
    def __call__(self, value, *args, **kwargs):
        if self.service.parent is None:
            return

        inactive_ancestors = self.service.get_ancestors().exclude(state__in=Service.states.ACTIVE_STATES)
        if inactive_ancestors.exists() and value in Service.states.ACTIVE_STATES:
            raise exceptions.AncestorsMustBeActive()


class CannotCloseWithImportantResources(ServiceValidator):
    def __call__(self, value, *args, **kwargs):
        if value in Service.states.ACTIVE_STATES:
            return

        if self.service.has_important_resources():
            raise exceptions.HasImportantResources()


def service_is_active(service):
    if not service.is_active:
        raise ValidationError(
            _('Мы не можем выполнить это действие над сервисом в неактивном статусе')
        )


class FryParadoxValidator(object):
    def __call__(self, attrs, *args, **kwargs):
        service = attrs['service']
        destination = attrs['destination']
        if destination in service.get_descendants(include_self=True):
            raise exceptions.FryParadox()


class SameParentValidator(object):
    def __call__(self, attrs, *args, **kwargs):
        service = attrs['service']
        destination = attrs['destination']
        if destination == service.parent:
            raise exceptions.SameParent()


class CheckAllowedParentType:
    def __call__(self, attrs, *args, **kwargs):
        if not waffle.switch_is_active(settings.SWITCH_CHECK_ALLOWED_PARENT_TYPE):
            return
        service = attrs.get('service')
        if service:
            service_type = service.service_type
        else:
            service_type = attrs.get('service_type')

        if not service_type:
            raise PlanValidationError(
                message={
                    'ru': 'Необходимо указать тип сервиса',
                    'en': 'Service type is required'
                }
            )
        destination = attrs.get('destination') or attrs.get('parent')

        if destination and not service_type.available_parents.filter(
            pk=destination.service_type_id
        ).exists():
            raise exceptions.UnallowedParentType()


class RestrictMoveToJunk(object):
    def __call__(self, attrs, *args, **kwargs):
        destination = attrs['destination']
        if destination.get_ancestors(include_self=True).filter(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG).exists():
            raise exceptions.RestrictMoveToJunk()


def gradient_tags_validate(instance, requester, new_gradient_tag, old_gradient_tags, parent=None):
    if len(new_gradient_tag) > 1:
        raise ServiceTagsDuplicatingError(
            message={
                'ru': 'Нельзя добавить несколько градиентных тегов одновременно.',
                'en': 'Gradient tags cannot be used together.',
            }
        )

    gradient_tag = new_gradient_tag[0] if len(new_gradient_tag) > 0 else None

    for tag in [settings.GRADIENT_VS, settings.BUSINESS_UNIT_TAG]:
        if tag in old_gradient_tags and gradient_tag != tag and not requester.user.is_superuser:
            raise PermissionDenied(
                message={
                    'ru': f"У вас недостаточно прав для удаления тега {tag}.",
                    'en': f"You don't have sufficient permissions to delete tag '{tag}'."
                }
            )

    if len(new_gradient_tag) == 0:
        return

    for tag in [settings.GRADIENT_VS, settings.BUSINESS_UNIT_TAG]:
        if tag not in old_gradient_tags and gradient_tag == tag and not requester.user.is_superuser:
            raise PermissionDenied(
                message={
                    'ru': f"У вас недостаточно прав для добавления тега {tag}.",
                    'en': f"You don't have sufficient permissions to add tag '{tag}'."
                }
            )

    ancestors = umbrella = contour = service = None
    # если указан parent, то он в приоритете
    include_self = True if parent else False
    if parent:
        service = parent
    elif instance:
        parent = instance.parent
        service = instance

    if service:
        ancestors = service.get_ancestors(include_self=include_self).gradient().prefetch_related('tags').order_by('level')
        umbrella = service.search_gradient_parent(settings.GRADIENT_UMB, ancestors)
        contour = service.search_gradient_parent(settings.GRADIENT_CONTOUR, ancestors)

    if gradient_tag == settings.GRADIENT_UMB:
        # проверяем, что прямой родитель == VS и выше по дереву нет зонтиков и контуров
        parent_is_vs = parent is not None and parent.gradient_type() == settings.GRADIENT_VS
        if not parent_is_vs:
            raise GradientStructureError(
                message={
                    'ru': "Прямой предок сервиса не является valuestream.",
                    'en': "The service parent is not valuestream."
                }
            )

        if umbrella or contour:
            raise GradientStructureError(
                message={
                    'ru': "Выше по дереву уже есть контур или зонтик.",
                    'en': "One or more ancestors have the tags 'outline' or 'umb'."
                }
            )

    if gradient_tag == settings.GRADIENT_CONTOUR:
        # проверяем, что выше по дереву есть зонтик и vs, но нет контуров
        valuestream = None
        if service:
            valuestream = service.search_gradient_parent(settings.GRADIENT_VS, ancestors)

        if contour or not umbrella or not valuestream:
            raise GradientStructureError(
                message={
                    'ru': "Выше по дереву нет зонтика или уже есть контур.",
                    'en': "One or more ancestors have the tag 'outline', or no ancestor has the tag 'umb'."
                }
            )


def validate_oebs_tags(instance, new_tags):
    if instance and not _has_allowed_oebs_subtree_in_parents(instance):
        # не проверяем OEBS теги вне синхронизированного дерева
        return

    oebs_tags = list(OEBS_FLAGS.values())
    oebs_tags.append(settings.MONEY_MAP_TAG)
    new_oebs_tags = {tag.slug for tag in new_tags if tag.slug in oebs_tags}
    current_oebs_tags = set()
    if instance:
        current_oebs_tags = (
            set(instance.tags.filter(slug__in=oebs_tags).values_list('slug', flat=True)) if instance else set()
        )

    if new_oebs_tags != current_oebs_tags:
        raise PlanValidationError(
            message={
                'ru': 'Редактирование тегов связанных с OEBS запрещено',
                'en': 'Editing tags related to OEBS is prohibited'
            }
        )


def verify_tags_permissions(requester, old_tags, new_tags):
    changed_tags = set(new_tags) ^ set(old_tags)
    for tag in changed_tags:
        perm_codename = CHANGE_SERVICE_TAG_CODENAME % tag.slug
        tag_permission_owners = get_user_model().user_permissions.through.objects.filter(
            permission__content_type__app_label=SERVICE_TAG_APP,
            permission__content_type__model=SERVICE_TAG_MODEL,
            permission__codename=perm_codename,
        ).values_list('user_id', flat=True)
        if tag_permission_owners and requester.user_id not in tag_permission_owners:
            raise PermissionDenied(
                message={
                    'ru': f"У вас недостаточно прав для изменения тэга {tag.name}",
                    'en': f"You don't have permission to change tag {tag.name_en}",
                }
            )


class ServiceOwnerValidator:
    def __call__(self, value):
        if value.is_robot:
            raise ValidationError(
                _('Владелец сервиса не может быть роботом.')
            )


class CheckOEBSRestrictionsForMovingService(object):

    def set_context(self, serializer):
        if getattr(serializer, 'parent', None) is not None:
            serializer = serializer.parent
        self.requester = serializer.context['request'].person

    def __call__(self, attrs, *args, **kwargs):
        if is_oebs_related(attrs['service']):
            if not _can_manage_oebs(requester=self.requester):
                raise PermissionDenied(
                    message={
                        'ru': 'Перемещение OEBS синхронизированных сервисов разрешено только группе MDM - mdm-support@yandex-team.ru',
                        'en': 'Only MDM group is allowed to move OEBS synchronized services - mdm-support@yandex-team.ru'
                    }
                )

            require_no_active_oebs_agreement(attrs['service'])
            require_oebs_related_destination(attrs['service'], attrs['destination'])
            restrict_moving_outside_allowed_oebs_subtree(attrs['destination'])
        else:
            require_no_active_oebs_agreement(attrs['service'], exclude_ancestors=True)


class CheckBillingPointRestrictionsForMovingService:
    def __call__(self, attrs, *args, **kwargs):
        service = attrs['service']
        destination = attrs['destination']
        has_billing_point_tag_in_tree = service.get_ancestors(include_self=False).filter(
            tags__slug=settings.OEBS_BILLING_AGGREGATION_TAG
        ).exists()
        if has_billing_point_tag_in_tree:
            has_billing_point_tag = service.tags.filter(
                slug=settings.OEBS_BILLING_AGGREGATION_TAG
            ).exists()
            if not has_billing_point_tag:
                # при переезде должна сохраниться точка биллинга
                has_billing_point_tag_in_destination_tree = destination.get_ancestors(include_self=True).filter(
                    tags__slug=settings.OEBS_BILLING_AGGREGATION_TAG
                ).exists()
                if not has_billing_point_tag_in_destination_tree:
                    raise PermissionDenied(
                        message={
                        'ru': 'Перемещение сервиса противоречит логике агрегации точек биллинга',
                        'en': 'Service moving contradicts the logic of aggregation of billing points',
                        }
                    )


class ChangeOEBSFlags(ServiceValidator):
    def __call__(self, attrs, *args, **kwargs):
        if self._oebs_flags_changed(attrs):
            if not _can_manage_oebs(requester=self.requester):
                raise PermissionDenied(
                    message={
                        'ru': 'Управление OEBS флагами разрешено только группе MDM - mdm-support@yandex-team.ru',
                        'en': 'Only MDM group is allowed to manage OEBS flags - mdm-support@yandex-team.ru'
                    }
                )
            if not permissions.is_service_responsible(self.service, self.requester):
                raise PermissionDenied(
                    message={
                        'ru': 'Вы не можете управлять OEBS флагами в данном сервисе',
                        'en': 'You are not allowed to manage OEBS flags in this service'
                    }
                )
            require_service_in_allowed_oebs_subtree(self.service)
            require_all_oebs_flags_for_gradient_service(self.service, attrs)
            require_no_active_oebs_agreement(self.service)

            hardware_flag = attrs.get(OEBS_HARDWARE_FLAG, self.service.use_for_hardware)
            hr_flag = attrs.get(OEBS_HR_FLAG, self.service.use_for_hr)
            procurement_flag = attrs.get(OEBS_PROCUREMENT_FLAG, self.service.use_for_procurement)
            revenue_flag = attrs.get(OEBS_REVENUE_FLAG, self.service.use_for_revenue)

            group_flag = attrs.get(OEBS_GROUP_ONLY_FLAG)
            if group_flag is not None and group_flag != self.service.use_for_group_only:
                if not any((hardware_flag, hr_flag, procurement_flag, revenue_flag)):
                    # нельзя ставить только этот флаг
                    raise exceptions.ConflictingGroupOnlyFlagValue()
                if group_flag and any((
                    self.service.use_for_procurement,
                    self.service.use_for_hr,
                    self.service.use_for_revenue,
                    self.service.use_for_hardware,
                )):
                    # этот флаг можно выставить только при первоначальной синхронизации
                    raise exceptions.CannotChangeGroupOnlyFlagValue()

            if revenue_flag is False:
                if hardware_flag or hr_flag or procurement_flag:
                    # нельзя снимать OEBS_REVENUE_FLAG, если проставлены эти флаги
                    # так же нельзя ставить эти флаги без OEBS_REVENUE_FLAG
                    raise exceptions.ConflictingOebsFlagsValue()

    def _oebs_flags_changed(self, attrs):
        return any(flag in attrs and attrs.get(flag) != getattr(self.service, flag) for flag in OEBS_FLAGS)


class CannotChangeStateWithOEBSAgreement(ServiceValidator):
    def __call__(self, value, *args, **kwargs):
        if (
            value in Service.states.ACTIVE_STATES and
            self.service.state in Service.states.ACTIVE_STATES
        ):
            # если меняют с одного активного статуса на другой -
            # согласование не требуется
            return

        if is_oebs_related(self.service):
            require_no_active_oebs_agreement(self.service)
            if not _can_manage_oebs(self.requester):
                raise PermissionDenied(
                    message={
                        'ru': 'Управление статусами OEBS синхронизированных сервисов разрешено только группе MDM - mdm-support@yandex-team.ru',
                        'en': 'Only MDM group is allowed to manage statuses of OEBS synchronized services - mdm-support@yandex-team.ru'
                    }
                )

            if value not in Service.states.ACTIVE_STATES and self.service.has_oebs_gradient_tags:
                raise exceptions.ClosingOebsGradientService()

        else:
            require_no_active_oebs_agreement(self.service, exclude_ancestors=True)


def require_no_active_oebs_agreement(service, exclude_ancestors=False):
    active_agreements = OEBSAgreement.objects.active()
    descendants = service.get_descendants(include_self=True)
    if exclude_ancestors:
        active_agreements = active_agreements.filter(service__in=descendants)
    else:
        ancestors = service.get_ancestors()
        active_agreements = active_agreements.filter(Q(service__in=ancestors) | Q(service__in=descendants))
    if active_agreements.exists():
        raise exceptions.HasActiveOebsAgreement()


def require_oebs_related_destination(service, destination):
    ancestors_ids = destination.get_ancestors(include_self=True).values_list('id', flat=True)
    if not (
        service.has_oebs_gradient_tags
        or destination.slug == settings.OEBS_EXPERIMENTS_ROOT
        or ServiceResource.objects.filter(
            service_id__in=ancestors_ids,
            type__code=settings.OEBS_PRODUCT_RESOURCE_TYPE_CODE,
        ).exists()
    ):
        raise exceptions.InvalidOebsDestination()


def require_all_oebs_flags_for_gradient_service(service, attrs):
    flags = [
        OEBS_HR_FLAG,
        OEBS_PROCUREMENT_FLAG,
        OEBS_REVENUE_FLAG,
    ]
    if not all(attrs.get(flag) for flag in flags) and service.has_oebs_gradient_tags:
        raise exceptions.ConflictingOebsFlagsValue()


def _has_allowed_oebs_subtree_in_parents(service):
    if not settings.RESTRICT_OEBS_SUBTREE:
        return True
    return (
        service
        .get_ancestors(include_self=True)
        .filter(slug__in=settings.ALLOWED_OEBS_SUBTREE)
        .exists()
    )


def restrict_moving_outside_allowed_oebs_subtree(destination):
    if not _has_allowed_oebs_subtree_in_parents(destination):
        raise exceptions.InvalidOebsDestination(
            title={
                'ru': 'Нельзя переносить связанные с OEBS сервисы из связанного с OEBS поддерева',
                'en': 'Moving OEBS-related services from allowed OEBS subtree is restricted',
            }
        )


def require_service_in_allowed_oebs_subtree(service):
    if not _has_allowed_oebs_subtree_in_parents(service):
        raise exceptions.ConflictingOebsFlagsValue(
            title={
                'ru': 'Нельзя выставлять OEBS флаги сервисам вне связанного с OEBS поддерева',
                'en': 'Setting OEBS flags for services outside OEBS allowed subtree is restricted',
            }
        )
