from plan.roles.models import Role

from plan.services import permissions
from plan.services.models import Service, ServiceMember

'''
Этоn модуль нужен для совместимости с легаси кодом yawf и dehydrate в старом апи.

За современными проверками прав иди в plan.services.permissions.
'''


class ServicePermissionChecker(object):
    SERVICE_TMP_MODERATOR_ATTRNAME = '_temporary_moderator'

    _bound = False
    _service = None
    _sender = None
    _owner_of_service_levels = None

    def __init__(self, service=None, sender=None):
        if service:
            self.bind(service, sender)

    def set_temporary_moderator(self, obj, person):
        assert isinstance(obj, Service)
        setattr(self, self.SERVICE_TMP_MODERATOR_ATTRNAME, person)

    def bind(self, service, sender):
        self._bound = True
        self._service = service
        self._sender = sender
        self._owner_of_service_levels = set()
        self._exclusive_owner_of_service = False
        self._member_of_services = set()
        self._approved_departments = set()

        query = ServiceMember.objects.filter(service__in=service.get_ancestors())

        service_departments = {}
        for member in query.select_related('role', 'service'):

            if member.staff_id == sender.id:
                self._member_of_services.add(member.service_id)

                if member.role.is_owner:
                    self._owner_of_service_levels.add(member.service.level)

                if member.role.code == Role.EXCLUSIVE_OWNER:
                    self._exclusive_owner_of_service = True

            if member.service == self._service and member.from_department_id:
                dpt_id = member.from_department_id
                dpt_members = service_departments.setdefault(dpt_id, [])
                dpt_members.append(member)

        for dpt_id, members in service_departments.items():
            self._approved_departments.add(dpt_id)

    # -----
    # compatibility shims

    def can_edit_member(self, obj, sender):
        return False

    def can_remove_department(self, obj, sender):
        assert isinstance(obj, Service)
        return permissions.is_service_responsible(service=obj, person=sender)

    def can_remove_member(self, obj, sender):
        assert isinstance(obj, ServiceMember)
        return permissions.can_remove_member(service_member=obj, person=sender)

    def is_service_responsible(self, obj, sender):
        assert isinstance(obj, Service)
        return permissions.is_service_responsible(service=obj, person=sender)

    def is_service_support(self, obj, sender):
        assert isinstance(obj, Service)
        return permissions.is_service_support(person=sender)

    def can_approve_move_request(self, obj, sender):
        assert isinstance(obj, Service)
        return permissions.can_approve_move_request(service=obj, person=sender)

    def can_cancel_move_request(self, obj, sender):
        assert isinstance(obj, Service)
        return permissions.can_cancel_move_request(service=obj, person=sender)

    def is_owner_of_metaservice(self, obj, sender):
        assert isinstance(obj, Service)
        return permissions.is_owner_of_metaservice(service=obj, person=sender)

    @classmethod
    def is_hr(cls, obj, sender):
        return permissions.is_hr(person=sender)

    # -----

    @staticmethod
    def is_active(obj, sender):
        assert isinstance(obj, (Service, ServiceMember))
        if isinstance(obj, ServiceMember):
            obj = obj.service
        return obj.state in Service.states.ACTIVE_STATES

    @staticmethod
    def is_being_created(obj, sender):
        assert isinstance(obj, ServiceMember)
        return False

    @staticmethod
    def is_department_member(obj, sender):
        assert isinstance(obj, ServiceMember)
        return obj.from_department_id is not None


service_checker = ServicePermissionChecker()
