"""
Здесь лежат функции для вычисления и проверки дегидрируемых полей.
"""


from django.conf import settings


def count_unique_members(service):
    from plan.services.models import ServiceMember

    service_members = ServiceMember.objects.filter(
        service__in=service.get_alive_descendants(include_self=True)
    )
    return service_members.values('staff').distinct().count()


def count_unique_immediate_members(service):
    return service.members.team().filter(staff__is_robot=False).values('staff').distinct().count()


def count_unique_immediate_robots(service):
    return service.members.team().filter(staff__is_robot=True).values('staff').distinct().count()


def count_unique_immediate_external_members(service):
    return service.members.filter(
        staff__affiliation='external', staff__is_robot=False
    ).values('staff').distinct().count()


def count_service_member_weight(service_member):
    from plan.services.models import ServiceMember
    # ну два, чтобы всегда можно было между ними сеньора вставить
    member_weight = service_member.role.weight * 2
    if ServiceMember.objects.heads().filter(
            service=service_member.service,
            staff=service_member.staff,
    ).exists():
        member_weight += 1
    return member_weight


def get_service_path(service):
    path = '/{0}/'.format(service.slug)
    parent = service.parent
    while parent:
        path = '/{0}{1}'.format(parent.slug, path)
        parent = parent.parent
    return path


def get_service_ancestors(service):
    from plan.api.fields import MappingField
    from plan.services.api.services import ServiceTagSerializer
    from plan.services.models import Service
    from plan.api.serializers import BasePlanModelSerializer

    class AncestorSerializer(BasePlanModelSerializer):
        name = MappingField({'ru': 'name', 'en': 'name_en'})
        tags = ServiceTagSerializer(many=True)

        class Meta:
            model = Service
            fields = (
                'id',
                'slug',
                'name',
                'parent',
                'tags'
            )

    parents = service.get_ancestors().prefetch_related('tags')
    serializer = AncestorSerializer(parents, many=True)
    return serializer.data


def get_has_external_members(service):
    return service.members.filter(staff__affiliation='external', staff__is_robot=False).exists()


def get_vacancies_count(service):
    from plan.resources.models import ServiceResource

    return ServiceResource.objects.filter(
        service=service, state=ServiceResource.GRANTED, type__code=settings.FEMIDA_RESOURCE_TYPE_CODE
    ).count()


def has_forced_suspicious_reason(service):
    from plan.services.models import ServiceSuspiciousReason

    return ServiceSuspiciousReason.objects.filter(
        marked_by=None,
        reason=ServiceSuspiciousReason.FORCED,
        service=service,
    ).exists()
