from requests import codes as http_codes

from plan.api import exceptions


class ServiceReadonlyError(exceptions.ABCAPIException):

    def __init__(self, service):
        self.status_code = http_codes.conflict
        self.code = 'service_readonly'
        self.detail = 'service is read-only'
        self.title = {
            'ru': 'Извините, мы сейчас не можем редактировать этот сервис',
            'en': "Sorry, we cannot edit this service right now"
        }

        readonly_state_ru = dict(service.READONLY_STATES).get(service.readonly_state, '').lower()
        self.message = {
            'ru': 'Этот сервис сейчас доступен только для чтения, потому что он {}. '
                  'Попробуйте еще раз, когда эта операция завершится.'.format(readonly_state_ru),
            'en': 'This service is currently read-only because it is {}. '
                  'Please try again, once this process completes.'.format(service.readonly_state)
        }


class ActionAlreadyInProgress(ServiceReadonlyError):
    def __init__(self, service):
        super(ActionAlreadyInProgress, self).__init__(service)
        readonly_state_ru = dict(service.READONLY_STATES).get(service.readonly_state, '').lower()
        self.message = {
            'ru': 'Мы не можем повторить это действие, так как сервис и так уже {}'.format(readonly_state_ru),
            'en': 'We cannot repeat this action, as the service is already {}'.format(service.readonly_state)
        }


class CannotDeleteService(exceptions.PermissionDenied):
    default_message = {
        'ru': 'Удалить сервис могут только его управляющие.',
        'en': 'Only people responsible for this service may delete it.',
    }


class CannotCloseService(exceptions.PermissionDenied):
    default_message = {
        'ru': 'Закрыть сервис могут только его управляющие.',
        'en': 'Only people responsible for this service may close it.',
    }


class CannotCloseOrDeleteBaseService(exceptions.PermissionDenied):
    default_message = {
        'ru': 'Только суперпользователи ABC могут закрыть или удалить этот сервис',
        'en': 'Only ABC superusers can close or delete this service',
    }


class ImpossibleAction(exceptions.BadRequest):
    default_title = {
        'ru': 'Действие невозможно.',
        'en': 'Action impossible.',
    }


class CannotRestoreService(ImpossibleAction):
    default_message = {
        'ru': 'Восстановление сервиса из удаленных невозможно.',
        'en': 'Restoring services from deleted state is impossible.',
    }


class AncestorsMustBeActive(ImpossibleAction):
    default_message = {
        'ru': 'Нельзя изменить статус сервиса на активный, пока один из его родителей в неактивном статусе.',
        'en': "Can't change service to an active state while on of its parents is inactive.",
    }


class HasImportantResources(ImpossibleAction):
    default_message = {
        'ru': 'Этому сервису или его детям выданы важные ресурсы, для удаления/закрытия их нужно отозвать.',
        'en': 'This service or its children has important resources assigned, they must be relinquished first.',
    }


class RestrictMoveToJunk(ImpossibleAction):
    default_message = {
        'ru': 'Невозможно переместить сервис в Песочницу',
        'en': 'Move to Junk restricted',
    }


class FryParadox(ImpossibleAction):
    default_code = 'fry_paradox'
    default_title = {
        'ru': 'Парадокс Фрая',
        'en': 'Fry Paradox'
    }
    default_message = {
        'ru': "Мы не можем сделать сервис потомком самого себя.",
        'en': "We can't make a service its own ancestor."
    }


class SameParent(ImpossibleAction):
    default_message = {
        'ru': 'Сервис уже находится в месте назначения',
        'en': 'Service is already at this destination',
    }


class ServicesDontHaveActiveResponsibles(Exception):
    pass


class HasActiveOebsAgreement(exceptions.BadRequest):
    default_title = {
        'ru': 'Действие невозможно, так как уже есть активное согласование.',
        'en': 'Action is impossible because active agreement already exists.',
    }


class InvalidOebsDestination(exceptions.BadRequest):
    default_title = {
        'ru': 'Невозможно переместить OEBS сервис в несвязанное с OEBS поддерево',
        'en': 'Can\'t move OEBS service to OEBS unrelated subtree',
    }


class ConflictingOebsFlagsValue(exceptions.BadRequest):
    default_message = {
        'ru': 'Значения флагов "Используется в HR", "Используется в Закупках", "Сервис используется в OEBS" конфликтуют.',
        'en': 'Values of the flags "Used in HR", "Used in Procurement", "Used in Revenue" are conflicting',
    }


class ClosingOebsGradientService(exceptions.BadRequest):
    default_title = {
        'ru': 'Действие невозможно, так как сервис связан с OEBS и имеет VS/BU тег',
        'en': 'Action is impossible because service is OEBS related and has VS/BU tag',
    }


class ConflictingGroupOnlyFlagValue(exceptions.BadRequest):
    default_message = {
        'ru': 'Нельзя выставлять флаг "Используется только для группировки" без какого-либо другого OEBS флага',
        'en': 'You cannot set the "Used only for grouping" flag without any other OEBS flag.',
    }


class CannotChangeGroupOnlyFlagValue(exceptions.BadRequest):
    default_message = {
        'ru': 'Нельзя установить флаг "Используется только для группировки" у синхронизированного с OEBS сервиса.',
        'en': 'You cannot set the "Used only for grouping" flag for a service synchronized with OEBS.',
    }


class UnallowedParentType(exceptions.BadRequest):
    default_message = {
        'ru': 'Тип выбранного родителя не является допустимым родительским для текущего типа сервиса',
        'en': 'Type of the selected parent is not a valid parent for the current type of service',
    }
