from django import forms
from django.utils.translation import ugettext_lazy as _

from rest_framework import serializers

from plan.services.api.slug import MAX_SLUG_LEN, make_unique_slug
from plan.common.utils.rest_fw import PlanSerializer, LocaleServiceNameSerializer
from plan.services.helpers import is_service_name_unique

from plan.services.models import (
    Service,
    MAX_SERVICE_LEVEL,
    SERVICE_SLUG_PATTERN
)


class ServiceCreateForm(forms.ModelForm):
    move_to = forms.ModelChoiceField(
        queryset=Service.objects.alive(), required=False
    )

    def clean_move_to(self):
        dest = self.cleaned_data.get('move_to')

        if dest:
            if dest.level > MAX_SERVICE_LEVEL - 2:
                raise forms.ValidationError(_(
                    'Невозможно создать новый сервис внутри выбранного - '
                    'слишком большой уровень вложенности.'))

            if dest.state not in Service.states.ACTIVE_STATES:
                raise forms.ValidationError(_(
                    'Невозможно создать новый сервис внутри выбранного - '
                    'родительский сервис должен быть в активном статусе.'))

        return dest

    def clean_name(self):
        name = self.cleaned_data['name']

        if not is_service_name_unique(name=name):
            raise forms.ValidationError(_('Сервис с таким названием уже существует.'))

        return name

    def clean_name_en(self):
        name_en = self.cleaned_data['name_en']

        if not is_service_name_unique(name_en=name_en):
            raise forms.ValidationError(_('Сервис с таким английским названием уже существует.'))

        return name_en

    def clean_slug(self):
        slug = self.cleaned_data['slug'].lower()

        if not slug or not SERVICE_SLUG_PATTERN.match(slug):
            raise forms.ValidationError(
                _("Код сервиса может состоять только из латинских букв, цифр, знаков подчеркивания и дефисов.")
            )

        if len(slug) > MAX_SLUG_LEN:
            forms.ValidationError(
                _(f"Длина слага должна быть не более {MAX_SLUG_LEN} символов.")
            )

        unique_slug = make_unique_slug(slug)
        if unique_slug != slug:
            raise forms.ValidationError(_('Сервис с таким слагом уже существует.'))

        if slug.isdigit():
            raise forms.ValidationError(_('Слаг не может состоять из одних цифр.'))

        return slug

    class Meta:
        model = Service
        fields = (
            'name',
            'name_en',
            'slug',
            'owner',
            'description',
        )


class ServiceCreateSerializer(PlanSerializer):
    name = LocaleServiceNameSerializer()
    slug = serializers.CharField(required=False)
    owner = serializers.IntegerField(required=False)
    description = serializers.CharField(required=False)
    move_to = serializers.CharField(required=False)


class ImportantServicesForm(forms.Form):
    year_from = forms.IntegerField(min_value=2010, max_value=3000)
    month_from = forms.IntegerField(min_value=1, max_value=12)
    year_to = forms.IntegerField(min_value=2010, max_value=3000)
    month_to = forms.IntegerField(min_value=1, max_value=12)
