import collections
import logging
import datetime

from django.conf import settings
from django.db import models

from plan.idm import nodes
from plan.roles.models import Role
from plan.services import models as services_models

RoleCount = collections.namedtuple(
    'RoleCount', 'role count'
)

log = logging.getLogger(__name__)


def guess_default_service_role(person_ids, service_id=None,
                               individual_results=False):
    queryset = (
        services_models.ServiceMember.objects
        .select_related('role')
        .filter(staff__id__in=person_ids, role__service=None)
    )

    # key -> {role -> count}
    groups = collections.defaultdict(lambda: collections.defaultdict(int))

    for member in queryset:
        key = member.staff_id if individual_results else None
        groups[key][member.role] += 1

    # key -> [RoleCount, ...]
    groups = {key: [
        RoleCount(role, count) for role, count in values.items()
    ] for key, values in groups.items()}

    # key -> role
    # by role entry count desc, role name asc
    top_roles = {}
    exclusive_owners_occured = False
    for key, rolecounts in groups.items():
        rolecounts.sort(key=lambda role_count: (-role_count.count, role_count.role.name))
        top_role = rolecounts[0].role
        top_roles[key] = top_role

        exclusive_owners_occured |= top_role.is_exclusive_owner

    exclusive_owner_replacement_role = None
    if service_id and exclusive_owners_occured:
        existing_owners = services_models.ServiceMember.objects.filter(
            service__id=service_id,
            role__code=Role.EXCLUSIVE_OWNER,
        )

        if existing_owners.count():
            exclusive_owner_replacement_role = (
                Role.get_deputy_owner()
            )

    if exclusive_owner_replacement_role:
        for key, role in top_roles.items():
            if role.is_exclusive_owner:
                top_roles[key] = exclusive_owner_replacement_role

    default_role = Role.objects.get(
        code=settings.ABC_DEFAULT_SERVICE_ROLE
    )
    if individual_results:
        return {person_id: top_roles.get(person_id, default_role)
                for person_id in person_ids}
    else:
        return top_roles.get(None, default_role)


def is_service_name_unique(name=None, name_en=None):
    if not name and not name_en:
        raise ValueError()
    base_qs = services_models.Service.objects.filter(
        state__in=services_models.Service.states.ALIVE_STATES
    )
    condition = models.Q()
    if name:
        condition = condition | models.Q(name=name)
    if name_en:
        condition = condition | models.Q(name_en=name_en)
    return not base_qs.filter(condition).exists()


def generate_pre_move_service_nodes(move_request):
    # строим узлы перенесенных сервисов, как они выглядели до переноса

    move_source_node = nodes.get_service_node(move_request.source)

    for service in move_request.service.get_descendants(include_self=True):

        if service == move_request.service:
            old_node = nodes.build_service_node(service, move_source_node)

        else:
            family_member = service

            # строим для каждого ребенка сервиса узлы до его переноса
            family = list(service.get_ancestors(include_self=True))
            moved_service_index = family.index(move_request.service)
            family = family[moved_service_index:]

            subfamily_parent_node = move_source_node
            for family_member in family:
                old_node = nodes.build_service_node(family_member, subfamily_parent_node)
                subfamily_parent_node = old_node

        yield service, old_node


def get_expires_at(data):
    expires_at = None
    deprive_at = data.get('deprive_at')
    deprive_after = data.get('deprive_after')
    if deprive_at and deprive_after:
        return

    if deprive_at:
        expires_at = deprive_at
    elif deprive_after:
        expires_at = datetime.datetime.today() + datetime.timedelta(days=deprive_after)

    return expires_at
