import csv
import sys
from itertools import chain

from django.core.management.base import BaseCommand
from django.utils import translation

from plan.roles.models import Role
from plan.services.models import Service


def has_own_team(service):
    product_head = Role.get_exclusive_owner()
    return service.members.exclude(role=product_head).count()


APPROVAL_STATES = {
    False: 'не подтверждён',
    True: 'подтверждён',
}


class Command(BaseCommand):
    def handle_noargs(self, *args, **options):
        translation.activate('ru')

        writer = csv.writer(sys.stdout, delimiter=',', quotechar='"')

        query = Service.objects.exclude(state=Service.states.DELETED)
        for service in query.order_by('name'):
            if not has_own_team(service):
                continue

            # имя
            # ссылка
            service_cols = [
                service.name,
                'https://abc.yandex-team.ru/services/{}/'.format(service.slug),
            ]

            # метасервис
            # родительские сервисы
            if service.level > 0:
                ancestors = list(service.get_ancestors())
                service_cols.extend([
                    ancestors.pop(0).name,
                    ', '.join([s.name for s in ancestors]) or '—'
                ])
            else:
                service_cols.extend(['—', '—'])

            # руководитель
            if service.owner:
                service_cols.append(service.owner.login)
            else:
                service_cols.append('—')

            # статус
            service_cols.append(str(Service.states[service.state].name))

            # дата изменения
            members = list(service.members.team())
            member_timestamps = [m.modified_at for m in members]
            service_cols.append(
                max(service.modified_at, *member_timestamps).isoformat())

            for member in members:
                departments = member.staff.department.get_ancestors(include_self=True)

                member_cols = [
                    member.staff.get_full_name(),
                    member.staff.login,
                    member.role.name,
                    APPROVAL_STATES[True],
                    ' / '.join([dpt.name for dpt in departments]),
                ]

                row = [s.encode('utf-8')
                       for s in chain(service_cols, member_cols)]

                writer.writerow(row)
