import logging

import re
import waffle
from typing import Iterable, Union

from django.conf import settings
from django.contrib.postgres.fields import JSONField
from django.core.cache import cache
from django.core.exceptions import ValidationError
from django.db import models, transaction
from django.db.models import Q
from django.utils import timezone
from django.utils.encoding import force_text

from django_fsm import FSMField, transition, GET_STATE

from django.utils.translation import ugettext_lazy as _

from closuretree.models import ClosureModel, ClosureModelBase
from plan.common.fields import StrictForeignKey

from plan.api.dispenser.actions import get_unclosable_services
from plan.api.exceptions import PermissionDenied, ServiceTagsDuplicatingError
from plan.common.models import TimestampedModel
from plan.common.utils.oauth import get_abc_zombik
from plan.contacts.models import AbstractContactBase
from plan.roles.models import Role
from plan.services import denormalization
from plan.services.exceptions import (
    ActionAlreadyInProgress,
    CannotDeleteService,
    CannotCloseService,
    CannotCloseOrDeleteBaseService,
    FryParadox,
    SameParent,
    ServiceReadonlyError,
    ServicesDontHaveActiveResponsibles,
)
from plan.suspicion.constants import ServiceIssueStates
from plan.services.state import (
    SERVICE_STATE,
    SERVICEMEMBER_STATE,
)
from plan.staff.base import I18nModel, I18nBaseModel
from plan.staff.models import Department, Staff
from plan.oebs.utils import is_oebs_related
from plan.oebs.models import OEBSAgreement
from plan.oebs.exceptions import NotAppliedOEBSAgreementExists
from plan.oebs.constants import OEBS_FLAGS, OEBS_GROUP_ONLY_FLAG


log = logging.getLogger(__name__)

MAX_SERVICE_LEVEL = 10

# Пока в базе есть слаги в верхнем регистре, используем два паттерна
LEGACY_SERVICE_SLUG_PATTERN = re.compile(r'^[a-zA-Z0-9_-]+$')
SERVICE_SLUG_PATTERN = re.compile(r'^[a-z0-9_-]+$')


def get_default_flags():
    return {
        'duty1': False,
        'duty2': True
    }


def get_default_functions():
    from plan.services.functionality import FUNCTIONALITIES_CHECKER
    return list(FUNCTIONALITIES_CHECKER.keys())


class ServiceRequestMixin:
    PROCESSING_OEBS = 'processing_oebs'
    PROCESSING_D = 'processing_d'
    PROCESSED_D = 'processed_d'
    PROCESSING_ABC = 'processing_abc'
    PROCESSING_IDM = 'processing_idm'
    APPROVED = 'approved'
    REQUESTED = 'requested'
    COMPLETED = 'completed'
    PARTIALLY_APPROVED = 'partially_approved'
    REJECTED = 'rejected'
    FAILED = 'failed'

    def get_active_agreement(self):
        return self.oebsagreement_set.not_applied().first()

    @transition(field='state', target=PROCESSING_OEBS)
    def start_oebs_process(self):
        pass

    @transition(field='state', source=(PROCESSING_OEBS, APPROVED, PROCESSING_IDM, REQUESTED), target=PROCESSING_D)
    def process_oebs(self):
        oebs_agreement = self.get_active_agreement()
        if oebs_agreement:
            raise NotAppliedOEBSAgreementExists()

    @transition(field='state', source=PROCESSING_D, target=PROCESSED_D)
    def process_d(self):
        pass

    @transition(field='state', source=PROCESSING_OEBS, target=FAILED)
    def fail(self):
        service_with_descendants = self.service.get_descendants(include_self=True)
        service_with_descendants.update(readonly_state=None)

    def is_closable_from_d_side(self):
        service_with_descendants = self.service.get_descendants(include_self=True).active()
        service_ids = [service.id for service in service_with_descendants]
        if service_ids:
            return get_unclosable_services(service_ids)


class MemberWithState(models.Model):
    states = SERVICEMEMBER_STATE

    all_states = models.Manager()

    state = models.CharField(
        default=states.REQUESTED,
        max_length=32,
        db_index=True,
        verbose_name=_('Статус'),
        editable=True,
        choices=[
            (state, SERVICEMEMBER_STATE[state].name)
            for state in SERVICEMEMBER_STATE
        ]
    )
    granted_at = models.DateTimeField(null=True, blank=True)
    depriving_at = models.DateTimeField(null=True, blank=True)
    deprived_at = models.DateTimeField(null=True, blank=True)
    expires_at = models.DateField(null=True, blank=True)

    idm_role_id = models.IntegerField(null=True, blank=True)

    class Meta:
        abstract = True

    def request(self, role_id, expires_at=None, update_fields=('state', 'idm_role_id', 'expires_at')):
        self.idm_role_id = role_id
        self.expires_at = expires_at
        if self.state != SERVICEMEMBER_STATE.ACTIVE:
            self.state = self.states.REQUESTED
        self.save(update_fields=update_fields)

    def activate(self):
        self.state = self.states.ACTIVE
        self.granted_at = timezone.now()
        self.save(update_fields=('state', 'granted_at'))

    def set_depriving_state(self):
        self.state = SERVICEMEMBER_STATE.DEPRIVING
        self.depriving_at = timezone.now()
        self.save(update_fields=('state', 'depriving_at'))

    def deprive(self):
        self.state = SERVICEMEMBER_STATE.DEPRIVED
        self.deprived_at = timezone.now()
        self.save(update_fields=('state', 'deprived_at'))

    def send_update_schedule(self, *args, **kwargs):
        from plan.duty.tasks import update_schedule
        update_schedule.delay(*args, **kwargs)


def validate_is_hexcode(value):
    msg = '{} is not a color hexcode'.format(value)

    if not value.startswith('#'):
        raise ValidationError(msg)

    value = value.lstrip('#')
    if not len(value) in (3, 6):
        raise ValidationError(msg)

    try:
        int(value, 16)
    except ValueError:
        raise ValidationError(msg)


class ServiceTagQuerySet(models.QuerySet):
    def gradient(self):
        return self.filter(slug__in=settings.GRADIENT_TAGS)

    def oebs(self):
        return self.filter(slug__in=OEBS_FLAGS.values())


class ServiceTag(TimestampedModel, I18nModel):
    objects = ServiceTagQuerySet.as_manager()

    VTEAM_NAME = 'v-team'

    name = models.CharField(max_length=255, verbose_name=_('Название'))
    name_en = models.CharField(max_length=255, verbose_name=_('Название (en)'))
    slug = models.SlugField(verbose_name=_('Код'), unique=True)
    color = models.CharField(max_length=255, verbose_name=_('Цвет'), validators=[validate_is_hexcode])
    description = models.TextField(
        blank=True, default='',
        verbose_name='Описание',
    )
    description_en = models.TextField(
        blank=True, default='',
        verbose_name='Описание (en)',
    )
    is_protected = models.BooleanField(
        default=False,
        verbose_name='Запрещено выдавать права на редактирование'
    )

    def __str__(self):
        return 'ServiceTag: {} / {}'.format(self.name, self.name_en)

    class Meta:
        unique_together = ('name', 'name_en', 'color')
        verbose_name = _('Тег сервиса')
        verbose_name_plural = _('Теги сервисов')


def _get_service_by_pk_or_slug(queryset, value):
    # todo: deprecate
    if isinstance(value, int) or value.isdigit():
        return queryset.get(id=int(value))
    elif LEGACY_SERVICE_SLUG_PATTERN.match(value):
        return queryset.get(slug__iexact=value)
    else:
        raise queryset.model.DoesNotExist


class ServiceTypeFunction(models.Model):
    code = models.CharField(max_length=64, unique=True, db_index=True)
    name = models.CharField(max_length=64)
    name_en = models.CharField(max_length=64)
    active = models.BooleanField(default=True)
    description = models.TextField(
        default='',
        blank=True,
        verbose_name='Описание',
    )
    description_en = models.TextField(
        default='',
        blank=True,
        verbose_name='Описание (en)',
    )

    def __str__(self):
        return self.__repr__()

    def __repr__(self):
        return f'Function {self.code}, name: {self.name}, active: {self.active}'


class ServiceType(models.Model):
    SERVICE = 'service'
    TEAM = 'team'
    UNDEFINED = 'undefined'

    code = models.CharField(max_length=64, unique=True, db_index=True)
    name = models.CharField(max_length=64)
    name_en = models.CharField(max_length=64)
    functions = models.ManyToManyField(
        ServiceTypeFunction,
        related_name='used_in_types',
        verbose_name=_('Functions'),
        blank=True,
        default=None,
        null=True,
    )
    available_for_user = models.BooleanField(default=True)
    available_parents = models.ManyToManyField(
        'self',
        null=True,
        blank=True,
        related_name='available_children',
    )

    def __str__(self):
        return self.__repr__()

    def __repr__(self):
        return f'ServiceType {self.code}, name: {self.name}'

    @classmethod
    def get_default_type(cls):
        if not waffle.switch_is_active(settings.SWITCH_CHECK_ALLOWED_PARENT_TYPE):
            return cls.objects.get(code=cls.UNDEFINED)


class ServiceQuerySet(models.QuerySet):
    def get_resource_suppliers_pks(self, resource_category, fresh=False):
        pks = None
        if not fresh:
            pks = cache.get(f'{resource_category}_suppliers_pks')
        if pks is None:
            pks = list(
                self.filter(slug__in=settings.RESOURCE_SUPPLIERS_SLUGS[resource_category]).values_list('pk', flat=True)
            )
            cache.set(f'{resource_category}_suppliers_pks', pks, timeout=settings.RESOURCE_SUPPLIERS_CACHE_TIMEOUT)
        return pks

    def get_by_pk_or_slug(self, value):
        return _get_service_by_pk_or_slug(self, value)

    def exportable(self):
        return self.filter(is_exportable=True)

    def active(self):
        return self.filter(state__in=Service.states.ACTIVE_STATES)

    def inactive(self):
        return self.exclude(state__in=Service.states.ACTIVE_STATES)

    def alive(self):
        return self.filter(state__in=Service.states.ALIVE_STATES)

    def in_readonly(self):
        return self.filter(readonly_state__isnull=False)

    def with_oebs_flags(self):
        return self.filter(
            Q(use_for_hardware=True) | Q(use_for_hr=True) | Q(use_for_procurement=True) | Q(use_for_revenue=True))

    def base_non_leaf(self):
        return self.base().filter(children__is_base=True)

    def oebs_deviation(self):
        return self.filter(
            oebs_parent_id__isnull=False,
            oebs_data__deviation_reason__isnull=False,
        )

    def base(self):
        return self.filter(is_base=True)

    def unsent_sandbox_announcements(self, next_workday):
        services_with_notifications = (
            ServiceNotification.objects.sent_notifications(ServiceNotification.SANDBOX_ANNOUNCEMENT)
            .select_related('service')
            .values_list('service__pk', flat=True)
        )

        return self.filter(sandbox_move_date=next_workday).exclude(pk__in=services_with_notifications)

    def without_notifications(self, notification_id, day):
        services_with_notifications = (
            ServiceNotification.objects
            .filter(notification_id=notification_id, sent_at=day,)
            .select_related('service')
            .values_list('service__pk', flat=True)
        )
        return self.exclude(pk__in=services_with_notifications)

    def by_parent_list(self, parents):
        if 0 in parents:
            queryset = self.filter(models.Q(parent_id__in=parents) | models.Q(parent_id=None))
        else:
            queryset = self.filter(parent_id__in=parents)
        return queryset

    def get_parents(self):
        parent_pks = Service._closure_model.objects.filter(
            ~models.Q(parent=models.F('child')),
            child__in=self,
        ).values('parent')
        return Service.objects.filter(models.Q(pk__in=parent_pks))

    def children_of(self, pk):
        return self.filter(serviceclosure_parents__parent_id=pk).exclude(id=pk)

    def own_only(self, requester):
        services_ids = ()
        if requester:
            services_ids = (
                ServiceMember.objects
                .filter(staff_id=requester.id)
                .values_list('service_id', flat=True)
            )
        return self.filter(id__in=services_ids)

    def with_responsible(self, person, with_descendants=False, with_duty=False):
        target_roles = Role.RESPONSIBLES
        if with_duty:
            target_roles = Role.RESPONSIBLE_FOR_DUTY_OR_SERVICE
        ancestors = Service.objects.alive().filter(
            members__staff=person.staff,
            members__role__code__in=target_roles,
            members__state__in=[SERVICEMEMBER_STATE.ACTIVE, SERVICEMEMBER_STATE.DEPRIVING]
        )
        if with_descendants:
            return self.filter(serviceclosure_parents__parent__in=ancestors)
        else:
            return self.filter(pk__in=ancestors.values('pk'))

    def gradient(self):
        return self.active().filter(valuestream__isnull=False)


class ServiceManager(models.Manager.from_queryset(ServiceQuerySet)):
    use_in_migrations = True

    # ToDo: кешировать после подключения redis
    def get_sandbox(self):
        return self.get(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG)


class ClosureI18nModelBase(ClosureModelBase, I18nBaseModel):
    pass


class Service(ClosureModel, I18nModel, metaclass=ClosureI18nModelBase):
    objects = ServiceManager()

    states = SERVICE_STATE

    state = models.CharField(
        default=states.IN_DEVELOP,
        max_length=32,
        db_index=True,
        verbose_name=_('Статус'),
        editable=True,
        choices=[
            (state, SERVICE_STATE[state].name)
            for state in SERVICE_STATE
        ]
    )

    parent = models.ForeignKey(
        'self',
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name='children',
        editable=True,
    )

    membership_inheritance = models.BooleanField(
        default=False,
        verbose_name=_('Разрешить наследование свойств родителя'),
    )

    name = models.CharField(
        max_length=255,
        verbose_name=_('Название'),
    )
    name_en = models.CharField(
        max_length=255,
        default='',
        blank=True,
        verbose_name='Название (en)',
    )
    slug = models.SlugField(
        unique=True,
        verbose_name=_('Код'),
    )
    team = models.ManyToManyField(
        Staff,
        through='services.ServiceMember',
        related_name='services',
    )
    description = models.TextField(
        default='',
        blank=True,
        verbose_name='Описание',
    )
    description_en = models.TextField(
        default='',
        blank=True,
        verbose_name='Описание (en)',
    )
    activity = models.TextField(
        default='',
        blank=True,
        verbose_name='Деятельность',
    )
    activity_en = models.TextField(
        default='',
        blank=True,
        verbose_name='Деятельность (en)',
    )
    # TODO: выпилить это поле, когда можно будет
    # Сейчас оно отображается в каталоге сервисов
    url = models.CharField(
        max_length=255,
        default='',
        blank=True,
        verbose_name=_('Адрес сервиса'),
    )
    keywords = models.TextField(
        default='',
        blank=True,
        verbose_name=_('Ключевые слова'),
    )
    position = models.IntegerField(
        db_index=True,
        default=0,
        editable=False,
    )
    owner = StrictForeignKey(
        Staff,
        null=True,
        blank=True,
        related_name='owner_of_services',
        verbose_name=_('Владелец'),
        on_delete=models.SET_NULL,
    )
    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)

    member_departments = models.ManyToManyField(
        Department,
        through='services.ServiceMemberDepartment',
        verbose_name=_('Связанные подразделения'),
    )

    base_of = models.ManyToManyField(
        'self',
        blank=True,
        related_name='based_on',
        verbose_name='Основа для',
        symmetrical=False,
    )

    is_external = models.BooleanField(
        default=False,
        verbose_name=_('Внешний сервис'),
    )

    is_important = models.BooleanField(
        default=False,
        verbose_name=_('Важный'),
    )

    suspicious_date = models.DateField(
        blank=True,
        null=True,
        verbose_name=_('Дата выставления подозрительности')
    )

    suspicious_notification_date = models.DateField(
        blank=True,
        null=True,
        verbose_name=_('Дата последнего напоминания о подозрительности')
    )

    sandbox_move_date = models.DateField(
        blank=True,
        null=True,
        verbose_name=_('Дата пермещения в Песочницу')
    )

    service_type = models.ForeignKey(ServiceType)

    related_services = models.ManyToManyField(
        'self',
        blank=True,
        default=None,
        verbose_name=_('Связанные сервисы'),
        symmetrical=True
    )

    departments = models.ManyToManyField(
        Department,
        default=None,
        blank=True,
        related_name='services',
        verbose_name=_('Подразделения'),
    )

    is_base = models.BooleanField(
        default=False,
        null=False,
        verbose_name=_('Базовый сервис'),
    )

    CREATING = 'creating'
    MOVING = 'moving'
    RENAMING = 'renaming'
    DELETING = 'deleting'
    CLOSING = 'closing'
    READONLY_STATES = (
        (CREATING, _('Создается')),
        (MOVING, _('Перемещается')),
        (RENAMING, _('Переименовывается')),
        (DELETING, _('Удаляется')),
        (CLOSING, _('Закрывается')),
    )

    readonly_state = models.CharField(
        default=None, null=True, blank=True,
        max_length=50,
        choices=READONLY_STATES,
    )

    readonly_start_time = models.DateTimeField(default=None, null=True, blank=True)

    staff_id = models.PositiveIntegerField(db_index=True, null=True, blank=True)

    # denormalization
    unique_members_count = models.PositiveSmallIntegerField(
        default=0,
        verbose_name='Число уникальных участников сервиса',
    )

    unique_immediate_members_count = models.PositiveSmallIntegerField(
        default=0,
        verbose_name='Число уникальных непосредственных участников сервиса',
    )

    unique_immediate_robots_count = models.PositiveSmallIntegerField(
        default=0,
        verbose_name='Число уникальных роботов в сервисе',
    )

    unique_immediate_external_members_count = models.PositiveSmallIntegerField(
        default=0,
        verbose_name='Число уникальных непосредственных внешних участников сервиса',
    )

    weight = models.PositiveSmallIntegerField(
        default=0,
        verbose_name='Важность (вес) сервиса',
    )
    path = models.CharField(
        verbose_name='Путь по slug проектов',
        max_length=1000,
        default='',
        blank=True
    )

    kpi_bugs_count = models.PositiveIntegerField(
        verbose_name='Количество багов',
        null=True, blank=True, default=None,
    )
    kpi_release_count = models.PositiveIntegerField(
        verbose_name='Количество релизов',
        null=True, blank=True, default=None,
    )
    kpi_lsr_count = models.PositiveIntegerField(
        verbose_name='Количество LSR',
        null=True, blank=True, default=None,
    )
    vacancies_count = models.PositiveIntegerField(
        verbose_name='Количество вакансий',
        null=True, blank=True, default=None,
    )

    ancestors = JSONField(verbose_name='Родители', blank=True, default=dict)
    functions = JSONField(verbose_name='Service functions', blank=True, default=get_default_functions)

    children_count = models.PositiveIntegerField(
        verbose_name='Children', null=True, blank=True, default=None,
        help_text='Число непосредственных дочерних сервисов',
    )

    descendants_count = models.PositiveIntegerField(
        verbose_name='Descendants', null=True, blank=True, default=None,
        help_text='Число дочерних сервисов с учетом вложенных',
    )

    tags = models.ManyToManyField(ServiceTag, blank=True)

    is_exportable = models.BooleanField(
        default=False,
        verbose_name='Доступен в апи по умолчанию'
    )

    idm_roles_count = models.PositiveIntegerField(
        verbose_name='Количество запрошенных на сервис ролей в IDM',
        null=True,
        blank=True,
        default=None
    )
    puncher_rules_count = models.PositiveIntegerField(
        verbose_name='Количество запрошенных на сервис правил в Puncher',
        null=True,
        blank=True,
        default=None
    )

    has_external_members = models.BooleanField(
        default=False,
        verbose_name='Есть внешние',
    )
    has_forced_suspicious_reason = models.BooleanField(
        default=False,
        verbose_name='Помечен вышестоящим руководителем как подозрительный',
    )

    valuestream = models.ForeignKey(
        'self',
        verbose_name='Valuestream',
        null=True,
        blank=True,
        related_name='gradient_descendant_for_valuestream',
    )

    umbrella = models.ForeignKey(
        'self',
        verbose_name='Зонтик',
        null=True,
        blank=True,
        related_name='gradient_descendant_for_umbrella',
    )

    use_for_hardware = models.BooleanField(
        default=False,
        verbose_name=_('Используется для учета железа'),
    )
    use_for_group_only = models.BooleanField(
        default=False,
        verbose_name=_('Используется только для группировки'),
    )
    use_for_hr = models.BooleanField(
        default=False,
        verbose_name=_('Используется в HR'),
    )
    use_for_procurement = models.BooleanField(
        default=False,
        verbose_name=_('Используется в Закупках'),
    )
    use_for_revenue = models.BooleanField(
        default=False,
        verbose_name=_('Используется в Выручке'),
    )

    oebs_parent_id = models.IntegerField(
        db_index=True,
        null=True,
        blank=True,
        verbose_name=_('ID родителя в OEBS'),
    )
    oebs_data = JSONField(blank=True, default=dict, verbose_name='Данные из OEBS')

    flags = JSONField(verbose_name='Флаги сервиса', blank=True, default=get_default_flags)

    def state_display(self):
        return force_text(self.states[self.state])

    @property
    def active_agreement(self):
        return self.oebs_agreements.active().first()

    @property
    def has_oebs_gradient_tags(self):
        return self.tags.filter(
            slug__in=[settings.GRADIENT_VS, settings.BUSINESS_UNIT_TAG]
        ).exists()

    @property
    def oebs_resource(self):
        return self.serviceresource_set.filter(
            type__code=settings.OEBS_PRODUCT_RESOURCE_TYPE_CODE
        ).first()

    # можно это автоматически инспектить или сделать моделфилд для этого и туда
    # писать, чтобы не расходилось с полями, но пока так для простоты.
    DENORMALIZED_FIELDS = {
        'unique_members_count': {
            'calculator': denormalization.count_unique_members,
        },
        'unique_immediate_members_count': {
            'calculator': denormalization.count_unique_immediate_members
        },
        'unique_immediate_robots_count': {
            'calculator': denormalization.count_unique_immediate_robots
        },
        'unique_immediate_external_members_count': {
            'calculator': denormalization.count_unique_immediate_external_members,
        },
        'weight': {
            'calculator': denormalization.count_unique_members
        },
        'path': {
            'calculator': denormalization.get_service_path
        },
        'ancestors': {
            'calculator': denormalization.get_service_ancestors
        },
        'children_count': {
            'calculator': lambda service: service.get_alive_children().count()
        },
        'descendants_count': {
            'calculator': lambda service: service.get_alive_descendants().count()
        },
        'has_external_members': {
            'calculator': denormalization.get_has_external_members
        },
        'has_forced_suspicious_reason': {
            'calculator': denormalization.has_forced_suspicious_reason
        },
        'vacancies_count': {
            'calculator': denormalization.get_vacancies_count
        }
    }

    class Meta:
        app_label = 'services'
        verbose_name = _('Сервис')
        verbose_name_plural = _('Сервисы')
        permissions = [('view_service', _('Просмотр модели в админке'))]

    def is_base_non_leaf(self):
        return self.is_base and self.children.filter(is_base=True).exists()

    def has_change_state_execution(self):
        from plan.suspicion.models import ServiceIssue, ServiceExecutionAction

        return ServiceExecutionAction.objects.filter(
            applied_at__lte=timezone.now(),
            service_issue__service=self,
            service_issue__state=ServiceIssue.STATES.ACTIVE,
            execution__code__in=settings.CHANGE_STATES_EXECUTIONS,
        ).exists()

    @staticmethod
    def can_staff_place_service_here(staff, parent, is_base):
        if is_base:
            return staff.user.is_superuser
        else:
            return staff.user.is_superuser or (parent is None or not parent.is_base_non_leaf())

    @property
    def type(self):
        if self.parent_id is None:
            return 'meta'
        elif self.descendants_count is not None and self.descendants_count > 0:
            return 'group'
        else:
            return 'service'

    def __str__(self):
        return self.__repr__()

    def __repr__(self):
        return 'Service %s (%s)' % (self.pk, self.slug)

    def save(self, *args, **kwargs):
        if self.pk:
            try:
                prev = self.__class__.objects.select_related('service_type').prefetch_related('related_services').get(pk=self.pk)
            except self.__class__.DoesNotExist:
                pass
            else:
                self.check_is_base(prev)
                self.check_state(prev)

        return super(Service, self).save(*args, **kwargs)

    def check_is_base(self, prev):
        if prev.is_base != self.is_base:
            base_tag = ServiceTag.objects.get(slug=settings.BASE_TAG_SLUG)
            if self.is_base:
                self.tags.add(base_tag)
            else:
                self.tags.remove(base_tag)

    def check_state(self, prev):
        if prev.state == self.states.DELETED and self.state != self.states.DELETED:
            raise ValidationError('Cannot restore service')
        if self.state == SERVICE_STATE.DELETED and prev.state != SERVICE_STATE.DELETED and self.has_moving_descendants():
            raise ValidationError('Cannot delete service with moving descendants')

    @property
    def human_readonly_state(self):
        if self.readonly_state is None:
            return None

        state = force_text(dict(self.READONLY_STATES).get(self.readonly_state, 'Доступен только для чтения'))
        state_en = {
            'creating': 'Is registered in IDM',
            'moving': 'Is moved',
            'renaming': 'Is being renamed',
        }.get(self.readonly_state, 'Is read-only')

        return {
            'id': self.readonly_state,
            'name': state,
            'name_i18n': {
                'ru': state,
                'en': state_en,
            }
        }

    def get_alive_children(self):
        return self.get_children().exclude(state=self.states.DELETED)

    def get_alive_descendants(self, include_self=False):
        return self.get_descendants(include_self=include_self).exclude(state=self.states.DELETED)

    def get_max_descendant_level(self):
        return max(self.get_descendants(include_self=True).values_list('level', flat=True))

    def get_responsible(self, parents=False, include_self=True):
        if parents:
            services = self.get_ancestors(include_self=include_self)
        else:
            services = [self]

        return ServiceMember.objects.responsibles().filter(service__in=services)

    def get_notifiable_about_duty(self):
        recipients = self.members.responsible_for_duty()
        if not recipients:
            recipients = self.members.responsibles()
        return recipients

    def unique_error_message(self, model_class, unique_check):
        if unique_check == ('slug',):
            return _('Сервис с таким кодом уже существует.')
        return super(Service, self).unique_error_message(model_class,
                                                         unique_check)

    @models.permalink
    def get_absolute_url(self):
        return 'services:service', (), {'pk_or_slug': self.pk}

    def get_absolute_abc_url(self):
        return '{}services/{}/'.format(settings.ABC_PREFIX, self.slug)

    @property
    def pk_or_slug(self):
        """
        Без этого ломается GroupedSubscriptionDehydrator:_get_object_url
        plan/notify/dehydration.py
        """
        return self.id

    @property
    def is_alive(self):
        return self.state in self.states.ALIVE_STATES

    @property
    def is_active(self):
        return self.state in self.states.ACTIVE_STATES

    @property
    def is_deleted(self):
        return self.state == self.states.DELETED

    @property
    def review_required(self):
        if not hasattr(self, '_review_required'):
            self._review_required = self.tags.filter(
                slug__in=settings.REVIEW_REQUIRED_TAG_SLUGS
            ).exists()
        return self._review_required

    def has_important_resources(self):
        from plan.resources.models import ServiceResource

        descendants = self.get_descendants(include_self=True).values_list('id', flat=True)

        active_important_resources = ServiceResource.objects.filter(
            type__is_important=True,
            service_id__in=descendants,
            state__in=ServiceResource.ALIVE_STATES
        )

        return active_important_resources.exists()

    def with_oebs_flags(self):
        return any(getattr(self, flag) for flag in OEBS_FLAGS if flag != OEBS_GROUP_ONLY_FLAG)

    def get_link(self) -> str:
        return f'{settings.ABC_URL}/services/{self.slug}/'

    @property
    def is_suspicious(self):
        return self.suspicious_date is not None

    def should_be_moved_to_sandbox(self):
        deadline = timezone.now().date()
        if self.has_important_resources() or self.is_base:
            return False
        if self.sandbox_move_date is not None and self.sandbox_move_date <= deadline:
            return True
        return False

    def remove_suspicion(self):
        self.suspicious_date = None
        self.sandbox_move_date = None
        self.suspicious_notification_date = None
        self.has_forced_suspicious_reason = False

        suspicious_fields = [
            'suspicious_date',
            'sandbox_move_date',
            'suspicious_notification_date',
            'has_forced_suspicious_reason']
        self.save(update_fields=suspicious_fields)
        ServiceSuspiciousReason.objects.filter(service=self, marked_by=None).delete()

        ServiceNotification.objects.filter(
            service=self,
            notification_id=ServiceNotification.SANDBOX_ANNOUNCEMENT
        ).delete()

    def get_responsibles_for_email(self, include_self=False, send_only_to_direct_responsibles=False):
        """
        Возвращает руководителей сервиса и сам сервис, если у сервиса отсутствуют руководители, то идет по дереву вверх
        до ближайшего сервиса, имеющего руководителей.
        Если никого найтие не удалось - выбрасывает исключение.
        """
        if send_only_to_direct_responsibles:
            ancestors = [self]
        else:
            ancestors = self.get_ancestors(include_self=include_self).order_by('-level')

        for ancestor in ancestors:
            responsibles = ancestor.members.responsibles().active().select_related('staff')
            if responsibles:
                return [member.staff for member in responsibles], ancestor
        else:
            raise ServicesDontHaveActiveResponsibles()

    def get_service_traffic_status(self, service_issue_groups):
        if service_issue_groups:
            service_traffic_status = []

            for traffic_status in self.traffic_statuses.all():
                if traffic_status.issue_group.pk not in service_issue_groups:
                    continue

                service_traffic_info = {}
                service_traffic_info['name'] = traffic_status.issue_group.name
                service_traffic_info['code'] = traffic_status.issue_group.code
                # у нас храняться инвертированные данные и не в процентах
                service_traffic_info['weight'] = round(
                    100 - (traffic_status.current_weight * 100)
                )
                service_traffic_info['level'] = traffic_status.level
                service_traffic_status.append(service_traffic_info)

        else:
            service_traffic_status = None

        return service_traffic_status

    def has_moving_descendants(self):
        pks = self.get_descendants(include_self=True).alive().values_list('pk', flat=True)
        active_move_requests = ServiceMoveRequest.objects.filter(service_id__in=pks).active()
        return active_move_requests.exists()

    def is_in_sandbox(self):
        return (
            self.get_ancestors()
            .filter(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG)
            .exists()
        )

    def gradient_type(self) -> str:
        """
        Возвращает градиентный тип сервиса.
        Если у сервиса два градиентных тега, выбрасываем исключением.
        """
        gradient_tags = [tag for tag in self.tags.all() if tag.slug in settings.GRADIENT_TAGS]
        if len(gradient_tags) > 1:
            raise ServiceTagsDuplicatingError

        elif len(gradient_tags) == 1:
            return gradient_tags[0].slug

    def search_gradient_parent(self, gradient_type_parent: str, ancestors: Union[Iterable['Service'], None] = None):
        """
        Ищет для сервиса valuestream (или зонтик). Приоритетней тот vs (зонтик), что выше.
        Если зонтик без valuestream, не считаем его полноценным зонтиком.
        Если у сервиса есть два тега, то зануляем valuestream и umbrella для всей ветки с его потомками.

        gradient_type_parent - слаг градиентного тега, который нужно найти для сервиса
        """

        if gradient_type_parent not in settings.GRADIENT_TAGS:
            return None

        if not ancestors:
            ancestors = self.get_ancestors().gradient().prefetch_related('tags').order_by('level')

        gradient_parent = None
        for ancestor in ancestors:
            gradient_type = ancestor.gradient_type()
            if not gradient_parent and gradient_type == gradient_type_parent:
                # нашли самый высокий vs или зонтик
                gradient_parent = ancestor

        return gradient_parent

    def update_gradient_fields(self, gradient_fields: dict, include_self: bool = True):
        """
        Обновляет градиентные поля gradient_fields для указанного сервиса
            include_self: включая себя
        Возвращает queryset обновленных сервисов
        """

        update_services = self.get_descendants(include_self=include_self)
        update_services.update(**gradient_fields)
        return update_services

    def is_closable_from_d_side(self) -> list[int]:
        service_with_descendants = self.get_descendants(include_self=True).active()
        service_ids = [service.id for service in service_with_descendants]
        if service_ids:
            return get_unclosable_services(service_ids)


def action_to_service_issue(action):
    issue = action.service_issue
    issue.action = action
    return issue


class ServiceCreateRequestQuerySet(models.QuerySet):
    def active(self):
        return self.exclude(state__in=ServiceCreateRequest.INACTIVE_STATES)

    def inactive(self):
        return self.filter(state__in=ServiceCreateRequest.INACTIVE_STATES)

    def requested(self):
        return self.filter(state=ServiceCreateRequest.REQUESTED)

    def approvable(self, services: Iterable[Service]) -> Iterable['ServiceCreateRequest']:
        """
        Ищем все запросы на создание сервисов, которые можно заапрувить для указанных сервисов.
        """

        create_requests = self.requested().filter(
            move_to__in=services,
            approver_incoming__isnull=True
        )
        return create_requests


class ServiceCreateRequest(TimestampedModel):
    objects = ServiceCreateRequestQuerySet.as_manager()

    service = models.ForeignKey(
        Service,
        related_name='create_requests',
    )
    requester = models.ForeignKey(Staff)
    move_to = models.ForeignKey(
        Service,
        default=None,
        null=True,
        related_name='move_to'
    )

    REQUESTED = 'requested'
    APPROVED = 'approved'
    REJECTED = 'rejected'
    PROCESSING_IDM = 'processing_idm'
    PROCESSING_HEAD = 'processing_head'
    PROCESSING_ABC = 'processing_abc'
    COMPLETED = 'completed'

    STATES = (
        (REQUESTED, 'Запрошен'),
        (APPROVED, 'Одобрен'),
        (REJECTED, 'Отклонен'),
        (PROCESSING_IDM, 'Обрабатывается в IDM'),
        (PROCESSING_HEAD, 'Запрашивается роль для руководителя'),
        (PROCESSING_ABC, 'Обрабатывается в ABC'),
        (COMPLETED, 'Завершен'),
    )

    INACTIVE_STATES = {REJECTED, COMPLETED}

    completed_at = models.DateTimeField(null=True)
    interactive_at = models.DateTimeField(null=True)

    state = FSMField(
        verbose_name=_('Статус'),
        choices=STATES,
        default=REQUESTED,
        db_index=True,
    )

    approver_incoming = models.ForeignKey(
        Staff,
        null=True,
        blank=True,
        verbose_name='Подтвердил в родительском сервисе',
        related_name='+',
    )

    def __str__(self):
        if self.move_to is not None:
            service_slug = '{}/{}'.format(self.move_to.slug, self.service.slug)
        else:
            service_slug = self.service.slug
        return 'Create {} by {}'.format(service_slug, self.requester.login)

    @classmethod
    def request(cls, service: Service, requester: 'Person', move_to: Service | None = None) -> 'ServiceCreateRequest':
        from plan.services import permissions
        if service.readonly_state is not None:
            if service.readonly_state != Service.CREATING:
                raise ServiceReadonlyError(service)
            elif cls.objects.filter(service=service).active().exists():
                raise ActionAlreadyInProgress(service)
        elif not Service.can_staff_place_service_here(requester.staff, service.parent, service.is_base):
            raise PermissionDenied(
                code='incorrect_parent',
                detail='incorrect service parent',
                title={
                    'ru': 'Неправильный родитель сервиса',
                    'en': 'Incorrect service parent',
                },
                message={
                    'ru': 'Вы не можете сделать сервис потомком "{}"'.format(service.parent.name),
                    'en': 'You can\'t make service a descendant of "{}"'.format(service.parent.name_en)
                }
            )
        else:
            service.readonly_state = Service.CREATING
            service.readonly_start_time = timezone.now()
            log.info('Changing readonly_state(CREATING) on %s', service.slug)
            service.save()

        create_request = cls.objects.create(
            service=service,
            requester=requester.staff,
            move_to=move_to,
        )

        create_in_sandbox = move_to is None or move_to.slug == settings.ABC_DEFAULT_SERVICE_PARENT_SLUG

        if permissions.can_approve_create_request(service, requester) or create_in_sandbox:
            create_request.approve(approver=requester)
            create_request.refresh_from_db()

        return create_request

    @transition(field=state, source=(REQUESTED, APPROVED), target=PROCESSING_IDM)
    def process_idm(self):
        pass

    @transition(field=state, source=PROCESSING_IDM, target=PROCESSING_HEAD)
    def process_head(self):
        pass

    @transition(field=state, source=PROCESSING_HEAD, target=PROCESSING_ABC)
    def process_abc(self):
        pass

    @transition(field=state, source=PROCESSING_ABC, target=COMPLETED)
    def complete(self):
        log.info('Changing readonly_state(None) on %s', self.service.slug)
        self.service.readonly_state = None
        self.completed_at = timezone.now()

    def approve(self, approver: 'Person'):
        self._approve_transition(approver)
        self.save(update_fields=['state', 'approver_incoming'])
        # при отказе от создания в песочнице ABC-8821
        # тут должен появится код про регистрацию узлов register_service
        # и про рассчет градиентных полей calculate_gradient_fields

    @transition(field=state, source=REQUESTED, target=APPROVED)
    def _approve_transition(self, approver: 'Person'):
        from plan.services.permissions import can_approve_create_request

        # если создаётся непосредственно в песочнице, можно без апрува
        create_in_sandbox = self.move_to is None or self.move_to.slug == settings.ABC_DEFAULT_SERVICE_PARENT_SLUG
        if not create_in_sandbox and not can_approve_create_request(self.service, approver):
            raise PermissionDenied()

        self.approver_incoming = approver.staff

    def reject(self, rejector: 'Person'):
        self._reject_transition(rejector)
        self.save(update_fields=['state', 'approver_incoming'])
        # пока просто переводим статус в отказ,
        # по плану, в ABC-8821, тут появится ещё код,
        # когда сервисы не будут создаваться сразу в песочнице, при отказе будем переносить их в песочницу

    @transition(field=state, source=REQUESTED, target=REJECTED)
    def _reject_transition(self, rejector: 'Person'):
        from plan.services.permissions import can_approve_create_request
        if not can_approve_create_request(self.service, rejector):
            raise PermissionDenied()

        self.approver_incoming = rejector.staff


class ServiceDeleteRequest(ServiceRequestMixin, TimestampedModel):
    service = models.ForeignKey(Service)
    requester = models.ForeignKey(Staff)

    STATES = (
        (ServiceRequestMixin.REQUESTED, 'Запрошен'),
        (ServiceRequestMixin.PROCESSING_IDM, 'Обрабатывается в IDM'),
        (ServiceRequestMixin.PROCESSING_OEBS, 'Согласовывается в OEBS'),
        (ServiceRequestMixin.PROCESSING_D, 'Обрабатывается на стороне сервиса управления ресурсами'),
        (ServiceRequestMixin.PROCESSED_D, 'Согласовано на стороне сервиса управления ресурсами'),
        (ServiceRequestMixin.PROCESSING_ABC, 'Обрабатывается в ABC'),
        (ServiceRequestMixin.COMPLETED, 'Завершен'),
        (ServiceRequestMixin.FAILED, 'Ошибка'),
    )

    state = FSMField(
        verbose_name=_('Статус'),
        choices=STATES,
        default=ServiceRequestMixin.REQUESTED,
        db_index=True,
    )

    def __str__(self):
        return 'Delete {} by {}'.format(self.service.slug, self.requester.login)

    @classmethod
    @transaction.atomic()
    def request(cls, service, requester):
        from plan.services import permissions
        if not permissions.is_service_responsible(service, requester):
            raise CannotDeleteService()

        if not requester.staff.user.is_superuser and service.is_base:
            raise PermissionDenied(code='Cannot delete base service')

        service_with_descendants = service.get_descendants(include_self=True)
        log.info('Changing readonly_state(DELETING) on %s', list(service_with_descendants.values_list('slug', flat=True)))
        service_with_descendants.update(readonly_state=Service.DELETING, readonly_start_time=timezone.now())
        ServiceNotification.objects.filter(
            service__in=service_with_descendants,
            sent_at=None,
        ).delete()

        request = cls.objects.create(service=service, requester=requester.staff)

        oebs_related = is_oebs_related(service, ignore_dormant=True)
        if oebs_related:
            OEBSAgreement.objects.delete_service(
                service=service,
                requester=requester.staff,
                delete_request=request,
            )
        else:
            from plan.services.tasks import delete_service
            delete_service.apply_async(
                args=[request.id],
                countdown=settings.ABC_DEFAULT_COUNTDOWN,
            )

        return request

    @transition(field=state, source=ServiceRequestMixin.PROCESSED_D, target=ServiceRequestMixin.PROCESSING_IDM)
    def process_idm(self):
        pass

    @transition(field=state, source=ServiceRequestMixin.PROCESSING_IDM, target=ServiceRequestMixin.PROCESSING_ABC)
    def process_abc(self):
        pass

    @transition(field=state, source=ServiceRequestMixin.PROCESSING_ABC, target=ServiceRequestMixin.COMPLETED)
    def complete(self):
        pass


class ServiceCloseRequest(ServiceRequestMixin, TimestampedModel):
    service = models.ForeignKey(Service)
    requester = models.ForeignKey(Staff)

    STATES = (
        (ServiceRequestMixin.REQUESTED, 'Запрошен'),
        (ServiceRequestMixin.PROCESSING_ABC, 'Обрабатывается в ABC'),
        (ServiceRequestMixin.PROCESSING_OEBS, 'Согласовывается в OEBS'),
        (ServiceRequestMixin.PROCESSING_D, 'Согласовывается в D'),
        (ServiceRequestMixin.PROCESSED_D, 'Согласовано в D'),
        (ServiceRequestMixin.COMPLETED, 'Завершен'),
        (ServiceRequestMixin.FAILED, 'Ошибка'),
    )

    state = FSMField(
        verbose_name=_('Статус'),
        choices=STATES,
        default=ServiceRequestMixin.REQUESTED,
        db_index=True,
    )

    def __str__(self):
        return 'Close {} by {}'.format(self.service.slug, self.requester.login)

    @classmethod
    @transaction.atomic()
    def request(cls, service, requester):
        from plan.services import permissions
        if not permissions.is_service_responsible(service, requester):
            raise CannotCloseService()

        if not requester.staff.user.is_superuser and service.is_base:
            raise CannotCloseOrDeleteBaseService(code='Cannot close base service')

        service_with_descendants = service.get_alive_descendants(include_self=True)
        log.info(
            'Changing readonly_state(CLOSING) on %s',
            list(service_with_descendants.values_list('slug', flat=True))
        )

        service_with_descendants.update(
            readonly_state=Service.CLOSING,
            readonly_start_time=timezone.now(),
        )
        ServiceSuspiciousReason.objects.filter(service__in=service_with_descendants).delete()
        ServiceNotification.objects.filter(
            service__in=service_with_descendants,
            sent_at=None,
        ).delete()

        request = cls.objects.create(service=service, requester=requester.staff)

        oebs_related = is_oebs_related(service, ignore_dormant=True)
        if oebs_related:
            OEBSAgreement.objects.close_service(
                service=service,
                requester=requester.staff,
                close_request=request,
            )
        else:
            from plan.services.tasks import close_service
            close_service.apply_async(
                args=[request.id],
                countdown=settings.ABC_DEFAULT_COUNTDOWN,
            )

        return request

    @transition(field=state, source=ServiceRequestMixin.PROCESSED_D, target=ServiceRequestMixin.PROCESSING_ABC)
    def process_abc(self):
        pass

    @transition(field=state, source=ServiceRequestMixin.PROCESSING_ABC, target=ServiceRequestMixin.COMPLETED)
    def complete(self):
        pass


class ServiceMoveRequestQuerySet(models.QuerySet):

    def approved(self):
        return self.filter(
            approver_incoming__isnull=False,
            approver_outgoing__isnull=False
        )

    def processing_abc(self):
        return self.filter(state=ServiceMoveRequest.PROCESSING_ABC)

    def active(self):
        return self.filter(state__in=ServiceMoveRequest.ACTIVE_STATES)

    def dead(self):
        return self.filter(state__in=ServiceMoveRequest.INACTIVE_STATES)

    def reject(self):
        self.filter(
            state__in=(ServiceMoveRequest.REQUESTED, ServiceMoveRequest.PARTIALLY_APPROVED)
        ).update(state=ServiceMoveRequest.REJECTED)

    def approvable(self, services: Iterable[Service]) -> Iterable['ServiceMoveRequest']:
        """
        Ищем все запросы на перенос сервисов, которые можно заапрувить в указанных сервисах.
        """

        move_requests = self.active().filter(
            (Q(service__in=services) & Q(approver_outgoing__isnull=True)) |
            (Q(destination__in=services) & Q(approver_incoming__isnull=True))
        )
        return move_requests


class ServiceMoveRequest(ServiceRequestMixin, TimestampedModel):

    objects = ServiceMoveRequestQuerySet.as_manager()

    requester = models.ForeignKey(
        Staff,
        related_name='service_move_requests',
    )
    service = models.ForeignKey(
        Service,
        related_name='move_requests',
    )
    destination = models.ForeignKey(
        Service,
        related_name='incoming_move_requests',
        null=True, blank=True,
    )
    source = models.ForeignKey(
        Service,
        default=None, null=True, blank=True,
        verbose_name='Бывший родитель сервиса.',
    )

    start_moving_idm = models.BooleanField(
        default=False,
        verbose_name='Migration process on the IDM side is started',
    )

    approver_outgoing = models.ForeignKey(
        Staff,
        null=True, blank=True,
        related_name='+',
    )
    approver_incoming = models.ForeignKey(
        Staff,
        null=True, blank=True,
        related_name='+',
    )
    from_creation = models.BooleanField(
        default=False,
        verbose_name='Этот реквест создан из ServiceCreateRequest',
    )

    completed_at = models.DateTimeField(null=True)
    interactive_at = models.DateTimeField(null=True)
    outgoing_notified = models.DateTimeField(null=True)
    incoming_notified = models.DateTimeField(null=True)

    STATES = (
        (ServiceRequestMixin.REQUESTED, 'Запрошен'),
        (ServiceRequestMixin.PARTIALLY_APPROVED, 'Частично одобрен'),
        (ServiceRequestMixin.APPROVED, 'Одобрен'),
        (ServiceRequestMixin.REJECTED, 'Отклонен'),
        (ServiceRequestMixin.PROCESSING_IDM, 'Обрабатывается на стороне IDM'),
        (ServiceRequestMixin.PROCESSING_ABC, 'Обрабатывается на стороне ABC'),
        (ServiceRequestMixin.PROCESSING_OEBS, 'Согласовывается в OEBS'),
        (ServiceRequestMixin.PROCESSING_D, 'Согласовывается в D'),
        (ServiceRequestMixin.PROCESSED_D, 'Согласовано в D'),
        (ServiceRequestMixin.COMPLETED, 'Завершен'),
        (ServiceRequestMixin.FAILED, 'Ошибка'),
    )

    INACTIVE_STATES = {ServiceRequestMixin.REJECTED, ServiceRequestMixin.COMPLETED, ServiceRequestMixin.FAILED}
    ACTIVE_STATES = {_state for _state, desc in STATES} - INACTIVE_STATES

    state = FSMField(
        verbose_name=_('Статус'),
        choices=STATES,
        default=ServiceRequestMixin.REQUESTED,
        db_index=True,
    )

    @classmethod
    def request(cls, service, destination, requester, from_creation=False):
        from plan.services import permissions
        if destination in service.get_descendants(include_self=True):
            raise FryParadox()

        if destination == service.parent:
            raise SameParent()

        if not requester.staff.user.is_superuser:
            if service.is_base:
                raise PermissionDenied(
                    code='Cannot move base service',
                    title={
                        'ru': 'Нельзя перемещать базовый сервис',
                        'en': 'Cannot move base service'
                    },
                    message={
                        'ru': 'У вас нет прав на перемещение базовых сервисов',
                        'en': 'You do not have permissions to move base services'
                    }
                )
            if destination.is_base_non_leaf():
                raise PermissionDenied(
                    code='Cannot move non base service to base non leaf destination',
                    title={
                        'ru': 'Нельзя перемещать не базовый сервис под базовый не листовой',
                        'en': 'Cannot move non base service to base non leaf destination'
                    },
                    message={
                        'ru': 'У вас нет прав на перемещение не базовых сервисов под базовые не листовые',
                        'en': 'You do not have permissions to move non base service to base non leaf destination'
                    }
                )

        try:
            existing_request = service.move_requests.active().get()
            if existing_request.destination == destination:
                return existing_request

            if permissions.can_cancel_move_request(service, requester):
                existing_request.reject(rejector=requester)
                existing_request.save()
            else:
                raise PermissionDenied()

        except ServiceMoveRequest.DoesNotExist:
            pass

        move_request = cls.objects.create(
            service=service,
            destination=destination,
            source=service.parent,
            requester=requester.staff,
            from_creation=from_creation,
        )

        oebs_related = is_oebs_related(service)
        if oebs_related:
            OEBSAgreement.objects.move_service_create(
                service=service,
                requester=requester.staff,
                move_request=move_request,
            )

        if permissions.can_approve_move_request(service, requester, from_creation=from_creation):
            move_request.approve(approver=requester)
            move_request.refresh_from_db()

        return move_request

    @property
    def requested(self):
        return self.determine_approval_state() == self.REQUESTED

    @property
    def approved(self):
        return self.determine_approval_state() == self.APPROVED

    def _destination_is_sandbox(self):
        return self.destination.slug == settings.ABC_DEFAULT_SERVICE_PARENT_SLUG

    def determine_approval_state(self, approver=None, from_creation=False):
        conditions = (
            self.approver_outgoing,
            self.approver_incoming,
        )
        if all(conditions):
            return self.APPROVED
        if any(conditions):
            return self.PARTIALLY_APPROVED
        else:
            return self.REQUESTED

    def approve(self, approver):

        self._approve_transition(approver, from_creation=self.from_creation)
        self.save()

        if self.approved:

            oebs_agreement = self.get_active_agreement()
            if oebs_agreement:
                oebs_agreement.start_moving_service()
            else:
                from plan.services.tasks import move_service
                move_service.apply_async(
                    args=[self.id], countdown=settings.ABC_DEFAULT_COUNTDOWN
                )
            log.info('Changing readonly_state(MOVING) on %s', self.service.slug)
            self.service.get_descendants(include_self=True).update(
                readonly_state=Service.MOVING,
                readonly_start_time=timezone.now()
            )

    @transition(field=state, source=(ServiceRequestMixin.REQUESTED, ServiceRequestMixin.PARTIALLY_APPROVED),
                target=GET_STATE(determine_approval_state))
    def _approve_transition(self, approver, from_creation=False):
        from plan.services.permissions import is_service_responsible

        manager_of_destination = is_service_responsible(self.destination, approver)
        manager_of_subject = is_service_responsible(self.service, approver)

        if from_creation:  # можно создать сервис под своим без подтверждения
            manager_of_subject = manager_of_subject or manager_of_destination

        if not any((manager_of_subject, manager_of_destination)):
            raise PermissionDenied()

        if manager_of_subject and not self.approver_outgoing:
            self.approver_outgoing = approver.staff
        elif self.service.is_in_sandbox():
            self.approver_outgoing = get_abc_zombik()

        if manager_of_destination and not self.approver_incoming:
            self.approver_incoming = approver.staff
        elif self._destination_is_sandbox():
            self.approver_incoming = get_abc_zombik()

    @transition(field='state', source=ServiceRequestMixin.PROCESSING_OEBS, target=ServiceRequestMixin.FAILED)
    def fail(self):
        from plan.services.tasks import cleanup_service_requests
        cleanup_service_requests.apply_async(countdown=settings.ABC_DEFAULT_COUNTDOWN)
        self.service.readonly_state = None
        self.service.save(update_fields=['readonly_state'])

    def reject(self, rejector=None):
        self._reject_transition(rejector)
        self.save()

        oebs_agreement = self.get_active_agreement()
        if oebs_agreement:
            oebs_agreement.decline(fail_request=False)

        from plan.services.tasks import cleanup_service_requests
        cleanup_service_requests.apply_async(countdown=settings.ABC_DEFAULT_COUNTDOWN)

    @transition(
        field=state,
        source=(ServiceRequestMixin.REQUESTED, ServiceRequestMixin.PARTIALLY_APPROVED),
        target=ServiceRequestMixin.REJECTED
    )
    def _reject_transition(self, rejector=None):
        if rejector is None:
            raise ValidationError('Cannot reject a move request without supplying a rejector')

        from plan.services.permissions import can_cancel_move_request
        if not can_cancel_move_request(self.service, rejector):
            raise PermissionDenied()

    @transition(field=state, source=ServiceRequestMixin.PROCESSED_D, target=ServiceRequestMixin.PROCESSING_IDM)
    def process_idm(self):
        pass

    @transition(field=state, source=ServiceRequestMixin.PROCESSING_IDM, target=ServiceRequestMixin.PROCESSING_ABC)
    def process_abc(self):
        pass

    @transition(field=state, source=ServiceRequestMixin.PROCESSING_ABC, target=ServiceRequestMixin.COMPLETED)
    def complete(self):
        if not self.service.parent == self.destination:
            raise ValueError('Move request {} did not change parent properly!'.format(self.id))
        self.completed_at = timezone.now()

    class Meta(object):
        get_latest_by = 'updated_at'
        verbose_name = _('Перемещение сервиса')
        verbose_name_plural = _('Перемещения сервисов')

    def __str__(self):
        return 'ServiceMoveRequest {id} {old}/{service} -> {new}/{service} by {login}'.format(
            id=self.id,
            service=self.service.slug,
            old=self.source.slug if self.source else '',
            new=self.destination.slug if self.destination else '<root>',
            login=self.requester.login)

    def __repr__(self):
        return self.__str__()


class ServiceMemberQuerySet(models.QuerySet):
    def team(self):
        return self.exclude(models.Q(role__code=Role.RESPONSIBLE))

    def owners(self):
        return self.filter(role__code=Role.EXCLUSIVE_OWNER)

    def heads(self):
        return self.filter(role__code__in=Role.HEADS)

    def responsibles(self):
        return self.filter(role__code__in=Role.RESPONSIBLES)

    def non_head_responsibles(self):
        return self.filter(role__code=Role.RESPONSIBLE)

    def responsible_for_duty(self):
        return self.filter(role__code=Role.RESPONSIBLE_FOR_DUTY)

    def active(self):
        return self.exclude(staff__is_dismissed=True)

    def of_schedule(self, schedule):
        return self.filter(schedule.get_role_q())

    def from_department(self, from_department_id):
        return self.filter(from_department=from_department_id)

    def activate(self):
        self.update(
            state=SERVICEMEMBER_STATE.ACTIVE,
            granted_at=timezone.now(),
            deprived_at=None,
        )

    def set_depriving_state(self):
        self.update(state=SERVICEMEMBER_STATE.DEPRIVING, depriving_at=timezone.now())

    def deprive(self):
        self.update(state=SERVICEMEMBER_STATE.DEPRIVED, deprived_at=timezone.now())


class OnlyCurrentMembersManager(models.Manager):

    def get_queryset(self):
        return super().get_queryset().filter(state__in=(SERVICEMEMBER_STATE.ACTIVE, SERVICEMEMBER_STATE.DEPRIVING))


class ServiceMember(MemberWithState):
    objects = OnlyCurrentMembersManager.from_queryset(ServiceMemberQuerySet)()
    all_states = models.Manager.from_queryset(ServiceMemberQuerySet)()

    FULL = 8
    PARTIAL = 4
    SLIGHTLY = 1
    OCCUPANCY = (
        (FULL, _('полная')),
        (PARTIAL, _('частичная')),
        (SLIGHTLY, _('одним глазом')),
    )

    staff = models.ForeignKey(
        Staff,
        related_name='servicemembers',
        verbose_name=_('Сотрудник'),
    )
    service = models.ForeignKey(
        Service,
        related_name='members',
        verbose_name=_('Сервис')
    )
    role = models.ForeignKey(Role, verbose_name=_('Роль'))
    autorequested = models.BooleanField(default=False, verbose_name=_('Запрошена автоматически'))
    custom_role = models.CharField(
        max_length=255,
        default='',
        blank=True,
        verbose_name=_('Своё название роли'),
    )
    description = models.CharField(
        max_length=255,
        default='',
        blank=True,
        verbose_name=_('Описание'),
    )
    position = models.IntegerField(
        db_index=True,
        editable=False,
        default=0
    )
    is_temp = models.BooleanField(
        default=False,
        verbose_name=_('Временно исполняющий обязанности'),
    )
    occupancy = models.PositiveSmallIntegerField(
        choices=OCCUPANCY,
        default=0,
        blank=True,
        verbose_name=_('Участие'),
    )
    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)
    found_in_staff_at = models.DateTimeField(null=True, db_index=True)

    weight = models.PositiveSmallIntegerField(
        default=0,
        verbose_name=_('Вес в сервисе')
    )

    from_department = models.ForeignKey(
        'ServiceMemberDepartment',
        related_name='members',
        null=True, blank=True,
    )
    resource = models.ForeignKey(
        'resources.Resource',
        related_name='memberships',
        null=True, blank=True, default=None,
    )

    part_rate = models.DecimalField(max_digits=5, decimal_places=4,
                                    null=True, blank=True)

    DENORMALIZED_FIELDS = {
        'weight': {
            'calculator': denormalization.count_service_member_weight,
        }
    }

    @property
    def subject(self):
        return self.staff

    def get_absolute_url(self):
        return self.service.get_absolute_url() + '?team=true'

    def __str__(self):
        return 'Member {0} ({1}@{2} by {3})'.format(
            self.id,
            self.staff.login,
            self.service.slug,
            self.from_department_id if self.from_department else '-',
        )

    def validate_unique(self, exclude=None):
        if self.from_department_id:
            return

        qs = ServiceMember.objects.filter(
            service=self.service_id,
            staff=self.staff_id,
            role=self.role_id,
            from_department=None,
        )

        if not self._state.adding and self.pk is not None:
            qs = qs.exclude(pk=self.pk)

        errors = {}
        message_text = _('Этот сотрудник уже находится в команде сервиса '
                         'в этой роли')
        errors.setdefault('__all__', []).append(str(message_text))

        if qs.exists():
            raise ValidationError(errors)

    class Meta:
        verbose_name = _('Участник сервиса')
        verbose_name_plural = _('Команды сервисов')
        unique_together = ('service', 'staff', 'role', 'from_department', 'resource')

    def request(self, role_id, autorequested=None, expires_at=None, update_fields=('state', 'idm_role_id', 'autorequested', 'expires_at')):
        if autorequested is not None:
            self.autorequested = autorequested
        super().request(role_id, expires_at=expires_at, update_fields=update_fields)

    def deprive(self):
        super().deprive()
        self.send_update_schedule(
            self.service.id,
            self.role.id,
            removed_user_id=self.staff_id,
        )

    def activate(self):
        super().activate()
        self.send_update_schedule(
            self.service.id,
            self.role.id,
            # в данном случае должен передаваться наш внутренний id staff'a, а не его staff_id
            added_user_id=self.staff_id,
        )


class ServiceMemberDepartment(MemberWithState):
    objects = OnlyCurrentMembersManager()

    service = models.ForeignKey(Service, related_name='department_memberships')
    department = models.ForeignKey(Department)
    role = models.ForeignKey(Role, verbose_name=_('Роль'),
                             null=True, blank=True, default=None)

    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)

    resource = models.ForeignKey(
        'resources.Resource',
        related_name='dep_memberships',
        null=True, blank=True, default=None,
    )

    members_count = models.IntegerField(default=0)

    @property
    def subject(self):
        return self.department

    def __str__(self):
        return 'MemberDepartment {0} ({1}@{2})'.format(
            self.id,
            self.department.staff_id,
            self.service.slug,
        )

    def deprive(self):
        super().deprive()
        self.members.deprive()
        self.send_update_schedule(
            self.service_id,
            self.role_id,
            remove_department=True,
        )

    class Meta:
        verbose_name = _('Групповая роль')
        verbose_name_plural = _('Групповые роли')
        unique_together = ('service', 'department', 'role', 'resource')


class ServiceContactQuerySet(models.QuerySet):

    def startrek_contacts(self):
        return self.filter(
            type__code__in=(settings.STARTREK_BUGS_CONTACT, settings.STARTREK_RELEASES_CONTACT)
        )


class ServiceContact(AbstractContactBase):
    objects = ServiceContactQuerySet.as_manager()

    service = models.ForeignKey(Service, related_name='contacts')
    position = models.PositiveIntegerField(default=0, editable=False)

    class Meta:
        verbose_name = _('Контакт сервиса')
        verbose_name_plural = _('Контакты сервисов')


class ServiceSuspiciousReason(TimestampedModel):
    PARENT_IS_BASE_NON_LEAF = 'parent_is_base_non_leaf'
    EMPTY_SERVICE = 'empty_service'
    NO_ONWER = 'no_owner'
    WRONG_OWNER = 'wrong_owner'
    PARENT_IS_SUSPICIOUS = 'parent_is_suspicious'
    FORCED = 'forced'
    REASONS = (
        (
            PARENT_IS_BASE_NON_LEAF,
            _('У не базового сервиса базовый не листовой родитель')
        ),
        (EMPTY_SERVICE, _('У сервиса отсутствует описание, команда и привязанные ресурсы')),
        (NO_ONWER, _('Нет руководителя')),
        (WRONG_OWNER, _('Руководитель в другой иерархии')),
        (PARENT_IS_SUSPICIOUS, _('Один из верхнеуровневых сервисов является подозрительным')),
        (FORCED, _('Руководитель вышестоящего сервиса пометил этот сервис как подозрительный'))
    )

    UNREMOVABLE_REASONS = {
        PARENT_IS_SUSPICIOUS,
        FORCED,
    }

    REASONS_TEMPLATES = {
        PARENT_IS_BASE_NON_LEAF: {
            'ru': 'Разместите свой сервис под сервисом, находящимся внутри Бизнес-Юнита, в котором вы работаете\nhttps://wiki.yandex-team.ru/intranet/abc/faq/#create',
            'en': 'Place your service under the service located inside the Business Unit in which you work\nhttps://wiki.yandex-team.ru/intranet/abc/faq/#create'
        },
        EMPTY_SERVICE: {
            'ru': 'У сервиса отсутствует описание, команда и привязанные ресурсы',
            'en': 'Service has no description, team and resources.'
        },
        PARENT_IS_SUSPICIOUS: {
            'ru': '(({base_url}/services/{{parent}}/ Родительский сервис)) является подозрительным'.format(base_url=settings.ABC_URL),
            'en': '(({base_url}/services/{{parent}}/ Parent service)) is suspicious'.format(base_url=settings.ABC_URL)
        },
        WRONG_OWNER: {
            'ru': 'Руководитель сервиса (кто:{owner}) не находится в одном департаменте с руководителем вышестоящего сервиса (кто:{parent_owner})',
            'en': 'Service owner (кто:{owner}) is not in the same department with the higher service owner (кто:{parent_owner})'
        },
        NO_ONWER: {
            'ru': 'У сервиса нет руководителя',
            'en': 'Service has no owner'
        },
        FORCED: {
            'ru': 'Руководитель вышестоящего сервиса (кто:{parent_owner}) пометил этот сервис как подозрительный',
            'en': 'Service was marked as suspicious by parent service owner (кто:{parent_owner})',
        },
    }

    service = models.ForeignKey(Service, related_name='suspicious_reasons')
    reason = models.CharField(max_length=64, choices=REASONS)
    context = JSONField(blank=True, default=dict)
    marked_by = models.ForeignKey(Staff, null=True, blank=True)
    marking_text = models.TextField(blank=True, default='')

    @property
    def human_text(self):
        return {k: v.format(**self.context) for k, v in self.REASONS_TEMPLATES[self.reason].items()}


class ServiceNotificationQuerySet(models.QuerySet):
    def sent_notifications(self, notification_id):
        return self.filter(notification_id=notification_id, sent_at__isnull=False,)

    def unsent_notifications(self, notification_id):
        return self.filter(
            notification_id=notification_id,
            sent_at=None,
        )

    def without_destination(self):
        return self.filter(models.Q(email=None) | models.Q(email=''), recipient=None)

    def with_destination(self, destination):
        if isinstance(destination, str):
            return self.filter(email=destination)
        else:
            return self.filter(recipient=destination)

    def of_active_service(self):
        return self.filter(service__state__in=Service.states.ACTIVE_STATES)


class ServiceNotificationManager(models.Manager.from_queryset(ServiceNotificationQuerySet)):

    def create_suspicious_notification_with_recipient(self, service, notification_id, send_only_to_direct_responsibles=False):
        try:
            responsibles, _ = service.get_responsibles_for_email(
                include_self=True,
                send_only_to_direct_responsibles=send_only_to_direct_responsibles,
            )
        except ServicesDontHaveActiveResponsibles:
            responsibles = None

        # делаем иттерацию по всем элементам, не используя фильтр,
        # чтобы не делать доп запросов, ускоряя таску
        problem_service_issue_groups = set(
            service_issue.issue.issue_group_id
            for service_issue in service.service_issues.all()
            if (
                service_issue.issue_group is None
                and service_issue.state in ServiceIssueStates.PROBLEM_STATUSES
                and service_issue.issue.issue_group.send_suggest
            )
        )

        service_traffic_status = service.get_service_traffic_status(problem_service_issue_groups)

        seven_days_ago = timezone.now().date() - timezone.timedelta(days=7)
        complaints = service.complaints.filter(created_at__gte=seven_days_ago)
        if complaints.exists():
            complaints_count = {
                'new': complaints.count(),
                'all': service.complaints.count()
            }
        else:
            if not problem_service_issue_groups:
                log.warning('Service %s has no problem_service_issue_groups and complaints', service)
                return

            complaints_count = None

        if service.is_base:
            ServiceNotification.objects.create(
                service=service,
                notification_id=notification_id,
                traffic_status=service_traffic_status,
                complaints_count=complaints_count,
                email=settings.ABC_TEAM_EMAIL,
            )

        if not responsibles:
            log.warning('Service %s has no responsibles and team', service)
            return

        qs = self.filter(
            service=service,
            notification_id=notification_id,
            sent_at=None,
        )

        if responsibles:
            all_notification = []
            responsibles = set(responsibles)
            recipients_with_created_notification = qs.filter(
                recipient__in=responsibles,
            ).values_list('recipient_id', flat=True)

            for recipient in responsibles:
                if recipient.id not in recipients_with_created_notification:
                    notification = ServiceNotification(
                        service=service,
                        notification_id=notification_id,
                        traffic_status=service_traffic_status,
                        complaints_count=complaints_count,
                        recipient=recipient,
                    )

                    all_notification.append(notification)

            self.bulk_create(all_notification)


class ServiceNotification(TimestampedModel):
    objects = ServiceNotificationManager()

    SANDBOX_ANNOUNCEMENT = 'sandbox_annouocement'
    CHANGES_SUSPICION_DIGEST = 'changes_suspicion_digest'
    SUSPICION_DIGEST = 'suspicion_digest'
    DUTY_BEGIN_NOTIFICATION = 'duty_begin_notification'
    DUTY_PROBLEM_NOTIFICATION = 'duty_problem_notification'

    NEW_APPEAL_CREATED = 'new_appeal_created'
    APPEAL_APPROVED = 'appeal_approved'
    APPEAL_REJECTED = 'appeal_rejected'

    NOTIFICATIONS = (
        (SANDBOX_ANNOUNCEMENT, _('Сервис будет перемещён на следующий рабочий день')),
        (CHANGES_SUSPICION_DIGEST, _('Дайджест изменений подозрителньости')),
        (SUSPICION_DIGEST, _('Регулярный дайджест')),
        (DUTY_BEGIN_NOTIFICATION, _('Уведомление о начале дежурства')),
        (DUTY_PROBLEM_NOTIFICATION, _('Уведомление о проблеме в дежурстве')),
        (NEW_APPEAL_CREATED, _('Уведомление о новой апелляции')),
        (APPEAL_APPROVED, _('Уведомление о подтвержденной апелляции')),
        (APPEAL_REJECTED, _('Уведомление об отклоненной апелляции')),
    )

    service = models.ForeignKey(
        Service,
        related_name='service_notification')
    notification_id = models.CharField(max_length=64, choices=NOTIFICATIONS)
    sent_at = models.DateField(
        null=True,
        blank=True,
        verbose_name=_('Дата отправки уведомления')
    )
    recipient = models.ForeignKey(
        Staff,
        null=True,
        blank=True,
        verbose_name=_('Кому отправлено'),
    )
    email = models.CharField(max_length=50, null=True)

    parent_service = models.ForeignKey(
        Service,
        null=True,
        blank=True,
        verbose_name=_('Родительский сервис, руководитель которого является получателем уведомления'),
        related_name='+'
    )

    team_service = models.BooleanField(
        default=False,
        verbose_name=_('Отправляем письмо участнику команды'),
    )

    old_suspicious_date = models.DateField(
        blank=True,
        null=True,
        verbose_name=_('Предыдущая дата подозрительности')
    )

    shift = models.ForeignKey(
        'duty.Shift',
        null=True,
        verbose_name=_('Уведомления о дежурстве'),
        related_name='notifications',
        on_delete=models.CASCADE,
    )
    last_shift = models.ForeignKey(
        'duty.Shift',
        null=True,
        verbose_name=_('Последний шифт в цепочке проблемных'),
        on_delete=models.SET_NULL,
    )

    spliced_shifts_count = models.PositiveIntegerField(
        null=True,
        blank=True,
        verbose_name=_('Количество склееных смен'),
    )

    problem = models.ForeignKey(
        'duty.Problem',
        null=True,
        verbose_name=_('Проблема дежурства'),
        related_name='notifications',
        on_delete=models.CASCADE,
    )

    traffic_status = JSONField(
        blank=True,
        null=True,
        verbose_name=_('Проблемные светофоры'),
    )

    complaints_count = JSONField(
        blank=True,
        null=True,
        verbose_name=_('Количество сообщений о проблемах'),
    )

    days_before_shift = models.PositiveIntegerField(
        null=True,
        blank=True,
        verbose_name=_('За сколько дней до начала дежурства это уведомление'),
    )

    real_days_before_shift = models.PositiveIntegerField(
        null=True,
        blank=True,
        verbose_name=_('Реальное число дней до начала дежурства это уведомление'),
    )

    @property
    def destination(self):
        return self.recipient or self.email
