from plan.services import permissions


def validate_service_is_readonly(data):
    """
    SERVICE_READONLY
    Сервис временно недоступен для изменений

    Проверяем что сервис не в readonly
    """
    if data.service.readonly_state:
        return {'service': data.service.pk}


def validate_everyone_linked_from_only_department(data):
    """
    DEPARTMENT_LINK_CONFLICT
    Подразделение не может быть привязано

    Проверяем, что хакеры не добавили два раза одно и тоже подразделение, либо
    одновременно дочернее и родительское с пересекающимся множеством людей.
    """
    seen_persons_ids = set()
    for member_data in data.new_members_data:
        department = member_data.get('from_department')
        if department is None:
            continue

        person_id = member_data['person'].id
        if person_id in seen_persons_ids:
            # пусть для простоты тут как в других местах список
            return {'departments': [department.id]}
        else:
            seen_persons_ids.add(person_id)


def validate_departments_already_linked(data):
    """
    DEPARTMENT_LINK_CONFLICT
    Подразделение уже привязано

    Проверяем, что среди привязываемых нет уже привязанных.
    """
    duplicates = set(data.new_departments) & set(data.present_departments)
    if duplicates:
        return {'departments': list(duplicates)}


def validate_departments_relations(data):
    """
    DEPARTMENT_LINK_CONFLICT
    Нельзя привязывать дочернее подразделение, если привязано родительское

    Также проверяем эти отношения между добавляемыми подразделениями.
    """
    new, present = data.new_departments, data.present_departments
    all_ids = set(new) & set(present)

    # TODO: эффективнее получить всех родителей сразу
    for department in (value[0] for value in new.values()):
        ancestors_ids = department.get_ancestors().values_list('id', flat=True)
        if set(ancestors_ids) & all_ids:
            return {'departments': [department.id]}


def validate_membership_uniqueness(data):
    """
    MEMBERSHIP_DUPLICATE_CONFLICT
    Нельзя участвовать в одной роли несколько раз
    """
    person_id = data.member.staff_id
    role_id = data.member_data['role'].id

    present_person_roles = [
        membership.role_id
        for membership in data.present_members
        if membership.staff_id == person_id
        and membership.id != data.member.id
    ]

    if role_id in present_person_roles:
        return {
            'member': {
                'person': person_id,
                'role': role_id,
            }
        }


def validate_head_edit_access(data):
    """
    INSUFFICIENT_PRIVILEGES
    Недостаточно прав для данного действия
    """
    # https://st.yandex-team.ru/PLAN-3409
    is_head_affected = (
        # редактируем кого-то в роли «руководитель сервиса»
        data.member.role.is_owner or
        # редактируем кого-то с установкой роли «руководитель сервиса»
        data.member_data['role'].is_owner
    )

    if not is_head_affected:
        return

    if permissions.is_service_responsible(data.service, data.sender):
        return

    return {
        'member': {
            'person': data.member.staff_id,
            'role': data.member_data['role'].id,
        }
    }


def validate_head_delete_access(data):
    """
    INSUFFICIENT_PRIVILEGES
    Недостаточно прав для данного действия
    """
    # https://st.yandex-team.ru/PLAN-3409
    is_head_affected = data.member.role.is_exclusive_owner

    if not is_head_affected:
        return

    # Руководителя удалять нельзя
    return {
        'member': {
            'person': data.member.staff_id,
            'role': data.member.role.id,
        }
    }


def validate_non_group_role(data):
    """
    DEPARTMENT_MEMBERSHIP_CONFLICT
    Нельзя редактировать роль, выданную на подразделение
    """
    if data.member.from_department:
        return {
            'member': {
                'person': data.member.staff_id,
                'role': data.member_data['role'].id,
            }
        }


def validate_department_roles_are_same(data):
    """
    DEPARTMENT_ROLE_ERROR
    При выдаче ролей на группу все роли членов этой группы должны быть одинаковыми
    """

    role_by_department = {}

    for new_member in data.new_members:
        if not new_member.from_department:
            continue

        if new_member.from_department.id in role_by_department and \
                role_by_department[new_member.from_department.id] != new_member.role.id:
            return {
                'department': new_member.from_department.id,
            }

        role_by_department[new_member.from_department.id] = new_member.role.id


def validate_expiry_params_mutually_exclusive(data):
    """
    EXCLUSIVE_PARAMS_ERROR
    Параметры истечения срока действия ролей взаимоисключающие
    """
    for new_member in data.new_members:
        deprive_after = new_member.get('deprive_after')
        deprive_at = new_member.get('deprive_at')

        if deprive_after and deprive_at:
            return {
                'member': {
                    'deprive_after': 'Exclusive with deprive_at',
                    'deprive_at': 'Exclusive with deprive_after'
                }
            }
