from typing import Iterable, Union, Dict

from plan.common.person import Person
from plan.roles.models import Role
from plan.services.models import ServiceMember, ServiceSuspiciousReason, Service

SERVICE_MODERATOR_PERM = 'internal_roles.moderate'
SERVICE_HR_PERM = 'internal_roles.change_importance'
SERVICE_RESPONSIBLE_PERM = 'internal_roles.manage_responsible'


def is_hr(person):
    return person.user.has_perm(SERVICE_HR_PERM)


def is_service_moderator(person):
    """Суперюзеры ABC."""
    return person.user.is_superuser or person.user.has_perm(SERVICE_MODERATOR_PERM)


def is_service_support(person):
    """Саппорты, имеющие право управлять"""
    return person.user.has_perm(SERVICE_RESPONSIBLE_PERM)


def _maybe_service_responsible(service: Service, person: Person) -> Union[bool, None]:
    if is_service_moderator(person):
        return True
    elif service is None:
        return False
    else:
        return None


def is_service_responsible(service: Service, person: Person, include_self: bool = True) -> bool:
    """
    Управляющие сервиса или его родителей.
    """
    maybe_responsible = _maybe_service_responsible(service, person)
    if maybe_responsible is not None:
        return maybe_responsible
    else:
        if service.parent is not None and not include_self:
            return person.is_responsible(service.parent)
        return person.is_responsible(service)


def is_service_responsible_bulk(services: Iterable[Service], person: Person) -> Dict[Service, bool]:
    result = {
        service: _maybe_service_responsible(service, person)
        for service in services
    }
    maybe_responsible = {
        service
        for service in result
        if result[service] is None
    }
    services = (
        Service.objects
        .filter(id__in=[service.id for service in maybe_responsible])
        .with_responsible(person, with_descendants=True)
    )
    result.update(
        {
            service: service in services
            for service in maybe_responsible
        }
    )
    return result


def can_delete_service(service, person):
    if person.user.is_superuser:
        return True
    elif service.is_base:
        return False
    else:
        return is_service_responsible(service, person)


def can_close_service(service, person):
    return not service.is_base or person.user.is_superuser


def is_team_member(service, person):

    if is_service_moderator(person):
        return True

    return person.is_member_of_team(service, with_ancestors=False)


def is_owner_of_metaservice(service, person):
    '''Самые важные дяди на свете.'''
    query = ServiceMember.objects.responsibles().filter(
        service__in=service.get_ancestors(depth=0, include_self=True),
        staff=person.staff,
    )
    return query.exists() or is_service_moderator(person)


def can_approve_move_request(service, person, from_creation=False):
    '''
    Аппрувить могут управляющие переносимого сервиса и нового родителя.
    Перенос метасервиса могут заапрувить только модераторы сервисов.
    '''
    from plan.services.models import ServiceMoveRequest

    try:
        move_request = service.move_requests.active().get()
    except ServiceMoveRequest.DoesNotExist:
        return False

    # Если перенос уже начался, подтверждать его уже не надо
    if move_request.approver_outgoing and move_request.approver_incoming:
        return False

    if move_request.service.level > 0 and move_request.destination:
        can_be_incoming_approver = is_service_responsible(move_request.destination, person)
        can_be_outgoing_approver = is_service_responsible(move_request.service, person)

        if from_creation:  # можно создать сервис под своим без подтверждения
            can_be_outgoing_approver = can_be_outgoing_approver or can_be_incoming_approver

        if can_be_outgoing_approver and not move_request.approver_outgoing:
            return True
        elif can_be_incoming_approver and not move_request.approver_incoming:
            return True
        else:
            return False
    else:
        # Перенос в корень или из корня
        return is_service_moderator(person)


def can_cancel_move_request(service, person):
    '''Отменить перенос сервиса может его автор или аппрувящие.'''
    from plan.services.models import ServiceMoveRequest

    try:
        move_request = service.move_requests.active().get()
    except ServiceMoveRequest.DoesNotExist:
        return False

    # Если никто не подтвердил, удалить может кто угодно
    if move_request.requested:
        return True

    # Если перенос уже начался, отменять нельзя
    if move_request.approved:
        return False

    return person.staff == move_request.requester or can_approve_move_request(service, person)


def can_approve_create_request(service: Service, person: Person) -> bool:
    """
    Содание сервиса под другим должен заапрувить новый родитель.
    Создание в песочнице без апрува.
    """
    from plan.services.models import ServiceCreateRequest

    try:
        create_request = service.create_requests.requested().get()
    except ServiceCreateRequest.DoesNotExist:
        return False

    # Если запрос уже подтвержден:
    if create_request.approver_incoming:
        return False

    can_be_incoming_approver = is_service_responsible(create_request.move_to, person)
    return can_be_incoming_approver


def can_add_forced_suspicious_reason(service, person):
    return is_service_responsible(service, person, include_self=False)


def can_remove_forced_suspicious_reason(service, person):
    return can_add_forced_suspicious_reason(service, person)


def can_mark_as_not_suspicious(service, person):
    if not service.suspicious_reasons.exclude(reason__in=ServiceSuspiciousReason.UNREMOVABLE_REASONS).filter(marked_by=None).exists():
        return False

    include_self = True
    if service.suspicious_reasons.filter(reason=ServiceSuspiciousReason.FORCED):
        include_self = False

    return is_service_responsible(service, person, include_self=include_self)


def _maybe_can_remove_member(service_member: ServiceMember, person: Person) -> Union[bool, None]:
    # хозяин сервиса не может себя из него выпилить
    if service_member.role.code == Role.EXCLUSIVE_OWNER:
        return False

    # групповую роль нельзя индивидуально отозвать
    if service_member.from_department:
        return False

    # сам с собой делай чо хошь
    if service_member.staff == person.staff:
        return True

    return None


def can_remove_member(service_member: ServiceMember, person: Person) -> bool:
    maybe_can_remove = _maybe_can_remove_member(service_member, person)
    return is_service_responsible(service_member.service, person) if maybe_can_remove is None else maybe_can_remove


def can_remove_member_bulk(service_members: Iterable[ServiceMember], person: Person) -> Dict[ServiceMember, bool]:
    result = {
        member: _maybe_can_remove_member(member, person)
        for member in service_members
    }
    maybe_can_remove = {
        member
        for member in result
        if result[member] is None
    }
    responsible_in_service = is_service_responsible_bulk((sm.service for sm in maybe_can_remove), person)
    result.update(
        {
            member: responsible_in_service[member.service]
            for member in maybe_can_remove
        }
    )
    return result


def can_edit_membership_inheritance(service, person):
    """
    Может снимать управляющий самого сервиса или сервиса выше
    """
    return (
        is_service_responsible(service, person, include_self=False)
        or is_service_responsible(service, person, include_self=True)
    )


def can_edit_duty_settings(service, person):
    return person.can_modify_duty_in_service(service)


def can_approve_appeal(service, person):
    """
    Аппрувить апелляции по проблемам сервиса могут
    * либо управляющие вышестоящих сервисов,
    * либо суперпользователи.
    """
    if service.parent:
        return person.is_responsible(service.parent) or person.staff.user.is_superuser

    return person.staff.user.is_superuser
