from collections import defaultdict
from datetime import timedelta
from itertools import groupby
import functools
import logging
from time import sleep
from typing import Tuple, Union, Optional
from requests.exceptions import RequestException

import waffle

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.auth.models import Permission
from django.contrib.postgres.aggregates import ArrayAgg
from django.db import transaction, OperationalError
from django.db import models
from django.db.models import Q, Value
from django.db.models.functions import Concat
from django.utils import timezone

import plan.holidays.calendar
from plan.api.exceptions import IntegrationError, ServiceTagsDuplicatingError
from plan.api.idm import actions
from plan.celery_app import app
from plan.common.person import Person
from plan.common.utils import startrek
from plan.common.utils import timezone as utils
from plan.common.utils.http import Session
from plan.common.utils.oauth import get_abc_zombik
from plan.common.utils.tasks import lock_task
from plan.denormalization.check import check_obj_with_denormalized_fields
from plan.idm import nodes
from plan.idm.exceptions import IDMError
from plan.idm.roles import get_service_group_roles_count
from plan.notify.shortcuts import deliver_email
from plan.notify.shortcuts import send_to_team
from plan.puncher.rules import PuncherClient
from plan.roles.models import Role
from plan.services.constants import permissions as perm_constants
from plan.services.exceptions import ServicesDontHaveActiveResponsibles
from plan.resources.models import ResourceType, ServiceResource
from plan.services.functionality import get_service_functions
from plan.services.models import (
    Service,
    ServiceMoveRequest,
    ServiceMember,
    ServiceMemberDepartment,
    ServiceCreateRequest,
    ServiceDeleteRequest,
    ServiceCloseRequest,
    ServiceSuspiciousReason,
    ServiceNotification,
    ServiceTypeFunction,
    ServiceType,
)
from plan.duty.models import Schedule
from plan.staff.models import ServiceScope, Staff
from plan.staff.tasks import MembershipStaffImporter
from plan.api.idm.actions import set_review_policy_to_service, check_review_policy_of_service_roles

log = logging.getLogger(__name__)

HANDLED_NEWHIRES_EXPIRATION = 14  # дней


def log_task(func):
    @functools.wraps(func)
    def wrapper(*args, **kw):
        logger = logging.getLogger(__name__)
        logger.info('Start task %s with args %r', func.__name__, args)

        try:
            result = func(*args, **kw)
            logger.info('Finish task %s with args %r', func.__name__, args)
            return result

        except Exception as e:
            log.exception('Exception during task %s with args %r: %s', func.__name__, args, e)
            raise

    return wrapper


@app.task(bind=True, autoretry_for=[IDMError, OperationalError])
@log_task
def register_service(self, create_request_id):
    request = ServiceCreateRequest.objects.get(id=create_request_id)
    log.info('Processing request %s', request)

    if request.state == ServiceCreateRequest.REQUESTED or request.state == ServiceCreateRequest.APPROVED:
        request.process_idm()
        request.save()
    elif request.state != ServiceCreateRequest.PROCESSING_IDM:
        log.error(
            "Attempted to create a service that's already been processed: create request %s",
            create_request_id
        )
        return

    try:
        actions.add_service(request.service)

    except IDMError as exc:
        self.retry(exc=exc, args=(create_request_id,))

    except IntegrationError:
        try:
            service_node = nodes.get_service_node(request.service)
            if not service_node.exists():
                raise

            roles_node = nodes.get_service_roles_node(request.service)
            if not roles_node.exists():
                roles_node.register()

            owner_role_node = nodes.get_role_node(request.service, Role.get_exclusive_owner())
            if not owner_role_node.exists():
                owner_role_node.register()

        except IDMError as exc:
            self.retry(exc=exc, args=(create_request_id,))

    request.process_head()
    request.save()
    request_service_head.apply_async(
        args=(create_request_id,),
        countdown=settings.ABC_DEFAULT_COUNTDOWN
    )


@app.task(bind=True, autoretry_for=[IDMError])
@log_task
def request_service_head(self, create_request_id):
    request = ServiceCreateRequest.objects.select_related('service__owner', 'requester').get(id=create_request_id)

    log.info('Requesting role for service head from request %s', request)

    if not request.service.members.owners().exists() and request.service.owner is not None:
        try:
            actions.add_service_head(request.service, request.service.owner, requester=request.requester)
        except IDMError as exc:
            self.retry(exc=exc, args=(create_request_id,))

    request.process_abc()
    request.save()
    finalize_service_creation.apply_async(
        args=[create_request_id],
        countdown=settings.ABC_DEFAULT_COUNTDOWN
    )


@app.task(bind=True, autoretry_for=[OperationalError])
@transaction.atomic()
def finalize_service_creation(self, create_request_id):
    request = ServiceCreateRequest.objects.get(id=create_request_id)

    update_service_fields.apply_async(
        args=(request.service.id,),
        countdown=settings.ABC_DEFAULT_COUNTDOWN
    )

    log.info('Completed request %s', request)
    request.complete()
    request.save()

    log.info('Changing readonly_state(None) on service %s', request.service.slug)
    request.service.readonly_state = None
    request.service.save(update_fields=('readonly_state',))

    if request.move_to is not None:
        if request.service.parent.slug == settings.ABC_DEFAULT_SERVICE_PARENT_SLUG:
            if request.move_to != request.service.parent:
                ServiceMoveRequest.request(
                    service=request.service,
                    destination=request.move_to,
                    requester=Person(request.requester),
                    from_creation=True
                )
                log.info(
                    'Service %s was requested to be moved to %s after creation by %s',
                    request.service.slug,
                    request.move_to.slug,
                    request.requester.login
                )
        else:
            notify_staff.apply_async(
                args=[request.service_id],
                countdown=settings.ABC_DEFAULT_COUNTDOWN,
            )


@app.task(bind=True, max_retries=5)
def move_service(self, move_request_id):
    """Запросить перемещение сервиса на стороне IDM."""
    move_request = ServiceMoveRequest.objects.get(id=move_request_id)

    move_request.process_oebs()
    move_request.process_d()
    move_request.save()

    log.info('Performing %s', move_request)

    if move_request.state != ServiceMoveRequest.PROCESSING_IDM:
        move_request.process_idm()
        move_request.save()

    # переносим дерево в idm
    try:
        actions.move_service(move_request.service, move_request.destination)
    except IDMError as exc:
        raise self.retry(exc=exc, args=(move_request_id,))

    move_request.start_moving_idm = True
    move_request.save(update_fields=('start_moving_idm', ))

    verify_and_move_service.delay(move_request_id)


@app.task(
    autoretry_for=(IDMError,),
    retry_backoff=True,
    retry_kwargs={'max_retries': settings.ABC_IDM_MAX_RETRY},
)
def verify_and_move_service(move_request_id):
    """Проверить перенос сервиса в IDM и приступить к переносу в ABC."""
    move_request = ServiceMoveRequest.objects.get(id=move_request_id)

    actions.assert_service_node_exists(move_request.service, move_request.destination)

    move_request.process_abc()
    move_request.save()
    move_service_abc_side.apply_async(args=[move_request_id])


def update_gradient_fields(service: Service, fields: dict, update_services_dict: dict, error_found: bool) -> dict:
    update_services = service.update_gradient_fields(gradient_fields=fields)
    error_dict = {f'error_{key}': error_found for key in fields}
    fields.update(error_dict)
    for s in update_services:
        update_services_dict[s.slug] = fields
    return update_services_dict


def get_update_fields(service: Service,
                      update_services_dict: dict,
                      update_umb: bool = False) -> Tuple[Union[None, dict], bool]:
    """
    valuestream пытаемся обновить всегда, зонтик - по запросу
    """

    fields = {}
    error_found = False
    if service.slug in update_services_dict.keys():
        if 'valuestream' in update_services_dict[service.slug]:
            valuestream = update_services_dict[service.slug]['valuestream']
            if valuestream is not None or update_services_dict[service.slug]['error_valuestream']:
                # если не None или зафиксирована ошибка в структуре, значит испольузем указаный vs
                fields['valuestream'] = valuestream

        if 'umbrella' in update_services_dict[service.slug] and update_umb:
            umbrella = update_services_dict[service.slug]['umbrella']
            if umbrella is not None or update_services_dict[service.slug]['error_valuestream']:
                # если не None или зафиксирована ошибка в структуре, значит испольузем указаный vs
                fields['umbrella'] = umbrella

    has_vs = 'valuestream' in fields
    has_umb = 'umbrella' in fields
    if has_vs and (has_umb or not update_umb):
        # значит, только что всё обновили и больше не меняем
        return None, error_found

    else:
        if not has_vs:
            try:
                fields['valuestream'] = service.search_gradient_parent(settings.GRADIENT_VS)
            except ServiceTagsDuplicatingError:
                fields['valuestream'] = None
                error_found = True

        if not has_umb and update_umb:
            try:
                fields['umbrella'] = service.search_gradient_parent(settings.GRADIENT_UMB)
            except ServiceTagsDuplicatingError:
                fields['umbrella'] = None
                error_found = True

    return fields, error_found


@lock_task(
    lock_key=lambda service_id='', *args, **kwargs: 'calculate_gradient_fields_{}'.format(service_id)
)
def calculate_gradient_fields(service_id: int, include_descendants: bool = True):
    if include_descendants:
        services = Service.objects.get(pk=service_id).get_descendants(include_self=True).prefetch_related('tags').order_by('level')
    else:
        services = Service.objects.filter(pk=service_id).prefetch_related('tags').order_by('level')

    update_services_dict = dict()
    for service in services:
        service_gradient_tags = [tag for tag in service.tags.all() if tag.slug in settings.GRADIENT_TAGS]
        fields = {}
        error_found = False
        if len(service_gradient_tags) > 1:
            # если у текущего больше одного градиентного тега, то
            # мы не можем понять, что это за сервис и зануляем всю ветку с дичью
            fields = {'valuestream': None, 'umbrella': None}
            error_found = True

        elif len(service_gradient_tags) == 0 or service_gradient_tags[0].slug == settings.GRADIENT_CONTOUR:
            # если текущий является контуром или наоборот удалены все градиентные теги
            # нужно найти соотвествующих родителей vs и зонтик
            fields, error_found = get_update_fields(service, update_services_dict, update_umb=True)
            if not fields:
                continue

        elif service_gradient_tags[0].slug == settings.GRADIENT_VS:
            # если текущий является vs, проставим ему и всем детям vs
            # приоритетней тот vs, что выше
            fields, error_found = get_update_fields(service, update_services_dict)
            if not fields:
                continue

            if fields['valuestream'] is None:
                fields['valuestream'] = service

        elif service_gradient_tags[0].slug == settings.GRADIENT_UMB:
            # если текущий является зонтиком, проверим valuestream и проставим всем детям зонтик
            # выше может быть тоже зонтик, считаем его важней
            fields, error_found = get_update_fields(service, update_services_dict, update_umb=True)
            if not fields:
                continue

            if fields['umbrella'] is None:
                fields['umbrella'] = service

        update_services_dict = update_gradient_fields(service, fields, update_services_dict, error_found)


@transaction.atomic
def finish_service_move_request(move_request):
    service = move_request.service
    destination = move_request.destination
    # Сохраним старых ответственных для отправки письма
    try:
        responsible, parent_service = service.get_responsibles_for_email(include_self=True)
    except ServicesDontHaveActiveResponsibles:
        responsible = None
        parent_service = None

    # снимаем у сервиса наследование
    service.membership_inheritance = False

    # обновляем родителя сервиса
    service.parent = destination
    service.save(update_fields=('parent', 'level', 'membership_inheritance'))

    going_to_meta_other = (
        destination is not None and
        destination.get_ancestors(include_self=True).filter(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG).exists()
    )

    log.info('Set is_exportable=%s for %s after moving to %s', not going_to_meta_other, service, destination)
    # снимаем readonly и флипаем is_exportable у всей ветки
    moving_services = service.get_descendants(include_self=True)

    services_slugs = ','.join(moving_services.values_list('slug', flat=True))
    log.info('Changing readonly_state(None) for %s', services_slugs)
    moving_services.update(
        readonly_start_time=None,
        readonly_state=None,
        is_exportable=not going_to_meta_other,
        suspicious_date=None,
        suspicious_notification_date=None,
        sandbox_move_date=None,
    )

    ServiceSuspiciousReason.objects.filter(service__in=moving_services).delete()

    # запускает пересчет градиентных полей
    calculate_gradient_fields.delay(service.id)

    # если перенос был не в Песочницу,
    # то удалим нотификации о скором переносе, тк может понадобиться отправить вновь
    # а иначе удалим только неотправленные
    if not going_to_meta_other:
        ServiceNotification.objects.filter(
            service__in=moving_services,
            notification_id=ServiceNotification.SANDBOX_ANNOUNCEMENT,
        ).delete()

    else:
        ServiceNotification.objects.filter(
            service__in=moving_services,
            notification_id=ServiceNotification.SANDBOX_ANNOUNCEMENT,
            sent_at__isnull=True,
        ).delete()

    # нужно обновить эти поля у инстанса в памяти тоже,
    # т.к. он дальше закоммитится вместе с его родителем в move_request
    service.refresh_from_db()

    # чистим все активные мувреквесты у сервиса и его детей
    service.move_requests.reject()
    ServiceMoveRequest.objects.filter(destination__in=moving_services).reject()

    move_request.complete()
    move_request.save()
    if not move_request.from_creation:
        if responsible:
            deliver_email(
                'notifications.services.service_was_moved',
                context={'service': service, 'destination': destination, 'parent_service': parent_service},
                recipients=responsible,
            )
        else:
            log.error(
                'Try send email about move service with pk:%s, but recipients does not exist',
                service.pk,
            )

    log.info('Done %s', move_request)

    # если у сервиса висели не завершенные запросы ролей,
    # переделываем эти запросы на новые роли, получившиеся после переезда
    rerequest_roles.apply_async(args=[service.pk], countdown=settings.ABC_DEFAULT_COUNTDOWN)

    # пересчитываем денормализованные поля
    update_service_fields.apply_async(args=[service.id], countdown=settings.ABC_DEFAULT_COUNTDOWN)

    # сообщаем стаффу, что ему нужно обновить данные изменившегося дерева сервисов
    services_to_notify_about = set(moving_services.values_list('id', flat=True))

    if move_request.source:
        for parent in move_request.source.get_ancestors(include_self=True):
            services_to_notify_about.add(parent.id)

    if move_request.destination:
        for parent in move_request.destination.get_ancestors(include_self=True):
            services_to_notify_about.add(parent.id)

    for service_id in services_to_notify_about:
        notify_staff.apply_async(
            args=[service_id],
            countdown=settings.ABC_DEFAULT_COUNTDOWN,
        )


@lock_task
def finish_all_service_move_requests():
    for request in ServiceMoveRequest.objects.processing_abc():
        finish_service_move_request(request)


@app.task(bind=True, autoretry_for=(OperationalError,))
def move_service_abc_side(self, move_request_id):
    if not waffle.switch_is_active('async_move_service_abc_side'):
        return finish_all_service_move_requests()
    move_request = ServiceMoveRequest.objects.get(id=move_request_id)
    finish_service_move_request(move_request)


def move_to_root(service, person):
    ServiceMoveRequest.request(
        service=service,
        destination=None,
        requester=person,
    )


@lock_task(
    lock_key=lambda service_id='', *args, **kwargs: 'update_service_department_members_{}'.format(service_id)
)
def update_service_department_members(service_id=None):
    log.info(f'Start update_service_department_members with service_id={service_id}')
    department_memberships = ServiceMemberDepartment.objects.filter(service__state__in=Service.states.ALIVE_STATES)

    if service_id is not None:
        department_memberships = department_memberships.filter(service_id=service_id)

    set_services = set()
    for department_membership in department_memberships:
        departments = (
            department_membership.department
            .get_descendants(include_self=True)
            .values_list('id', flat=True)
        )

        current_service_members = {member.staff for member in department_membership.members.all()}
        department_members = set(
            Staff.objects
            .filter(is_dismissed=False, department__id__in=departments)
        )

        to_delete = current_service_members - department_members
        department_membership.members.filter(staff__in=to_delete).deprive()

        to_add = department_members - current_service_members
        for person in to_add:
            member, _ = ServiceMember.all_states.get_or_create(
                service=department_membership.service,
                staff=person,
                role=department_membership.role,
                from_department=department_membership,
                expires_at=department_membership.expires_at,
            )
            member.activate()

        if not to_delete or not to_add:
            log.info(f'update_service_department_members: deleted or added ServiceMember for {department_membership}')
            set_services.add(department_membership.service.id)

        if department_membership.members_count != len(department_members):
            department_membership.members_count = len(department_members)
            department_membership.save(update_fields=('members_count',))

    # не все таски ставим одновременно
    # мониторинг загорится WARN, если в очереди будет 200 тасок, в CRIT от 500
    # всего около 5000 актвиных сервисов
    offset_notify_staff.delay(list(set_services))


@app.task
def offset_notify_staff(service_ids, offset=0, limit=100):
    if settings.CROWDTEST:
        return
    # notify_staff выполняется примерно за 300ms
    # стандартный - 5с, берем 10
    countdown = settings.ABC_DEFAULT_COUNTDOWN * 2

    new_offset = offset + limit
    for service_id in service_ids[offset:new_offset]:
        notify_staff.delay(service_id)

    if new_offset < len(service_ids):
        offset_notify_staff.apply_async(
            args=[service_ids, new_offset, limit],
            countdown=countdown
        )


@app.task
def update_service_fields(service_id):
    service = Service.objects.get(id=service_id)
    for node in service.get_ancestors(include_self=True):
        check_obj_with_denormalized_fields(node, Service.DENORMALIZED_FIELDS, fix=True)
    for node in service.get_descendants():
        check_obj_with_denormalized_fields(node, Service.DENORMALIZED_FIELDS, fix=True)


@lock_task
def cleanup_service_requests():
    # на всякий случай ищем неудалённые запосы с одним и тем же сервисом
    # и оставляем только самый последний

    repeating_services = (
        ServiceMoveRequest.objects.active()
        .values('service_id')
        .annotate(cnt=models.Count('service_id'))
        .filter(cnt__gt=1)
    )

    for value in repeating_services:
        move_requests = (
            ServiceMoveRequest.objects.active()
            .filter(service_id=value['service_id'])
            .order_by('-updated_at')
        )
        latest_pk = move_requests.first().pk
        move_requests.exclude(pk=latest_pk).reject()


@app.task
@log_task
def rerequest_roles(service_id):
    service = Service.objects.get(pk=service_id)

    actions.rerequest_requested(service)


@app.task
@log_task
def drop_requests(service_id):
    """
        Закрыть все висящие запросы
    """
    service = Service.objects.get(pk=service_id)

    actions.delete_requests(service)


@app.task
@log_task
def drop_chown_requests(service_id, current_owner_login):
    """
        Закрыть все висящие запросы на смену руководителя проекта
    """
    service = Service.objects.get(pk=service_id)

    actions.delete_chown_requests(service, current_owner_login)


@app.task(bind=True, max_retries=5)
@log_task
def delete_service(self, service_delete_request_id):
    """
        Удалить сервис
    """
    request = ServiceDeleteRequest.objects.select_related('service').get(pk=service_delete_request_id)
    if request.service.has_moving_descendants():
        log.error('Cannot delete service %s, it has moving descendants', request.service_id)
        # Попробуем ещё раз через 5 минут
        raise self.retry(countdown=5*60, args=(service_delete_request_id,))
    request.process_oebs()
    unclosed_services = request.is_closable_from_d_side()
    if unclosed_services:
        log.info(
            f'Cannot delete service {request.service_id}, cause his subservices couldn’t be closed. '
            f'The following services are not closable from dispenser: {unclosed_services}'
        )
        return
    request.process_d()
    request.process_idm()
    request.save()

    log.info('Processing request %s in idm', request)
    # удаляем узлы сервиса из дерева ролей в idm
    try:
        actions.delete_service(request.service)
    except IDMError as exc:
        raise self.retry(exc=exc, args=(service_delete_request_id,))

    # дропаем все запросы ролей в этом сервисе
    drop_requests.delay(request.service.id)

    request.process_abc()
    request.save()
    delete_service_abc_side.delay(request.id)


@app.task(autoretry_for=(OperationalError,))
@log_task
@transaction.atomic
def close_service_abc_side(service_close_request_id):
    request = ServiceCloseRequest.objects.get(pk=service_close_request_id)

    log.info('Processing request %s in abc', request)

    # переводим сервис и его детей в статус "закрыт"
    closing_services = request.service.get_descendants(include_self=True)
    log.info('Changing readonly_state(None) on %s', list(closing_services.values_list('slug', flat=True)))
    closing_services.update(
        readonly_start_time=None,
        readonly_state=None,
        state=Service.states.CLOSED,
        suspicious_date=None,
        suspicious_notification_date=None,
        membership_inheritance=False,
        use_for_hardware=False,
        use_for_hr=False,
        use_for_procurement=False,
        use_for_revenue=False,
    )

    # удаляем все связанные дежурства
    Schedule.objects.filter(service__in=closing_services).safe_delete()

    for service in closing_services:
        notify_staff.delay(service.id)

    update_service_fields.apply_async(
        args=[request.service.id],
        countdown=settings.ABC_DEFAULT_COUNTDOWN
    )

    log.info('Completed request %s', request)
    request.complete()
    request.save()


@app.task(autoretry_for=(OperationalError,))
@log_task
@transaction.atomic
def delete_service_abc_side(service_delete_request_id):
    request = ServiceDeleteRequest.objects.get(pk=service_delete_request_id)

    log.info('Processing request %s in abc', request)

    # переводим сервис и его детей в статус "удален"
    deleting_services = request.service.get_descendants(include_self=True)
    log.info('Changing readonly_state(None) on %s', list(deleting_services.values_list('slug', flat=True)))
    deleting_services.update(
        readonly_start_time=None,
        readonly_state=None,
        state=Service.states.DELETED,
        suspicious_date=None,
        suspicious_notification_date=None,
        membership_inheritance=False,
        staff_id=None,
        use_for_hardware=False,
        use_for_hr=False,
        use_for_procurement=False,
        use_for_revenue=False,
    )

    # удаляем все связанные дежурства
    Schedule.objects.filter(service__in=deleting_services).safe_delete()

    # отклоняем все мувреквесты сервиса и его детей
    ServiceMoveRequest.objects.active().filter(
        Q(destination__in=deleting_services) | Q(service__in=deleting_services)
    ).reject()

    ServiceSuspiciousReason.objects.filter(service__in=deleting_services).delete()

    ServiceScope.objects.filter(service__in=deleting_services).update(staff_id=None)

    # пингуем стафф, чтоб обновил свои группы
    for service in deleting_services:
        notify_staff.delay(service.id)

    update_service_fields.apply_async(
        args=[request.service.id],
        countdown=settings.ABC_DEFAULT_COUNTDOWN
    )

    log.info('Completed request %s', request)
    request.complete()
    request.save()


@lock_task
@log_task
def sync_owners(fake=False):
    """
        Сверяет владельцев и руководителей сервисов
    """

    def set_service_owner(service, owner):
        log.info('Set %s owner to %s, fake=%s', service, owner, fake)
        if not fake:
            service.owner = owner
            service.save(update_fields=['owner'])

    log.info('Sync owners')

    head_role = Role.get_exclusive_owner()

    stat = {'double': [], 'set_head': [], 'set_owner': [], 'different': [], 'empty': [], 'dismissed': []}
    comments = {
        'double': 'Сервисы с больше чем одним руководителем',
        'set_head': 'Сервисы без руководителя (выставлен)',
        'set_owner': 'Сервисы без владельца (выставлен)',
        'different': 'Сервисы с отличающимися владельцем и руководителем',
        'empty': 'Сервисы без владельца и руководителя',
        'dismissed': 'Сервисы с уволенным владельцем',
    }

    services = Service.objects.active().select_related('owner')
    for service in services:
        heads = ServiceMember.objects.filter(
            service=service,
            role=head_role,
        )

        if len(heads) > 1:
            log.warning('Service %s has more than one head', service.slug)
            stat['double'].append(service)
            if not fake:
                real_head = service.owner
                for membership in heads:
                    if membership.staff != real_head:
                        actions.deprive_role(membership, comment='Неактуальный руководитель')
                set_service_owner(service, heads[0].staff)
        elif service.owner:
            if len(heads) == 0:
                if service.owner.is_dismissed:
                    log.warning('Service %s has dismissed owner', service.slug)
                    stat['dismissed'].append(service)
                    set_service_owner(service, None)
                else:
                    log.warning('Service %s has owner but no head', service.slug)
                    stat['set_head'].append(service)
                    if not fake:
                        actions.request_membership(
                            service, service.owner, head_role, comment='Синхронизация владельцев.'
                        )
            elif service.owner != heads[0].staff:
                service.owner = heads[0].staff
                log.warning('Service %s has different owner and head', service.slug)
                stat['different'].append(service)
                set_service_owner(service, heads[0].staff)
        elif service.is_active:
            if len(heads) == 0:
                log.warning('Service %s has no owner nor nead', service.slug)
                stat['empty'].append(service)
            else:
                log.warning('Service %s has no owner, take from head', service.slug)
                stat['set_owner'].append(service)
                set_service_owner(service, heads[0].staff)

    send_to_team(
        'notifications.services.management.sync_owners',
        {
            'sections': [
                {
                    'services': services,
                    'header': comments[section]
                }
                for section, services in stat.items()
                if services
            ]
        }
    )

    log.info(
        'Finish sync owners. Double - %s, set_head - %s, set_owner - %s, different - %s, empty - %s, dismissed - %s',
        len(stat['double']), len(stat['set_head']), len(stat['set_owner']),
        len(stat['different']), len(stat['empty']), len(stat['dismissed']),
    )


def notify_staff_base(self, service_id, url_format, json=None):
    """Пингуем стаффу, что ему нужно обновить у себя данные"""

    url = url_format.format(staff=settings.STAFF_URL, service_id=service_id)

    if not waffle.switch_is_active('push_to_staff'):
        return

    if not Service.objects.get(id=service_id).is_exportable:
        return

    try:
        log.info('Trying to push service #%s update to staff on %s', service_id, url)

        with Session(oauth_token=settings.OAUTH_ROBOT_TOKEN) as session:
            response = session.post(url, json=json)

        log.info('Push to staff successful.')
    except Exception:
        log.exception(
            'Staff push failed for service %s on %s, will retry in %ss',
            service_id, url, settings.STAFF_PUSH_RETRY_COUNTDOWN
        )
        raise self.retry(countdown=settings.STAFF_PUSH_RETRY_COUNTDOWN, args=(service_id,))

    if 300 <= response.status_code < 500:
        log.error(
            'Unexpected status code %s while pushing to staff for service %s on %s',
            response.status_code, service_id, url
        )

    if response.status_code >= 500:
        log.warning(
            'Unexpected status code %s while pushing to staff for service %s on %s, will retry in %ss',
            response.status_code, service_id, url, settings.STAFF_PUSH_RETRY_COUNTDOWN
        )
        raise self.retry(countdown=settings.STAFF_PUSH_RETRY_COUNTDOWN, args=(service_id,))


@app.task(bind=True)
def notify_staff(self, service_id):
    notify_staff_base(
        self,
        service_id,
        '{staff}/api/push/plan/service/{service_id}/',
    )


@app.task
def notify_staff_about_person(staff_id):
    services_ids = set(ServiceMember.objects.filter(
        staff_id=staff_id
    ).values_list('service_id', flat=True))
    for service_id in services_ids:
        notify_staff.delay(service_id=service_id)


@app.task(bind=True, max_retries=25)
def rename_service(self, service_id) -> None:
    service = Service.objects.get(id=service_id)
    log.info(
        'Trying to rename service %s to %s/%s in idm',
        service.slug, service.name, service.name_en,
    )
    try:
        actions.rename_service(service, service.name, service.name_en)
    except Exception as exc:
        log.info(
            'Failed to rename service %s to %s/%s in idm, retrying in 30s',
            service.slug, service.name, service.name_en,
        )
        raise self.retry(exc=exc, countdown=30, args=(service_id, ))  # ретраить через 30с

    log.info(
        'Renamed service %s to %s/%s in idm',
        service.slug, service.name, service.name_en,
    )

    log.info('Changing readonly state(None) on %s', service.slug)
    service.readonly_state = None
    service.save(update_fields=('readonly_state',))

    notify_staff.delay(service_id)


@lock_task
def update_services_kpi(service_id=None):
    qs = Service.objects.alive().order_by('pk')
    if service_id:
        qs = qs.filter(pk=service_id)

    for service in qs:
        sleep(0.5)
        update_fields = []
        kpi = {}

        try:
            kpi['kpi_lsr_count'] = startrek.get_lsr_amount(service)
        except Exception:
            log.warning('Cannot get LSR count for %r', service)

        for contact in service.contacts.startrek_contacts():
            field = {
                settings.STARTREK_BUGS_CONTACT: 'kpi_bugs_count',
                settings.STARTREK_RELEASES_CONTACT: 'kpi_release_count',
            }[contact.type.code]

            try:
                amount = startrek.get_filter_amount(contact.content)
                if amount is not None:
                    kpi[field] = kpi.get(field, 0) + amount

            except Exception:
                log.warning('Cannot get filter count for %r: %s', service, contact.content)

        for field, value in kpi.items():
            if getattr(service, field) != value:
                setattr(service, field, value)
                update_fields.append(field)

        if update_fields:
            log.info('Save new counters to %r: %s', service, kpi)
            service.save(update_fields=update_fields)


@lock_task
def update_service_idm_roles_count(service_id=None):
    services = Service.objects.alive().order_by('id')
    if service_id:
        services = services.filter(id=service_id)

    for service in services:
        try:
            service.idm_roles_count = get_service_group_roles_count(service)
            service.save(update_fields=['idm_roles_count'])
        except IDMError:
            continue


@lock_task(
    lock_key=lambda service_id, batch_id, *args, **kwargs: 'update_service_puncher_rules_count_from_api_{}'.format(batch_id)
)
def update_service_puncher_rules_count_from_api(services_ids=None, batch_id=None):
    client = PuncherClient()
    services = Service.objects.alive().order_by('id')
    if services_ids:
        services = services.filter(id__in=services_ids)
        for service in services:
            try:
                service.puncher_rules_count = client.get_rules_count(service)
                service.save(update_fields=['puncher_rules_count'])
            except RequestException:
                log.warning('Failed to fetch Puncher rules for service %s', service.id)
                continue
    else:
        queue = []
        counter = iteration = 0
        for service in services:
            queue.append(service.id)
            counter += 1
            if counter >= settings.PUNCHER_RULES_BATCH_SIZE:
                update_service_puncher_rules_count_from_api.apply_async(
                    args=[queue, iteration],
                    countdown=settings.ABC_DEFAULT_COUNTDOWN * iteration
                )
                iteration += 1
                counter = 0

        if queue:
            iteration += 1
            update_service_puncher_rules_count_from_api.apply_async(
                args=[queue, iteration],
                countdown=settings.ABC_DEFAULT_COUNTDOWN * iteration
            )


@lock_task
def notify_move_request(move_request_id=None, sender=None):
    if waffle.switch_is_active('notification__move_request__off'):
        return

    requests = ServiceMoveRequest.objects.active().select_related(
        'service__owner',
        'destination__owner',
        'approver_outgoing',
        'approver_incoming',
    )
    if move_request_id:
        requests = requests.filter(pk=move_request_id)
    else:
        requests = requests.filter(
            created_at__lt=timezone.now() - timedelta(days=settings.MOVE_REQUEST_NOTIFICATION_TRESHOLD)
        )

    for request in requests:
        log.info('Notify about %r', request)

        context = {
            'request': request,
        }

        recipient = request.service.owner
        if recipient and not request.approver_outgoing:
            if recipient.login != sender:
                deliver_email(
                    notification_id='notifications.services.move_request_outgoing',
                    context=context,
                    recipients=[recipient],
                )

            request.outgoing_notified = timezone.now()
            request.save(update_fields=['outgoing_notified'])

        recipient = request.destination.owner
        if recipient and not request.approver_incoming:
            if recipient.login != sender:
                deliver_email(
                    notification_id='notifications.services.move_request_incoming',
                    context=context,
                    recipients=[recipient],
                )

            request.incoming_notified = timezone.now()
            request.save(update_fields=['incoming_notified'])


@app.task
def close_service_admin(service_id):
    # для закрытия сервиса из админки
    service = Service.objects.get(id=service_id)
    service.state = Service.states.CLOSED
    service.save(update_fields=('state',))
    notify_staff.apply_async(args=[service_id], countdown=settings.ABC_DEFAULT_COUNTDOWN)
    drop_requests.apply_async(args=[service_id], countdown=settings.ABC_DEFAULT_COUNTDOWN)


@app.task
def close_service(service_close_request_id):
    """
    Закрыть сервис
    """
    request = ServiceCloseRequest.objects.select_related('service').get(pk=service_close_request_id)
    request.process_oebs()
    unclosed_services = request.is_closable_from_d_side()
    if unclosed_services:
        log.info(
            f'Cannot close service {request.service_id}, cause his subservices couldn’t be closed. '
            f'The following services are not closable from dispenser: {unclosed_services}'
        )
        return
    request.process_d()
    request.process_abc()
    request.save()
    drop_requests.delay(request.service.id)
    close_service_abc_side.delay(request.id)


@lock_task
def notify_services_owners():
    qs = (
        Service.objects
        .active()
        .filter(owner__isnull=False)
        .values('owner')
        .annotate(pks=ArrayAgg('id'))
        .values_list('owner', 'pks')
    )
    for owner, services_pks in qs:
        services = Service.objects.filter(pk__in=services_pks).values('id', 'ancestors')
        services_pks = set(services_pks)
        for service in services:
            ancestors_pks = {x['id'] for x in service['ancestors']}
            if ancestors_pks & services_pks:
                services_pks.remove(service['id'])

        trees = []
        for service in Service.objects.filter(pk__in=services_pks):
            trees.append((
                list(service.get_descendants(include_self=True).filter(state__in=Service.states.ACTIVE_STATES)),
                -len(service.ancestors)
            ))

        deliver_email(
            notification_id='notifications.services.your_services',
            context={'trees': trees},
            recipients=[Staff.objects.get(pk=owner)],
        )


@app.task
def process_suspicious_service(service_id, today_is_workday=False, no_owner=False, list_workdays=None):
    today = timezone.now().date()
    one_day_before_move = settings.MOVING_SERVICE_AFTER_DAYS - 1

    service = Service.objects.get(pk=service_id)

    if service.suspicious_date is not None and (service.sandbox_move_date is None or service.is_base):
        from_date = today if service.is_base else service.suspicious_date
        one_day_before_move_date = from_date + timedelta(days=one_day_before_move)
        service.sandbox_move_date = plan.holidays.calendar.next_workday(one_day_before_move_date, list_workdays)
        service.save(update_fields=['sandbox_move_date'])

    # пробуем перенести в песочницу
    if service.should_be_moved_to_sandbox():
        # проверка today_is_workday будет лишней, потому что когда выставляем sandbox_move_date учитывем только рабочие
        # используем условие меньше-равно, тк если с таской вчера было что-то не так,
        # то всех, кого не успели переместить вчера, должны пробовать переместить сегодня
        # помним, что могут проблемы с важными ресурсами
        ServiceMoveRequest.request(
            service=service,
            destination=Service.objects.get_sandbox(),
            requester=Person(get_abc_zombik()),
        )

        return True


def get_context(all_notification):
    context = {'context': []}
    today = utils.today()
    today_shifts_exists = False
    not_today_shifts_exists = False
    services_for_shift_problems = set()
    services_for_schedule_problems = set()
    schedule_new_members = defaultdict(lambda: defaultdict(list))
    for notification_index in range(len(all_notification)):
        notification = all_notification[notification_index]
        if (
                notification.shift and
                any([
                    notification.shift == other.shift and notification.notification_id == other.notification_id
                    for other in all_notification[notification_index+1:]
                ])
        ):
            continue
        shift = notification.shift
        today_shifts_exists |= bool(shift and shift.start_datetime.astimezone(settings.DEFAULT_TIMEZONE).date() == today)
        not_today_shifts_exists |= bool(shift and shift.start_datetime.astimezone(settings.DEFAULT_TIMEZONE).date() != today)
        if notification.service and notification.problem:
            if notification.shift:
                services_for_shift_problems.add(notification.service)
            else:
                services_for_schedule_problems.add(notification.service)
                schedule = notification.problem.schedule
                staff = notification.problem.staff
                schedule_new_members[notification.service][schedule].append(staff)
        context['context'].append({
            'team_service': notification.team_service,
            'parent_service': notification.parent_service,
            'service': notification.service,
            'shift': notification.shift,
            'last_shift': notification.last_shift if notification.last_shift else notification.shift,
            'shift_begin_today': shift and shift.start_datetime.astimezone(settings.DEFAULT_TIMEZONE).date() == today,
            'problem': notification.problem,
            'spliced_shifts_count': notification.spliced_shifts_count,
        })
    context['today_shifts_exists'] = today_shifts_exists
    context['not_today_shifts_exists'] = not_today_shifts_exists
    context['services_for_shift_problems'] = services_for_shift_problems
    context['services_for_schedule_problems'] = services_for_schedule_problems

    prepared_schedule_new_members = []
    for service, schedules in schedule_new_members.items():
        prepared_schedule_staffs = []
        for schedule, staffs in schedules.items():
            names_ru = ', '.join([f'{staff.first_name} {staff.last_name}' for staff in staffs])
            names_en = ', '.join([f'{staff.first_name_en} {staff.last_name_en}' for staff in staffs])
            prepared_schedule_staffs.append((schedule, {'ru': names_ru, 'en': names_en}))
        prepared_schedule_new_members.append((service, prepared_schedule_staffs))

    context['schedule_new_members'] = prepared_schedule_new_members

    return context


@lock_task(lock_key=lambda notification_id, *args, **kwargs: notification_id)
def send_notification_staffs(notification_id, notification_id_for_email, order_by):
    today = utils.today()
    destination_dict = defaultdict(list)
    notifications = (
        ServiceNotification.objects.unsent_notifications(notification_id)
        .select_related('service', 'recipient')
        .order_by(order_by)
    )
    for notification in notifications:
        if notification_id in [
            ServiceNotification.DUTY_PROBLEM_NOTIFICATION,
            ServiceNotification.DUTY_BEGIN_NOTIFICATION,
        ]:
            log.info(
                'Prepare notification %s for %s about shift %s',
                notification.id,
                notification.recipient_id,
                notification.shift_id)
        destination_dict[notification.destination].append(notification)

    if not destination_dict:
        log.warning('Нет данных для отправки {}.'.format(notification_id))
        return None

    for destination, all_notifications in destination_dict.items():
        if notification_id == ServiceNotification.SUSPICION_DIGEST:
            from plan.suspicion.tasks import get_suspicious_context
            context = get_suspicious_context(all_notifications)
        else:
            context = get_context(all_notifications)

        deliver_email(
            notification_id=notification_id_for_email,
            context=context,
            recipients=[destination],
        )
        ServiceNotification.objects.filter(pk__in=[n.pk for n in all_notifications]).update(sent_at=today)


@lock_task
def send_notification_sandbox_announcement():
    if not waffle.switch_is_active('send_suspicious_notifications'):
        return

    today = timezone.now().date()
    next_workday = plan.holidays.calendar.next_workday(today)

    # Необходимо исключить сервисы, которым уже отправлялось письмо о переносе
    services = Service.objects.active().unsent_sandbox_announcements(next_workday)

    for service in services:
        ServiceNotification.objects.create_suspicious_notification_with_recipient(service, ServiceNotification.SANDBOX_ANNOUNCEMENT)

    # группируем нотификации по стаффу
    notification_id_for_email = 'notifications.services.suspicious.move_suspicious_service_one_day'
    send_notification_staffs(ServiceNotification.SANDBOX_ANNOUNCEMENT, notification_id_for_email)


@lock_task
def send_notification_digest_of_changes():
    if not waffle.switch_is_active('send_suspicious_notifications'):
        return

    # отбираем все нотификации, которые подходят по условиям
    notifications = (
        ServiceNotification.objects
        .alive()
        .unsent_notifications(ServiceNotification.CHANGES_SUSPICION_DIGEST)
        .without_destination()
    )

    # для каждого изменения сравниваем прошлый статус подозрительности и текущий
    for notification in notifications:
        if notification.old_suspicious_date != notification.service.suspicious_date:
            # подбираем получателей для этого изменения
            ServiceNotification.objects.create_suspicious_notification_with_recipient(notification.service, ServiceNotification.CHANGES_SUSPICION_DIGEST)

        notification.delete()

    # группируем нотификации по стаффу
    notification_id_for_email = 'notifications.services.suspicious.has_modified'
    send_notification_staffs(ServiceNotification.CHANGES_SUSPICION_DIGEST, notification_id_for_email)


@lock_task
def find_memberships_in_staff():
    meta_other_services = Service.objects.get(slug='meta_other').get_descendants()
    members = ServiceMember.objects.filter(
        found_in_staff_at=None,
        role__is_exportable=True,
    ).exclude(service__in=meta_other_services)
    logins = members.values_list('staff__login', flat=True).distinct()
    importer = MembershipStaffImporter(logins)

    def login_key(gm):
        return gm['person']['login']

    memberships = sorted(importer.get_objects(None), key=login_key)
    staff_map = {
        login: set(gm['group']['url'] for gm in group)
        for login, group in groupby(memberships, key=login_key)
    }

    found_in_staff_pks = []
    for member_pk, login, group_url, group_url_with_scope_old, group_url_with_scope_new in (
        members
            .annotate(group_url=Concat(Value('svc_'), 'service__slug'))
            .annotate(group_url_with_scope=Concat(Value('svc_'), 'service__slug', Value('_'), 'role__scope__slug'))
            .annotate(group_url_with_scope_new=Concat(Value('role_svc_'), 'service__slug', Value('_'), 'role__scope__slug'))
            .values_list('pk', 'staff__login', 'group_url', 'group_url_with_scope', 'group_url_with_scope_new')
    ):
        urls = staff_map.get(login, [])
        if group_url in urls and (group_url_with_scope_old in urls or group_url_with_scope_new in urls):
            found_in_staff_pks.append(member_pk)

    ServiceMember.objects.filter(pk__in=found_in_staff_pks).update(
        found_in_staff_at=timezone.now()
    )


def check_interactive(service: Service, role: Role, robot: Staff) -> bool:
    try:
        actions.request_membership(role=role, subject=robot, service=service, simulate=True)
    except Exception:
        log.warning('Service {} cannot give role {}'.format(service.slug, role.name), exc_info=True)
        return False

    return True


@lock_task
def find_interactive_services():
    role = Role.get_responsible()
    robot = get_abc_zombik()

    for model in (ServiceCreateRequest, ServiceMoveRequest):
        qs = (
            model.objects
            .filter(completed_at__isnull=False, interactive_at=None)
            .select_related('service')
        )
        for req in qs:
            if check_interactive(req.service, role, robot):
                req.interactive_at = timezone.now()
                req.save(update_fields=['interactive_at'])


@lock_task
def sync_department_members_state():
    """
    Проверяем, что у отозванного департамента не осталось
    активных участников
    """
    service_members = ServiceMember.objects.filter(
        from_department__state=ServiceMemberDepartment.states.DEPRIVED
    )
    updated_count = service_members.update(state=ServiceMember.states.DEPRIVED)
    if updated_count:
        log.warning(f'Found and deprived {updated_count} active members of inactive departments')


@app.task(bind=True, default_retry_delay=60)  # 60 seconds
def purge_service_tag_permission(self, service_tag_slug):
    permission = Permission.objects.get(
        content_type__app_label=perm_constants.SERVICE_TAG_APP,
        content_type__model=perm_constants.SERVICE_TAG_MODEL,
        codename=perm_constants.CHANGE_SERVICE_TAG_CODENAME % service_tag_slug,
    )
    service_tag_user_permissions = get_user_model().objects.filter(user_permissions=permission)
    if service_tag_user_permissions.exists():
        self.retry()
    permission.delete()


@app.task
def update_service_review_policy(service_slug: str, review_required: bool = None):
    service = Service.objects.get(slug=service_slug)
    if review_required is None:
        review_required = service.review_required

    set_review_policy_to_service(service, review_required)


@lock_task
def check_service_review_policy_enabled():
    for service in Service.objects.filter(tags__slug__in=settings.REVIEW_REQUIRED_TAG_SLUGS):
        check_review_policy_of_service_roles(service)


@lock_task
def sync_oebs_structure():
    from yql.api.v1.client import YqlClient
    from plan.oebs.utils import get_oebs_products, _build_parents
    from plan.oebs.constants import OEBS_DEVIATIONS_REASONS

    yql_client = YqlClient(token=settings.YQL_OAUTH_TOKEN)
    table_name = f'`home/statkey/moneymap/dict/abc_oebs/prod/t_abc_to_oebs_gl_hier_{utils.today().isoformat()}` '
    query = (
        'USE hahn; '
        f'SELECT ABC_ID, REAL_PARENT_ABC_ID, GL_CODE, GROUP_GL_CODE, '
        f'OEBS_NAME, OEBS_NAME_EN, USE_FOR_GROUP_ONLY, USE_FOR_HR,'
        f' USE_FOR_PROCUREMENT, USE_FOR_REVENUE FROM  {table_name}'
        'where LEAF_FLAG = "Y" '
    )

    request = yql_client.query(query=query)
    request.run()
    oebs_services_map = {}
    for table in request.get_results():
        table.fetch_full_data()
        for row in table.rows:
            (
                abc_id, oebs_parent_id, leaf_oebs_id,
                parent_oebs_id, name, name_en,
                group_only, for_hr, for_procurement,
                for_revenue,
            ) = row
            oebs_services_map[int(abc_id)] = {
                'oebs_parent_id': int(oebs_parent_id),
                'leaf_oebs_id': leaf_oebs_id,
                'parent_oebs_id': parent_oebs_id,  # это id группирующего узла
                'name': name,
                'name_en': name_en,
                'use_for_group_only': True if group_only == 'true' else False,
                'use_for_revenue': True if for_revenue == 'true' else False,
                'use_for_hr': True if for_hr == 'true' else False,
                'use_for_procurement': True if for_procurement == 'true' else False,
            }
    if not oebs_services_map:
        log.error(f'No data in {table_name}')
        return

    services = Service.objects.filter(pk__in=oebs_services_map)
    for service in services:
        service.oebs_data.pop('deviation_reason', None)
        if service.oebs_data != oebs_services_map[service.id]:
            service.oebs_data = oebs_services_map[service.id]
            service.oebs_parent_id = oebs_services_map[service.id]['oebs_parent_id']
            service.save(update_fields=('oebs_data', 'oebs_parent_id', ))

    (Service.objects
            .filter(oebs_parent_id__isnull=False)
            .exclude(pk__in=oebs_services_map)
            .update(oebs_parent_id=None, oebs_data={})
     )

    # теперь поищем расхождения
    deviation_ids = []

    queryset = Service.objects.filter(oebs_parent_id__isnull=False)
    services_ids = [service.id for service in queryset]
    parents_map = {}
    for service in queryset:
        parents_map.update(dict(_build_parents(service)))

    oebs_products = get_oebs_products(
        services_ids
        + list(parents_map.values())
        + list(parents_map.keys())
    )

    for service in queryset:
        parent_id = service.parent_id
        deviation_reason = None
        for key in (
            'name', 'name_en', 'use_for_group_only',
            'use_for_revenue', 'use_for_hr',
            'use_for_procurement',
        ):
            if getattr(service, key) != service.oebs_data.get(key):
                deviation_ids.append(service.id)
                if key in ('name', 'name_en'):
                    deviation_reason = OEBS_DEVIATIONS_REASONS.NAME
                else:
                    deviation_reason = OEBS_DEVIATIONS_REASONS.FLAG
                break

        if not deviation_reason:
            attributes = oebs_products.get(service.id, {})
            if (
                service.oebs_data.get('leaf_oebs_id') != attributes.get('leaf_oebs_id') or
                service.oebs_data.get('parent_oebs_id') != attributes.get('parent_oebs_id')
            ):
                deviation_reason = OEBS_DEVIATIONS_REASONS.RESOURCE

        if not deviation_reason:
            if not parent_id:
                parent_id = 0
            else:
                while parent_id is not None and parent_id not in oebs_products:
                    parent_id = parents_map.get(parent_id)
            if parent_id != service.oebs_parent_id:
                deviation_reason = OEBS_DEVIATIONS_REASONS.PARENT

        if not deviation_reason:
            if 'deviation_reason' in service.oebs_data:
                service.oebs_data.pop('deviation_reason')
                service.save(update_fields=('oebs_data',))

        elif service.oebs_data.get('deviation_reason') != deviation_reason:
            log.info(f'Found deviation by {deviation_reason} for {service.slug}')
            service.oebs_data['deviation_reason'] = deviation_reason
            service.save(update_fields=('oebs_data', ))


@lock_task
def update_functionality(service_states: Optional[list] = None):
    from yql.api.v1.client import YqlClient
    yql_client = YqlClient(token=settings.YQL_OAUTH_TOKEN)

    if not service_states:
        service_states = Service.states.ALL_STATES

    # соберем расписания из watcher
    query = (
        'USE hahn; '
        'SELECT service_id FROM  `//home/abc/watcher_db/schedule` '
    )
    request = yql_client.query(query=query)
    request.run()
    for table in request.get_results():
        table.fetch_full_data()
        watcher_schedules = {
            int(row[0]) for row in table.rows
        }

    # соберем данные из abc
    abc_schedules = set(Schedule.objects.active().values_list('service_id', flat=True))
    suppliers = set(ResourceType.objects.values_list('supplier_id', flat=True))
    with_utility_scope = set(ServiceMember.objects.filter(
        role__scope__utility_scope=True
    ).values_list('service_id' , flat=True).distinct())
    with_functional_scope = set(ServiceMember.objects.filter(
        role__scope__utility_scope=False
    ).values_list('service_id' , flat=True).distinct())
    services_with_warden = set(ServiceResource.objects.filter(
        type__code=settings.WARDEN_RESOURCE_TYPE_CODE,
    ).active().values_list('service_id', flat=True))
    active_functionalities = ServiceTypeFunction.objects.filter(active=True).values_list('code', flat=True)
    functions_by_type = {
        service_type.id: list(service_type.functions.filter(active=True).values_list('code', flat=True))
        for service_type in ServiceType.objects.all()
    }

    services = Service.objects.filter(state__in=service_states)
    for service in services:
        functions = get_service_functions(
            service=service,
            functions_by_type=functions_by_type,
            active_functionalities=active_functionalities,
            services_with_schedules=abc_schedules | watcher_schedules,
            services_with_utility_scope=with_utility_scope,
            services_with_warden=services_with_warden,
            services_with_functional_scope=with_functional_scope,
            suppliers=suppliers,
        )
        if set(service.functions) != set(functions):
            log.info(f'Change functions for {service.slug}, before: {service.functions}, after: {functions}')
            service.functions = functions
            service.save(update_fields=('functions', ))
