import collections

from django.utils import translation
from django.conf import settings

import xlsxwriter

from plan.common.utils.i18n import restore_language
from plan.common.utils.restored_tree import build_object_map, build_restored_tree
from plan.services.models import Service
from plan.resources.models import ServiceResource
from plan.services.types import SERVICE_TYPE

ServiceXlsxData = collections.namedtuple('ServiceXlsxData', ('column_names', 'rows'))
ServiceXlsxRow = collections.namedtuple('ServiceXlsxRow', ('service', 'col_values'))
SORTING_FIELDS = ('-unique_members_count', 'name')


def _sort_nodes(nodes):
    for field in reversed(SORTING_FIELDS):
        reverse = field[0] == '-'
        field = field.lstrip('-')
        nodes.sort(key=lambda n: getattr(n.object, field), reverse=reverse)

    for node in nodes:
        _sort_nodes(node.children)


def restore_tree(services_qs, root):
    object_map = build_object_map(
        objects=services_qs,
        queryset=Service.objects.alive().select_related('owner'),
        load_skipped=True,
        root=root,
    )
    nodes = build_restored_tree(list(services_qs), object_map)
    _sort_nodes(nodes)

    return nodes


@restore_language
def create_xls_data(services_qs, root):
    data = ServiceXlsxData(
        column_names=[
            'ID', 'ID родителя', 'Уровень', 'Название', 'Код', 'Тип',
            'Родительский код ГК OEBS', 'Листовой код ГК OEBS',
            'Используется в HR', 'Используется в Закупках', 'Используется в Выручке',
            'Руководитель', 'Число участников', 'Статус', 'Теги',
        ],
        rows=[],
    )
    translation.activate('ru')
    resources = {
        resource.service_id: (
            resource.attributes.get('parent_oebs_id', ''),
            resource.attributes.get('leaf_oebs_id', ''),
        )
        for resource in ServiceResource.objects.filter(
            service__in=services_qs,
            state=ServiceResource.GRANTED,
            type__code=settings.OEBS_PRODUCT_RESOURCE_TYPE_CODE,
        )
    }
    tree = restore_tree(services_qs, root)

    def add_node(node):
        service = node.object
        parent_oebs_id, leaf_oebs_id = resources.get(service.id, ('', ''))
        col_values = [
            service.id,
            service.parent_id or '-',
            str(SERVICE_TYPE[service.type].name),
            service.name,
            service.slug,
            service.service_type.name,
            parent_oebs_id,
            leaf_oebs_id,
            str(service.use_for_hr),
            str(service.use_for_procurement),
            str(service.use_for_revenue),
        ]

        if not node.skipped:
            tags = ', '.join(tag.name for tag in service.tags.all())
            col_values.extend([
                service.owner.get_full_name() if service.owner else '-',
                service.unique_members_count,
                str(Service.states[service.state].name),
                tags,
            ])

        data.rows.append(ServiceXlsxRow(
            service=node.object,
            col_values=col_values,
        ))

        for _node in node.children:
            add_node(_node)

    for node in tree:
        add_node(node)

    return data


def build_xlsx_workbook(services_qs, root, buffer):
    data = create_xls_data(services_qs, root)
    workbook = xlsxwriter.Workbook(buffer, {'in_memory': True})
    worksheet = workbook.add_worksheet('Services')

    bold = workbook.add_format({'bold': True})
    for col_number, col_name in enumerate(data.column_names):
        worksheet.write(0, col_number, col_name, bold)

    services_with_children = {
        row.service.parent_id for row in data.rows
        if row.service.parent_id
    }

    shift = 0
    if root:
        shift = root.level + 1
    for row_number, row in enumerate(data.rows):
        service = row.service
        for col_number, col_value in enumerate(row.col_values):
            worksheet.write(row_number + 1, col_number, col_value)

        level = service.level - shift
        row_options = {'level': level}

        if service.id in services_with_children:
            row_options['collapsed'] = True

        if service.parent_id and level > 0:
            row_options['hidden'] = True

        worksheet.set_row(row_number + 1, None, None, row_options)

    workbook.close()
