import collections

from plan.api.serializers import serializer_wrapped
from plan.common.utils.restored_tree import build_object_map, build_restored_tree
from plan.services.models import Service, ServiceMember
from plan.services.views.catalog.serializers import ServiceListSerializer
from plan.services.state import SERVICEMEMBER_STATE


class ServiceBuilderBase(object):
    SORT_OPTIONS = {
        'name': ('name',),
        'team_size': ('-unique_members_count', 'name'),
        'kpi_bugs_count': ('kpi_bugs_count',),
        'kpi_release_count': ('kpi_release_count',),
        'kpi_lsr_count': ('kpi_lsr_count',),
        '-kpi_bugs_count': ('-kpi_bugs_count',),
        '-kpi_release_count': ('-kpi_release_count',),
        '-kpi_lsr_count': ('-kpi_lsr_count',),
    }

    def __init__(self, sorting, states, meta, max_level, only_important,
                 with_member=None, branch=None, department=None, tags=None, is_suspicious=None):
        self.sorting = sorting
        self.states = states or Service.states.ALIVE_STATES
        self.meta = meta
        self.branch = branch
        self.max_level = max_level
        self.only_important = only_important
        self.with_member = with_member
        self.department = department
        self.tags = tags
        self._member_approved_service_ids = None
        self.is_suspicious = is_suspicious

    def get_queryset(self):
        if self.meta or self.branch:
            branch = self.meta or self.branch
            query = branch.get_descendants(include_self=True)
        else:
            query = Service.objects.all()

        return query.select_related(*self.get_select_related())

    def get_filtered_queryset(self):
        queryset = self.get_queryset()

        if self.max_level == 'meta':
            queryset = queryset.filter(level=0)
        elif self.max_level == 'group':
            # TODO Убрать костыль после того как https://st.yandex-team.ru/ABC-6942
            parents = queryset.model._cloure_model.values_list('parent').distinct()
            queryset = queryset.exclude(pk__in=parents)

        if self.states:
            queryset = queryset.filter(state__in=self.states)

        if self.only_important:
            queryset = queryset.filter(is_important=True)

        if self.with_member:
            approved_query = Service.objects.filter(
                members__staff=self.with_member,
            )
            self._member_approved_service_ids = {s.id for s in approved_query}

            queryset = queryset.filter(
                members__staff=self.with_member,
                members__state__in=[SERVICEMEMBER_STATE.ACTIVE, SERVICEMEMBER_STATE.DEPRIVING],
            ).distinct()

        if self.department:
            department_ids = self.department.get_ancestors(include_self=True).values_list('id', flat=True)
            service_members = ServiceMember.objects.filter(staff__department_id__in=department_ids)
            services_ids = service_members.values_list('service_id', flat=True)
            services_ids = set(services_ids)
            queryset = queryset.filter(id__in=services_ids)

        if self.tags:
            queryset = queryset.filter(tags__in=self.tags)

        if self.is_suspicious is not None:
            if self.is_suspicious:
                queryset = queryset.exclude(suspicious_date=None)
            else:
                queryset = queryset.filter(suspicious_date=None)

        return queryset

    def get_service_serializer(self):
        attrname = '_service_serializer'
        if not hasattr(self, attrname):
            serializer = lambda obj: ServiceListSerializer(
                obj,
                context={'request': self._request},
            )
            setattr(self, attrname, serializer)
        return getattr(self, attrname)

    def get_select_related(self):
        return ('owner',)

    def sort_nodes(self, nodes):
        for field in reversed(self.SORT_OPTIONS[self.sorting]):
            reverse = field[0] == '-'
            field = field.lstrip('-')
            nodes.sort(key=lambda n: getattr(n.object, field), reverse=reverse)

        for node in nodes:
            self.sort_nodes(node.children)

    def get_result_nodes(self):
        query = self.get_filtered_queryset()

        object_map = build_object_map(
            objects=query,
            queryset=Service.objects.select_related('owner').alive(),
            branch=self.branch,
            load_skipped=True,
        )

        nodes = build_restored_tree(list(query), object_map)
        self.sort_nodes(nodes)

        return nodes

    def get_serialized_results(self, request):
        raise NotImplementedError


class ServiceBuilderNestedJson(ServiceBuilderBase):
    def serialize_node(self, node):
        serializer = self.get_service_serializer()

        result = serializer_wrapped(node.object, obj_type='service', serializer=serializer)
        result['entities'] = [self.serialize_node(_node) for _node in node.children]
        result['value']['skipped'] = node.skipped

        if (
            self.with_member and not node.skipped
            and result['value'].get('id') not in self._member_approved_service_ids
        ):
            result['value']['waiting_approval'] = True
        return result

    def get_select_related(self):
        superself = super(ServiceBuilderNestedJson, self)
        return superself.get_select_related()

    def get_serialized_results(self, request):
        self._request = request
        return [self.serialize_node(node) for node in self.get_result_nodes()]


ServiceXlsxRow = collections.namedtuple('ServiceXlsxRow',
                                        ('service', 'col_values'))


ServiceXlsxData = collections.namedtuple('ServiceXlsxData',
                                         ('column_names', 'rows'))
