from django import forms
from django.utils.translation import ugettext_lazy as _

from plan.common.utils.forms import DefaultDataMixin
from plan.services.models import Service, ServiceTag
from plan.staff.models import Department, Staff


class ModelAltChoiceIterator(forms.models.ModelChoiceIterator):
    def __init__(self, field):
        super(ModelAltChoiceIterator, self).__init__(field)
        self.queryset = field.choices_queryset


class ModelAltChoiceField(forms.ModelChoiceField):
    """
    Добавляет возможность указать альтернативный choices_queryset для choices.
    Валидация происходит как обычно по queryset.
    """

    def __init__(self, *args, **kwargs):
        self._choices_queryset = kwargs.pop('choices_queryset')
        super(ModelAltChoiceField, self).__init__(*args, **kwargs)

    def _get_choices(self):
        if hasattr(self, '_choices'):
            return self._choices
        return ModelAltChoiceIterator(self)

    choices = property(_get_choices, forms.ModelChoiceField._set_choices)

    @property
    def choices_queryset(self):
        return self._choices_queryset or self.queryset


class ServiceCatalogForm(DefaultDataMixin, forms.Form):
    default_data = {
        'max_level': 'all',
        'branch': '',
        'show': 'my',
        'sorting': 'team_size',
        'states': [state for state in Service.states
                   if state in Service.states.ALIVE_STATES],
    }

    sorting = forms.ChoiceField(required=False, choices=(
        ('name', 'По алфавиту'),
        ('team_size', 'По числу участников'),
        ('kpi_bugs_count', 'По количеству багов'),
        ('kpi_release_count', 'По количеству релизов'),
        ('kpi_lsr_count', 'По количеству LSR'),
        ('-kpi_bugs_count', 'По количеству багов'),
        ('-kpi_release_count', 'По количеству релизов'),
        ('-kpi_lsr_count', 'По количеству LSR'),
    ))

    max_level = forms.ChoiceField(
        required=False,
        choices=[
            ('all', _('Все сервисы')),
            ('group', _('Только группы')),
            ('meta', _('Только метасервисы')),
        ]
    )

    branch = ModelAltChoiceField(
        required=False,
        empty_label='Все метасервисы',
        queryset=Service.objects.all(),
        choices_queryset=Service.objects.alive().filter(level=0),
    )

    states = forms.MultipleChoiceField(
        required=False,
        choices=[(id, Service.states[id].name)
                 for id in Service.states if id in Service.states.ALIVE_STATES],
    )

    important = forms.BooleanField(
        required=False,
        label='Только важные',
    )

    show = forms.ChoiceField(choices=(
        ('my', 'Мои сервисы'),
        ('mydepartment', 'Сервисы моего подразделения'),
        ('all', 'Все сервисы'),
    ))

    tags = forms.ModelMultipleChoiceField(
        required=False,
        queryset=ServiceTag.objects.all(),
    )

    person = forms.ModelChoiceField(
        required=False,
        label='Участник',
        to_field_name='login',
        queryset=Staff.objects.all(),
    )

    department = forms.ModelChoiceField(
        required=False,
        label='Департамент',
        queryset=Department.objects.all(),
    )

    is_suspicious = forms.NullBooleanField(
        required=False,
        label='Подозрительный',
    )
