import dateutil
from collections import defaultdict, namedtuple
from copy import deepcopy
from typing import List, Set

from django.conf import settings

from rest_framework import serializers

from plan.api.idm import actions as idm_actions
from plan.api.idm import helpers as idm_helpers
from plan.api.intranet.persons import LegacyPersonSerializer
from plan.common.utils import memoize
from plan.common.utils.intranet import departments
from plan.common.utils.oauth import get_abc_zombik
from plan.idm import exceptions as idm_exceptions
from plan.idm.constants import STATES, ACTIVE_STATES, INACTIVE_STATES
from plan.services import permissions, checkers
from plan.services.api.contacts import OldContactSerializer
from plan.services.api.services import ListServiceSerializer, ServiceTagSerializer
from plan.services.api.moves import MoveRequestSerializer
from plan.services.api.team import PersonSerializer, RoleSerializer, ServiceMemberSerializer
from plan.services.constants.action import MEMBER_FIELDS, DEFAULT_ACTIONS
from plan.services.models import Service, Staff, Role, ServiceMoveRequest, ServiceMember, Department

ChownRequest = namedtuple('ChownRequest', ('service', 'requester', 'new_owner', 'can_approve'))


class IDMSerializer(serializers.ModelSerializer):
    raw_state = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()
    approvers = serializers.SerializerMethodField()
    expires = serializers.SerializerMethodField()
    role_url = serializers.SerializerMethodField()

    def __init__(self, *args, **kwargs):
        self.service = kwargs.pop('service', None)
        self.person = kwargs.pop('person', None)

        super(IDMSerializer, self).__init__(*args, **kwargs)

    def get_raw_state(self, obj):
        return {
            'value': obj['state'],
            'title': STATES[obj['state']],
        }

    def get_actions(self, obj):
        if 'permissions' not in obj:
            prefix = self.action_prefix(obj)
            visible_actions = self.visible_actions(self.person, self.service)
            possible_actions = self.guess_actions(obj)
            return [
                prefix + action
                for action in possible_actions
                if action in visible_actions
            ]

        actions = []
        prefix = self.action_prefix(obj)

        if obj['permissions'].get('can_be_approved'):
            actions.append(prefix + 'approve')
            actions.append(prefix + 'decline')

        if obj['permissions'].get('can_be_deprived'):
            actions.append(prefix + 'remove')

        if obj['permissions'].get('can_be_rerequested'):
            actions.append(prefix + 'rerequest')

        return actions

    def get_approvers(self, obj):
        # workflow системы ABC написан так, что здесь всегда один блок из нескольких ИЛИ подтверждающих
        approvers = obj.get('require_approve_from')
        if not approvers:
            return []

        staff_approvers = Staff.objects.filter(login__in=(approver['username'] for approver in approvers[0]))
        return [PersonSerializer(approver).data for approver in staff_approvers]

    def get_expires(self, obj):
        expires = obj.get('expire_at')
        role_is_active = obj.get('state') in [
            'rerequested',
            'need_request',
            'granted',
            'depriving',
            'review_request',
        ]

        if role_is_active and expires:
            return dateutil.parser.parse(expires)
        else:
            return None

    def get_role_url(self, obj):
        role_filter = 'role={id}'.format(id=obj['id'])
        return settings.IDM_ROLE_FILTER_URL.format(filter=role_filter)

    @staticmethod
    def guess_actions(role: dict) -> List[str]:
        action_states = {
            'approve': (
                'requested',
                'awaiting',
            ),
            'decline': (
                'requested',
                'awaiting',
            ),
            'remove': (
                'depriving_validation',
            ),
            'rerequest': (
                'requested',
                'declined',
                'failed',
                'need_request',
                'deprived',
            ),
        }
        state_actions = {}
        for action, states in action_states.items():
            for s in states:
                state_actions.setdefault(s, []).append(action)

        state = role.get('state')
        return state_actions.get(state, [])

    @staticmethod
    def visible_actions(person, service) -> Set[str]:
        is_responsible = person.is_responsible(service)
        is_superuser = person.user.is_superuser
        action_visible = {
            'approve': is_responsible or is_superuser,
            'decline': is_responsible or is_superuser,
            'remove': is_responsible or is_superuser,
            'rerequest': is_responsible or is_superuser,
        }
        return {action for action in action_visible if action_visible[action]}

    @staticmethod
    def action_prefix(role: dict) -> str:
        return 'member_' if role.get('user') else 'department_'


class IDMStateSerializer(IDMSerializer):
    class Meta:
        model = Service
        fields = ('id', 'actions', 'expires', 'raw_state', 'role_url')

    def get_id(self, obj):
        sm = idm_helpers.get_membership_by_role(obj)
        if sm:
            return sm.pk


class IDMPersonSerializer(IDMSerializer):
    person = serializers.SerializerMethodField()
    raw_state = serializers.SerializerMethodField()
    role = serializers.SerializerMethodField()
    state = serializers.SerializerMethodField()
    fromDepartment = serializers.SerializerMethodField(method_name='get_from_department')
    serviceMemberDepartmentId = serializers.SerializerMethodField(method_name='get_from_department_id')

    class Meta:
        model = Service
        fields = (
            'id', 'person', 'raw_state', 'role', 'role_url', 'state', 'actions',
            'approvers', 'expires', 'fromDepartment', 'serviceMemberDepartmentId'
        )

    def get_state(self, obj):
        return 'waiting_approval'  # Legacy

    def get_role(self, obj):
        role_id = int(obj['node']['data']['role'])
        role = Role.objects.get(pk=role_id)

        return RoleSerializer(role).data

    def get_person(self, obj):
        username = obj['user']['username']
        user = Staff.objects.get(login=username)

        return PersonSerializer(user).data

    def get_from_department(self, obj):
        if obj['group']:
            dpt = obj['group']['department']
            return {'id': dpt.id, 'name': dpt.name}
        else:
            return None

    def get_from_department_id(self, obj):
        if obj['group']:
            return obj['id']
        else:
            return None


class IDMGroupSerializer(IDMSerializer):
    id = serializers.SerializerMethodField(source='real_id')
    name = serializers.SerializerMethodField()
    state = serializers.SerializerMethodField()
    role = serializers.SerializerMethodField()
    isApproved = serializers.BooleanField(source='is_approved')
    membersCount = serializers.IntegerField(source='members_count')
    serviceMemberDepartmentId = serializers.SerializerMethodField(method_name='get_real_id')

    class Meta:
        model = Service
        fields = (
            'actions', 'approvers', 'expires', 'id', 'isApproved', 'membersCount',
            'name', 'raw_state', 'serviceMemberDepartmentId', 'role', 'state', 'role_url'
        )

    def get_id(self, obj):
        return obj['group']['department'].id

    def get_name(self, obj):
        return obj['group']['department'].i_name

    def get_role(self, obj):
        role_id = int(obj['node']['data']['role'])
        role = Role.objects.get(pk=role_id)

        return RoleSerializer(role).data

    def get_state(self, obj):
        return 'new'

    def get_is_approved(self, obj):
        return False

    def get_members_count(self, obj):
        return obj['group']['department_member_count']

    def get_real_id(self, obj):
        if obj['group']:
            return obj['id']
        else:
            return None


class MicroServiceSerializer(serializers.ModelSerializer):
    class Meta:
        model = Service
        fields = ('id', 'slug', 'name', )


class ResponsibleSerializer(serializers.ModelSerializer):
    person = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()

    def __init__(self, *args, **kwargs):
        self.is_responsible = kwargs.pop('is_responsible', False)
        super(ResponsibleSerializer, self).__init__(*args, **kwargs)

    class Meta:
        model = ServiceMember
        fields = ['person', 'actions']

    def get_person(self, obj):
        return PersonSerializer(obj.staff).data

    def get_actions(self, obj):
        if obj.service.state not in Service.states.ACTIVE_STATES:
            return []

        actions = []

        if obj.role.code == Role.RESPONSIBLE and self.is_responsible:
            actions.append('member_remove')

        return actions


class ServiceDepartmentSerializer(serializers.ModelSerializer):
    # TODO: сделать описание подразделения вложенным объектом
    role = RoleSerializer()
    name = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()
    state = serializers.SerializerMethodField()
    isApproved = serializers.SerializerMethodField(method_name='get_is_approved')
    membersCount = serializers.SerializerMethodField(method_name='get_members_count')
    serviceMemberDepartmentId = serializers.SerializerMethodField(method_name='get_real_id')

    def __init__(self, *args, **kwargs):
        self.person = kwargs.pop('person', None)
        self.prefetched = kwargs.pop('prefetched', None)
        self.checker = kwargs.pop('checker', None) or checkers.service_checker

        super(ServiceDepartmentSerializer, self).__init__(*args, **kwargs)

    class Meta:
        model = Service
        fields = ('id', 'name', 'role', 'actions', 'state', 'isApproved', 'membersCount', 'serviceMemberDepartmentId')

    def get_name(self, obj):
        return obj.department.i_name

    def get_state(self, obj):
        return 'approved'

    def get_id(self, obj):
        return obj.department_id

    def get_real_id(self, obj):
        return obj.id

    def get_is_approved(self, obj):
        return True

    def get_actions(self, obj):
        if obj.service.state not in Service.states.ACTIVE_STATES:
            return []

        subject_messages = {
            'department_remove': ('Отвязать подразделение', ),
            'department_approve': ('Подтвердить подразделение', ),
        }

        if self.get_is_approved(obj) or not self.get_members_count(obj):
            subject_messages.pop('department_approve')

        if not self.checker.can_remove_department(obj.service, self.person):
            subject_messages.pop('department_remove')

        return [
            {msg_id: {'verb': verb}}
            for msg_id, verb in subject_messages.items()
        ]

    def get_members_count(self, obj):
        return len([
            member for member in self.prefetched['all_members']
            if member.from_department and
            member.from_department_id == obj.id
        ])


class ServiceListSerializer(ListServiceSerializer):
    name = serializers.CharField(source='i_name')

    class Meta:
        model = Service
        fields = (
            'activity',
            'ancestors',
            'children_count',
            'created_at',
            'descendants_count',
            'description',
            'id',
            'is_exportable',
            'is_suspicious',
            'sandbox_move_date',
            'keywords',
            'modified_at',
            'name',
            'owner',
            'parent',
            'membership_inheritance',
            'path',
            'readonly_state',
            'slug',
            'state',
            'state_display',
            'state_display_i18n',
            'type',
            'related_services',
            'unique_immediate_members_count',
            'unique_immediate_robots_count',
            'unique_immediate_external_members_count',
            'has_forced_suspicious_reason',
            'has_external_members',
        )


class ServiceSerializer(ListServiceSerializer):
    """ Имитирует поведение FatServiceDehydrator """

    name = serializers.CharField(source='i_name')
    owner = PersonSerializer()
    actions = serializers.SerializerMethodField()
    teamCount = serializers.IntegerField(source='unique_members_count')
    subserviceCount = serializers.SerializerMethodField(method_name='get_subservice_count')
    teamImmediateCount = serializers.IntegerField(source='unique_immediate_members_count')
    teamImmediateRobotsCount = serializers.IntegerField(source='unique_immediate_robots_count')
    immediateExternalCount = serializers.IntegerField(source='unique_immediate_external_members_count')
    state = serializers.SerializerMethodField()
    availableStates = serializers.SerializerMethodField(method_name='get_available_states')
    tags = serializers.SerializerMethodField()
    team_statuses = serializers.SerializerMethodField()
    isImportant = serializers.BooleanField(source='is_important')
    contacts = serializers.SerializerMethodField()
    responsible = serializers.SerializerMethodField()
    departments = serializers.SerializerMethodField()
    team = serializers.SerializerMethodField()
    chownRequest = serializers.SerializerMethodField(method_name='get_chown_request')
    outgoingMoveRequest = serializers.SerializerMethodField(method_name='get_outgoing_move_request')
    incomingMoveRequests = serializers.SerializerMethodField(method_name='get_incoming_move_requests')
    ancestors = serializers.SerializerMethodField()
    readonly_state = serializers.SerializerMethodField()

    def __init__(self, *args, **kwargs):
        move_requests = kwargs.pop('move_requests', [])
        fields = kwargs.pop('fields', None)

        self.person = kwargs.pop('person', None)
        request = kwargs.get('context', {}).get('request')
        if not self.person and request:
            self.person = request.person
        self.checkers = {}
        self.immediate_members = kwargs.pop('immediate_members', None)
        self.oucoming_move_request_by_service = {}
        self.incoming_move_requests_count_by_service = defaultdict(lambda: 0)

        super(ServiceSerializer, self).__init__(*args, **kwargs)

        for s in move_requests:
            self.oucoming_move_request_by_service[s.service_id] = s
            self.incoming_move_requests_count_by_service[s.destination_id] += 1

        if fields:
            to_add = set(fields) - set(self.fields)
            to_delete = set(self.fields) - set(fields)
            for field in to_add:
                method_name = 'get_' + field

                if not hasattr(self, method_name) and hasattr(Service, field):
                    setattr(self, method_name, lambda obj, f=field: getattr(obj, f))

                if hasattr(self, method_name):
                    self.fields[field] = serializers.SerializerMethodField(method_name=method_name)

            for field in to_delete:
                self.fields.pop(field)

    class Meta:
        model = Service
        fields = [
            'id', 'name', 'level', 'type', 'slug', 'ancestors',
            'readonly_state', 'is_exportable', 'is_suspicious', 'state',
            'availableStates', 'incomingMoveRequests', 'outgoingMoveRequest',
            'chownRequest', 'owner', 'team', 'departments', 'responsible', 'team_statuses',
            'description', 'activity', 'contacts', 'url', 'actions', 'subserviceCount',
            'teamCount', 'teamImmediateCount', 'teamImmediateRobotsCount', 'immediateExternalCount',
            'kpi', 'isImportant', 'tags', 'has_external_members',
        ]

    def get_departments(self, obj):
        serializer = lambda department: ServiceDepartmentSerializer(
            department,
            person=self.person,
            prefetched={'all_members': self.get_all_members(service=department)},
            checker=self._get_checker(obj),
        )
        query = obj.department_memberships.all()
        query = query.select_related('department')
        return [serializer(department).data for department in query]

    @memoize.memoize_in_first_arg(keygetter=lambda service: service.id)
    def get_all_members(self, service):
        select_related = ['from_department', 'from_department__department',
                          'role', 'staff', 'role']
        return service.members.team().select_related(*select_related)

    def _get_checker(self, service):
        if service not in self.checkers:
            self.checkers[service] = checkers.ServicePermissionChecker(
                service=service,
                sender=self.person,
            )
        return self.checkers[service]

    def get_available_states(self, obj):
        unavailable_states = {obj.state}
        if not permissions.can_delete_service(obj, self.person):
            unavailable_states.add(Service.states.DELETED)
        if not permissions.can_close_service(obj, self.person):
            unavailable_states.add(Service.states.CLOSED)
        if obj.state not in Service.states.NORMAL_STATES and obj.has_change_state_execution():
            unavailable_states.update(Service.states.NORMAL_STATES)

        return [
            {'id': state, 'verbose': Service.states[state].name}
            for state in Service.states
            if state not in unavailable_states
        ]

    def get_readonly_state(self, obj):
        return obj.human_readonly_state

    def get_tags(self, obj):
        serializer = ServiceTagSerializer
        serializer.Meta.fields += ('name_en', )
        return [serializer(tag).data for tag in obj.tags.all()]

    def get_state(self, obj):
        return {
            'id': obj.state,
            'verbose': Service.states[obj.state].name
        }

    def get_contacts(self, obj):
        contacts = obj.contacts.all().order_by('position')

        return [OldContactSerializer(contact).data for contact in contacts]

    def get_kpi(self, obj):
        return {
            'bugs_count': obj.kpi_bugs_count,
            'releases_count': obj.kpi_release_count,
            'lsr_count': obj.kpi_lsr_count,
        }

    def get_subservice_count(self, obj):
        return obj.get_descendants().exclude(
            state=Service.states.DELETED
        ).count()

    def get_responsible(self, obj):
        is_responsible = permissions.is_service_responsible(obj, self.person)

        serializer = lambda responsible: ResponsibleSerializer(
            responsible,
            is_responsible=is_responsible
        )
        return [serializer(responsible).data for responsible in obj.get_responsible()]

    def get_ancestors(self, obj):
        return [MicroServiceSerializer(ancestor).data for ancestor in reversed(obj.get_ancestors())]

    def get_actions(self, obj):
        all_possible_actions = [
            {
                action: {'verb': verb}
            }
            for action, verb in DEFAULT_ACTIONS.items()
        ]

        is_responsible = permissions.is_service_responsible(obj, self.person)

        if is_responsible:
            all_possible_actions.extend([
                {'member_approve': {'verb': ('Подтвердить участие', )}},
                {'member_decline': {'verb': ('Отклонить участие', )}},
                {'department_approve': {'verb': ('Отклонить участие', )}},
                {'department_decline': {'verb': ('Отклонить подразделение', )}},
                {'chown_approve': {'verb': ('Подтвердить изменение руководителя', )}},
                {'delete': {'verb': ('Удалить сервис', )}},
            ])

        if obj.readonly_state is not None:
            for action in list(all_possible_actions):
                if list(action.keys())[0] in MEMBER_FIELDS:
                    all_possible_actions.remove(action)

        # редирект на страницу создания проекта.
        # Но для фронта это action, поэтому такой костыль
        fake_actions = []
        if obj.state != 'deleted':
            fake_actions.extend([
                {'request_resource': {'verb': 'Запросить ресурс'}},
            ])

            if is_responsible:
                fake_actions.append({
                    'provide_resource': {'verb': 'Предоставить ресурс'},
                })

        return all_possible_actions + fake_actions

    def expand_groups(self, members):
        """Вдобавок к роли на группу вставляет роль на каждого человека"""
        full_members = []
        for member in members:
            full_members.append(member)

            if member['group']:
                department = Department.objects.get(staff_id=member['group']['id'])

                persons = Staff.objects.filter(
                    departments.get_members_query(
                        department=department,
                        with_nested=True,
                    ),
                    is_dismissed=False,
                )

                for person in persons:
                    pm = deepcopy(member)
                    pm['user'] = {'username': person.login}
                    pm['group']['department'] = department
                    full_members.append(pm)

                member['group']['department'] = department
                member['group']['department_member_count'] = len(persons)

        return full_members

    def get_team_statuses(self, service):
        """
            Статусы ролей сервиса из IDM
        """
        result = {
            'inactive': {
                'persons': [],
                'departments': [],
            },
            'active': {
                'persons': [],
                'departments': [],
            }
        }

        if not service.is_alive or service.readonly_state == Service.CREATING:
            return result

        try:
            roles = idm_actions.get_roles(
                service=service,
                requester=self.person.staff,
                state=INACTIVE_STATES + ACTIVE_STATES,
                full=False,
                timeout=settings.IDM_GET_ROLES_TIMEOUT,
            )

        except idm_exceptions.BadRequest:
            return result

        active_roles = [role for role in roles if role['state'] in ACTIVE_STATES]
        inactive_roles = [role for role in roles if role['state'] in INACTIVE_STATES + ['rerequested']]

        for role in active_roles:
            section = 'persons' if role['user'] else 'departments'
            result['active'][section].append(
                IDMStateSerializer(role, service=service, person=self.person).data
            )

        for role in inactive_roles:
            current_roles = self.expand_groups([role])
            for current_role in current_roles:
                section, serializer = (
                    ('persons', IDMPersonSerializer)
                    if current_role['user'] else
                    ('departments', IDMGroupSerializer)
                )
                result['inactive'][section].append(
                    serializer(current_role, service=service, person=self.person).data
                )

        return result

    def get_team(self, obj):
        if self.immediate_members:
            all_members = obj.members.team()
        else:
            all_members = self.get_all_members(service=obj)

        members = [ServiceMemberSerializer(member).data for member in all_members]

        # новый сериалайзер не всегда возвращает fromDepartment, а старая ручка просит его
        for member in members:
            if 'fromDepartment' not in member:
                member['fromDepartment'] = None

        return members

    def get_chown_request(self, obj):
        try:
            roles = idm_actions.get_chown_requests(obj, requester=self.person)
        except idm_exceptions.BadRequest:
            return []

        requests = [
            ChownRequest(
                service=obj,
                requester=(Staff.objects.get(login=role['role_request']['requester']['username'])
                           if role.get('role_request')
                           else get_abc_zombik()),
                new_owner=Staff.objects.get(login=role['user']['username']),
                can_approve=role.get('permissions', {}).get('can_be_approved'),
            )
            for role in roles
        ]

        return [{
            'service': MicroServiceSerializer(request.service).data,
            'requester': PersonSerializer(request.requester).data,
            'new_owner': PersonSerializer(request.new_owner).data,
            'actions': ['chown_approve', 'chown_cancel'] if request.can_approve else []
        } for request in requests]

    def get_outgoing_move_request(self, obj):
        try:
            req = obj.move_requests.active().get()
        except ServiceMoveRequest.DoesNotExist:
            return

        serializer = lambda req: MoveRequestSerializer(
            req,
            context=self.context,
        )

        return serializer(req).data

    def get_incoming_move_requests(self, service):
        serializer = lambda obj: MoveRequestSerializer(
            obj,
            context=self.context,
        )
        query = service.incoming_move_requests.active()
        query = query.select_related(
            'requester',
            'approver_outgoing',
            'approver_incoming',
        )
        return [serializer(request).data for request in query]


class ServiceLiteSerializer(serializers.ModelSerializer):
    owner = LegacyPersonSerializer()
    state_display = serializers.SerializerMethodField()

    class Meta:
        model = Service
        fields = ('id', 'slug', 'name', 'owner', 'state', 'state_display')

    def get_state_display(self, obj):
        return obj.get_state_display()
