from rest_framework.views import APIView
from rest_framework.permissions import IsAuthenticated

from plan import exceptions
from plan.common.utils.rest_fw import plan_response
from .builders import ServiceBuilderNestedJson


class ServiceCatalogView(APIView):
    permission_classes = (IsAuthenticated, )
    http_method_names = ('get',)

    def get(self, request, *args, **kwargs):
        person = request.person
        can_filter_important = self.can_filter_important(
            person=person
        )
        actions = []
        if can_filter_important:
            actions.append({'filter_important': {
                'verb': 'filter_important'
            }})
        if person.user.has_perm('internal_roles.change_importance'):
            actions.extend([
                {'download_important': {
                    'verb': 'download_important'
                }},
                {'change_importance': {
                    'verb': 'change_importance'
                }}
            ])

        from plan.services.views.catalog.forms import ServiceCatalogForm
        form = ServiceCatalogForm(request.GET)
        if not form.is_valid():
            raise exceptions.FormConflictError(form)

        only_important = can_filter_important and form.cleaned_data['important']

        builder_cls = ServiceBuilderNestedJson

        show = form.cleaned_data['show']
        person = None
        department = None
        if form.cleaned_data['person']:
            person = form.cleaned_data['person']
        elif form.cleaned_data['department']:
            department = form.cleaned_data['department']
        elif show == 'my':
            person = request.user.staff
        elif show == 'mydepartment':
            department = request.user.staff.department

        builder = builder_cls(
            sorting=form.cleaned_data['sorting'],
            states=form.cleaned_data['states'],
            branch=form.cleaned_data['branch'],
            meta=form.cleaned_data['branch'],
            max_level=form.cleaned_data['max_level'],
            only_important=only_important,
            with_member=person,
            department=department,
            tags=form.cleaned_data['tags'],
            is_suspicious=form.cleaned_data['is_suspicious'],
        )

        return plan_response({
            # искусственно синтезированные права доступа like yawf
            'actions': actions,
            'services': builder.get_serialized_results(request=request),
        })

    @staticmethod
    def can_filter_important(person):
        return person.user.has_perm('internal_roles.change_importance')
