import logging

from django.conf import settings
from django.db import transaction
from django.http import HttpResponseRedirect
from django.utils import timezone

from plan.api.exceptions import ValidationError
from plan.common.utils import replicated
from plan.lib.views import base
from plan.services import forms
from plan.services import models
from plan.services.tasks import register_service

logger = logging.getLogger(__name__)


class ServiceCreateView(base.UniversalView):
    @replicated.updates_on_success()
    @transaction.atomic
    def dispatch(self, *args, **kwargs):
        return super(ServiceCreateView, self).dispatch(*args, **kwargs)

    def get(self, request):
        return HttpResponseRedirect('/services/?createService=yes')

    def post(self, request):
        raw_serializer = forms.ServiceCreateSerializer(data=self.json_data)
        if not raw_serializer.is_valid():
            raise ValidationError(extra=raw_serializer.errors)

        name_data = raw_serializer.data.pop('name')
        raw_data = raw_serializer.data.copy()
        raw_data['name'] = name_data['ru']
        raw_data['name_en'] = name_data['en']

        form = forms.ServiceCreateForm(raw_data)
        if not form.is_valid():
            raise ValidationError(extra=form.errors)

        requester = request.person.staff
        move_to = form.cleaned_data.get('move_to')
        if not models.Service.can_staff_place_service_here(requester, move_to, is_base=False):
            raise ValidationError(extra={'move_to': ['Вы не можете сделать сервис потомком "%s".' % move_to.name]})

        service = form.save(commit=False)
        service.parent = models.Service.objects.get(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG)
        service.readonly_state = 'creating'
        service.readonly_start_time = timezone.now()
        service.service_type = models.ServiceType.get_default_type()
        service.save()

        logger.info(
            'Service was created by %(login)s with id=%(service_id)s',
            {
                'login': requester.login,
                'service_id': service.id,
            }
        )

        create_request = models.ServiceCreateRequest.request(
            service=service,
            requester=request.person,
            move_to=move_to
        )
        register_service.apply_async(
            args=(create_request.id,),
            countdown=settings.ABC_DEFAULT_COUNTDOWN
        )

        return {'new_service_id': service.id}


create_view = ServiceCreateView.as_view()
