from typing import List, Dict

from rest_framework.views import APIView
from rest_framework.permissions import IsAuthenticated

from plan.common.utils.forms import FilterFieldsForm
from plan.common.utils.rest_fw import plan_response, call_old_exception
from plan.services.api.exceptions import LegacyServiceNotFound
from plan.services.models import Service
from plan.services.views.catalog.serializers import ServiceSerializer
from plan.api.mixins import FieldsByPermissionsCutterMixin


class ServiceDetailView(FieldsByPermissionsCutterMixin, APIView):
    permission_classes = (IsAuthenticated, )
    _permissions_to_proceed = 'view_own_services'

    _EMPTY = 'empty'
    PERMISSION_CODENAME_MAPPER = {
        'view_team': {
            'owner',
            'responsible',
            'team',
            'unique_members_count',
            'team_statuses',
            'unique_immediate_members_count',
            'unique_immediate_robots_count',
            'unique_immediate_external_members_count',
            'has_external_members',
        },
        'view_hierarchy': {
            'level',
        },
        'can_view': {
            'id',
            'slug',
            'name',
            'ancestors',
            'readonly_state',
            'departments',
        },
        'view_description': {
            'description',
        },
        'view_all_services': {
            'subservice_count',
        },
        'view_kpi': {
            'kpi',
        },
        'view_tags': {
            'tags',
        },
        'view_details': {
            'url',
            'type',
            'state',
            'available_states',
            'actions',
            'is_exportable',
            'is_suspicious',
            'is_important',
            'incoming_move_requests',
            'outgoing_move_request',
            'chown_request',
        },
        'view_activity': {
            'activity',
        },
        'view_contacts': {
            'contacts',
        }
    }
    FIELDS_RENAME = {
        'available_states': 'availableStates',
        'chown_request': 'chownRequest',
        'incoming_move_requests': 'incomingMoveRequests',
        'is_important': 'isImportant',
        'outgoing_move_request': 'outgoingMoveRequest',
        'subservice_count': 'subserviceCount',
        'unique_immediate_members_count': 'teamImmediateCount',
        'unique_immediate_robots_count': 'teamImmediateRobotsCount',
        'unique_immediate_external_members_count': 'immediateExternalCount',
        'unique_members_count': 'teamCount'
    }

    def get(self, request, pk_or_slug):
        try:
            service = Service.objects.select_related('owner').get_by_pk_or_slug(pk_or_slug)
            if not self.is_visible(service, request.user):
                return call_old_exception(LegacyServiceNotFound(detail=f'Service {pk_or_slug} not found'))
        except Service.DoesNotExist:
            return call_old_exception(LegacyServiceNotFound(detail=f'Service {pk_or_slug} not found'))

        person = request.person

        fields_filter = FilterFieldsForm(request.GET or None)
        fields = None
        if fields_filter.is_valid():
            fields = fields_filter.cleaned_data['fields']

        return plan_response({'service': self.serialize(request, person, service, fields)})

    @staticmethod
    def serialize(request, person, service, fields: List[str]) -> Dict:
        if not fields:
            return {}

        # если нужно фильтровать поля, нужно передать fields=fields в сериалайзер
        serializer = lambda service, **kwargs: ServiceSerializer(
            service,
            fields=fields,
            **kwargs
        )
        service = serializer(service, person=person, context={'request': request}).data

        for field, rename_field in ServiceDetailView.FIELDS_RENAME.items():
            if field in service:
                service[rename_field] = service[field]
                del service[field]

        return service

    @staticmethod
    def is_visible(service, user):
        if user.has_perm('internal_roles.view_all_services'):
            return True
        return service.members.filter(staff_id=user.staff.id).exists()
