from rest_framework import viewsets, mixins
from rest_framework.generics import get_object_or_404 as drf_get_object_or_404

from plan.api.mixins import OrderingMixin
from plan.staff.models import Department, Staff
from plan.api.serializers import (
    CompactStaffSerializer,
    UpdateStaffSerializer,
)
from plan.idm.manager import idm_manager
from .serializers import StaffDepartmentSerializer
from plan.swagger import SwaggerFrontend
from plan.api.filters import PlanFilterSet
from plan.api.mixins import DefaultFieldsMixin, SelectOnlyFieldsMixin
from plan.api.base import ABCCursorPagination
from plan.api.permissions import TvmAuthenticated
from plan.services.permissions import is_service_moderator
from plan.services.tasks import notify_staff_about_person


class StaffDepartmentsView(OrderingMixin, viewsets.ReadOnlyModelViewSet):
    """
    Департаменты со стаффа
    """
    default_swagger_schema = SwaggerFrontend

    _permissions_to_proceed = 'view_hierarchy'
    serializer_class = StaffDepartmentSerializer
    queryset = Department.objects.all()


class StaffFilterSet(PlanFilterSet):

    class Meta:
        model = Staff
        fields = ('is_frozen', 'login', 'staff_id', 'uid')


class StaffPermissions(TvmAuthenticated):
    def has_object_permission(self, request, view, obj):
        if view.action == 'list':
            return True
        elif view.action in ('update', 'partial_update'):
            return is_service_moderator(request.person)

        return False


class StaffPersonView(
    DefaultFieldsMixin, SelectOnlyFieldsMixin,
    OrderingMixin, mixins.UpdateModelMixin,
    mixins.ListModelMixin, viewsets.GenericViewSet,
):
    """
    Люди со стаффа
    """
    _permissions_to_proceed = 'view_hierarchy'
    serializer_class = CompactStaffSerializer
    update_serializer = UpdateStaffSerializer
    filterset_class = StaffFilterSet
    http_method_names = ['get', 'patch', ]
    permission_classes = [StaffPermissions]
    queryset = Staff.objects.filter(is_dismissed=False)
    pagination_class = ABCCursorPagination
    default_fields = [
        'id',
        'login',
        'is_frozen',
    ]

    def get_serializer_class(self):
        if self.action in ('update', 'partial_update'):
            return self.update_serializer

        return self.serializer_class

    def get_object(self):
        """
        Копия оригинального метода, отличие - в поддержке получения данных
        и по логину тоже
        """
        queryset = self.filter_queryset(self.get_queryset())
        lookup_url_kwarg = self.lookup_url_kwarg or self.lookup_field

        assert lookup_url_kwarg in self.kwargs, (
            'Expected view %s to be called with a URL keyword argument '
            'named "%s". Fix your URL conf, or set the `.lookup_field` '
            'attribute on the view correctly.' %
            (self.__class__.__name__, lookup_url_kwarg)
        )
        filter_value = self.kwargs[lookup_url_kwarg]
        lookup_field = self.lookup_field

        if not (isinstance(filter_value, int) or filter_value.isdigit()):
            # предполагаем тогда, что это login
            lookup_field = 'login'

        filter_kwargs = {lookup_field: filter_value}
        obj = drf_get_object_or_404(queryset, **filter_kwargs)

        self.check_object_permissions(self.request, obj)

        return obj

    def update(self, *args, **kwargs):
        result = super().update(*args, **kwargs)
        staff = self.get_object()
        manager = idm_manager()
        manager.set_is_frozen(
            login=staff.login,
            value=staff.is_frozen,
        )
        notify_staff_about_person.delay(staff_id=staff.id)
        return result
