# coding: utf-8
from __future__ import unicode_literals
import six

from closuretree.models import ClosureModelBase, ClosureModel

from django.db import models
from django.utils.translation import get_language

from plan.staff.querysets import IntranetManager


class I18nBaseModel(models.base.ModelBase):
    def __new__(cls, *args, **kwargs):
        model = super(I18nBaseModel, cls).__new__(cls, *args, **kwargs)
        model._meta._i18n_fields = {}
        for f in model._meta.fields:
            if f.name[-3:] == '_en':
                model._meta._i18n_fields[f.name[:-3]] = True
        return model


class I18nClosureBaseModel(ClosureModelBase, I18nBaseModel):
    pass


class I18nModel(six.with_metaclass(I18nBaseModel, models.Model)):
    """Основное правило: если у сущности есть интернационализованные поля
    (например office.name и office.name_en), то для правильной работы в поле
    native_lang должно быть записано значение языка, на котором записана
    информация в первом поле (для офиса office.name). Таким образом, если
    локаль человека не совпадает с native_lang для сущности, на которую он
    смотрит, то язык для него переключится на интернациональный английский.

    """
    _i18n_fields = {}

    native_lang = models.CharField(max_length=2, default='', blank=True)

    def __getattr__(self, attr_name):

        if not attr_name.startswith('i_'):
            # Если атрибут не найден по цепочке наследования и
            # не начинается с i_, значит его нигде нет. Обработка ситуации,
            # что у родительского класса есть __getattr__ с какой-то
            # дополнительной логикой все сильно усложнит и кажется маловероятной.
            raise AttributeError(attr_name)

        i18n_fields = self._meta._i18n_fields
        native_lang = self.get_native_lang()

        attr_name = attr_name[2:]
        if attr_name in i18n_fields and native_lang:
            lang = get_language()
            if lang:
                lang = lang.split('-')[0]

            # если выбранный язык не совпадает с native_lang, то
            # поле attr_name следует перевести на международный (английский).
            if lang != native_lang:
                translated = getattr(self, attr_name + '_en')
                if translated:
                    return translated

        return getattr(self, attr_name)

    def get_native_lang(self):
        return self.native_lang

    class Meta:
        abstract = True


class SoftDeletedModel(models.Model):
    intranet_status = models.IntegerField(default=1)

    objects = IntranetManager()
    all = models.Manager()

    class Meta:
        abstract = True


class IntranetModel(I18nModel, SoftDeletedModel):
    created_at = models.DateTimeField(editable=False, auto_now_add=True)
    modified_at = models.DateTimeField(editable=False, auto_now=True)

    class Meta:
        abstract = True


class IntranetClosureModel(six.with_metaclass(I18nClosureBaseModel, IntranetModel, ClosureModel)):
    class Meta:
        abstract = True
