from django.db import models
from django.conf import settings
from typing import Set, Iterable

from plan.common.fields import StrictForeignKey

from plan.roles.models import RoleScope
from plan.staff.base import IntranetModel, IntranetClosureModel
from plan.staff.constants import AFFILIATION, DEPARTMENT_ROLES, GENDER, LANG, USER_MODEL
from plan.staff.utils import StaffInflection, calculate_abc_ext


class Department(IntranetClosureModel):
    staff_id = models.PositiveIntegerField(db_index=True)
    parent = models.ForeignKey('self', null=True, blank=True, related_name='children', on_delete=models.CASCADE)
    name = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255, default='')
    short_name = models.CharField(max_length=255, default='')
    short_name_en = models.CharField(max_length=255, default='')
    url = models.CharField(max_length=255, default='', unique=True)

    class Meta:
        db_table = 'intranet_department'

    def __str__(self):
        return self.name

    @property
    def chief(self):
        chief = self.departmentstaff_set.select_related('staff').filter(role=DEPARTMENT_ROLES.CHIEF).first()
        return chief.staff if chief else None

    @property
    def deputy(self):
        deputy = self.departmentstaff_set.select_related('staff').filter(role=DEPARTMENT_ROLES.DEPUTY).first()
        return deputy.staff if deputy else None


class DepartmentStaff(models.Model):
    department = models.ForeignKey('Department', null=True, on_delete=models.CASCADE)
    staff = models.ForeignKey('Staff', null=True, on_delete=models.CASCADE)
    role = models.TextField(choices=DEPARTMENT_ROLES.choices())

    class Meta:
        db_table = 'intranet_departmentstaff'

    def __str__(self):
        return '%s is %s in %s' % (self.staff.login, self.role, self.department)


class StaffQuerySet(models.QuerySet):
    def active(self):
        return self.filter(is_dismissed=False)


class Staff(IntranetModel):
    user = models.OneToOneField(USER_MODEL, null=True, on_delete=models.CASCADE)
    login = models.CharField(max_length=50, db_index=True, default='')
    uid = models.CharField(max_length=16, db_index=True, null=True, unique=True)
    first_name = models.CharField(max_length=50, default='', db_index=True)
    first_name_en = models.CharField(max_length=50, default='', db_index=True)
    last_name = models.CharField(max_length=100, default='', db_index=True)
    last_name_en = models.CharField(max_length=100, default='', db_index=True)
    is_robot = models.BooleanField(default=False)
    affiliation = models.CharField(max_length=32, db_index=True, choices=AFFILIATION.choices())
    gender = models.CharField(max_length=1, choices=GENDER.choices(), default='')
    department = StrictForeignKey('Department', null=True, on_delete=models.CASCADE)
    is_dismissed = models.BooleanField(default=False)
    join_at = models.DateField(null=True)
    quit_at = models.DateField(null=True)
    lang_ui = models.CharField(max_length=2, choices=LANG.choices(), default='')
    work_email = models.CharField(max_length=100, default='', db_index=True)
    staff_id = models.IntegerField(db_index=True, unique=True)
    chief = StrictForeignKey('self', null=True, blank=True, on_delete=models.SET_NULL, related_name='subordinates',)
    is_frozen = models.BooleanField(default=False, db_index=True)
    telegram_account = models.CharField(max_length=100, null=True, blank=True)

    objects = StaffQuerySet.as_manager()

    class Meta(IntranetModel.Meta):
        db_table = 'intranet_staff'

    def get_native_lang(self):
        return settings.DEFAULT_NATIVE_LANGUAGE

    @property
    def is_active(self):
        return not self.is_dismissed

    @property
    def username(self):
        return self.login

    @property
    def date_joined(self):
        return self.join_at

    def get_full_name(self):
        name = ('%s %s' % (self.i_first_name, self.i_last_name)).strip()
        return name if name else self.login

    def get_email(self):
        return self.work_email if self.work_email is not None else ''

    @property
    def email(self):
        return self.work_email

    @property
    def inflections(self):
        return StaffInflection(self)

    @property
    def departments(self):
        if self.department is None:
            return []
        l = list(self.department.get_ancestors(False))
        l.append(self.department)
        return l

    __str__ = get_full_name

    def get_subordinate_departments_with_descendants(self):
        subordinate_departments = Department.objects.filter(
            departmentclosure_parents__parent__departmentstaff__role=DEPARTMENT_ROLES.CHIEF,
            departmentclosure_parents__parent__departmentstaff__staff=self,
        )
        subordinate_departments_ids = subordinate_departments.values_list('id', flat=True)
        return subordinate_departments_ids

    def get_subordinate_direct_departments(self):
        direct_subordinate_departments = set(
            self.departmentstaff_set
                .filter(role=DEPARTMENT_ROLES.CHIEF)
                .values_list('department_id', flat=True)
        )
        return direct_subordinate_departments

    def get_subordinate_staffs(self, only_direct: bool = True) -> Iterable['Staff']:
        """
        Возвращает кверисет подчинённых.
        Если only_direct => возвращаем только прямых
        """
        if only_direct:
            return self.subordinates.active()

        subordinate_departments_ids = self.get_subordinate_departments_with_descendants()
        staffs = Staff.objects.active().filter(department_id__in=subordinate_departments_ids)
        return staffs

    def get_subordinate_logins(self, only_direct: bool = True) -> Set[str]:
        """
        Возвращает список логинов подчиненных
        """
        staff_qs = self.get_subordinate_staffs(only_direct)
        return set(staff_qs.values_list('login', flat=True))

    def get_abc_ext(self):
        return calculate_abc_ext(self.login)


class ServiceScope(IntranetModel):
    service = models.ForeignKey('services.Service', related_name='service_scope')
    role_scope = models.ForeignKey(RoleScope)
    staff_id = models.PositiveIntegerField(db_index=True, null=True, blank=True)

    def __str__(self):
        return f'ServiceScope {self.service.slug} - {self.role_scope.slug}'

    class Meta:
        unique_together = ('service', 'role_scope')
