import logging

from django.db import transaction, IntegrityError
from django.conf import settings
from django.contrib.auth.models import User, Permission
from django.db.models import Q

from ids.registry import registry

from plan.common.models import date_from_str, update_model_fields
from plan.common.utils.tasks import get_last_success_start, lock_task
from plan.roles.models import RoleScope
from plan.staff.constants import DEPARTMENT_ROLES
from plan.staff.models import Department, DepartmentStaff, Staff, ServiceScope
from plan.services.models import Service


log = logging.getLogger(__name__)

GENDER_MAPPING = {
    'male': 'M',
    'female': 'F',
}

STAFF_RANKS_TO_DEPARTMENT_ROLES = {
    'chief': DEPARTMENT_ROLES.CHIEF,
    'deputy': DEPARTMENT_ROLES.DEPUTY,
}

STAFF_OBJECTS_LIMIT = 500


def get_staff_repo(staff_type):
    kwargs = {
        'user_agent': settings.ABC_IDS_USER_AGENT,
        'oauth_token': settings.OAUTH_ROBOT_TOKEN,
        'timeout': settings.STAFF_CHECK_TIMEOUT,
    }

    if settings.CROWDTEST:
        kwargs['host'] = settings.ASSESSORS_STAFF_API_URL

    return registry.get_repository(
        'staff', staff_type,
        **kwargs
    )


class BaseStaffImporter:
    staff_repo = None

    def __init__(self):
        self.lookup = {}
        self.query_template = 'id > %d'

    @classmethod
    def update_batch(cls, batch: list):
        pass

    def get_objects(self, since_at):
        query_template = self.query_template
        if since_at:
            query_template += f' and _meta.modified_at > "{since_at.strftime("%Y-%m-%dT%H:%M:%S%z")}"'

        last_id = 0
        while True:
            self.lookup['_query'] = query_template % last_id
            log.info(f'Import {self.__class__.__name__} with lookup {self.lookup}')
            batch = self.staff_repo.get_nopage(lookup=self.lookup)
            if len(batch) == 0:
                break
            last_id = batch[-1]['id']
            self.update_batch(batch)


class MembershipStaffImporter(BaseStaffImporter):
    staff_repo = get_staff_repo('groupmembership')

    def __init__(self, logins):
        super().__init__()
        self.lookup = {
            '_sort': 'id',
            '_limit': STAFF_OBJECTS_LIMIT,
            '_fields': 'id,person.login,group.url',
            'group.type': 'service,servicerole',
            'person.login': ','.join(logins),
        }

    def get_objects(self, since_at):
        query_template = 'id > %d'
        last_id = 0
        while True:
            self.lookup['_query'] = query_template % last_id
            response = self.staff_repo.get_nopage(lookup=self.lookup)
            if len(response) == 0:
                break
            last_id = response[-1]['id']
            yield from response


class PersonStaffImporter(BaseStaffImporter):
    staff_repo = get_staff_repo('person')

    def __init__(self):
        super().__init__()
        self.lookup = {
            '_sort': 'id',
            '_limit': STAFF_OBJECTS_LIMIT,
            '_fields': ','.join((
                'id',
                'login',
                'uid',
                'official.is_dismissed',
                'official.join_at',
                'official.quit_at',
                'official.is_robot',
                'official.affiliation',
                'department_group.id',
                'name.first.en',
                'name.last.en',
                'name.first.ru',
                'name.last.ru',
                'work_email',
                'language.ui',
                'personal.gender',
                'chief.login',
                'telegram_accounts',
            )),
        }
        if settings.CROWDTEST:
            self.query_template += f' and groups.group.id == {settings.STAFF_CROWDTEST_SYNC_GROUP}'

    def get_objects(self, since_at):
        query_template = self.query_template
        if since_at:
            query_template += f' and _meta.modified_at > "{since_at.strftime("%Y-%m-%dT%H:%M:%S%z")}"'
        last_id = 0
        while True:
            result = {}
            usernames = set()
            department_ids = set()
            self.lookup['_query'] = query_template % last_id
            log.info('Import users with lookup %s', self.lookup)
            persons = self.staff_repo.get_nopage(lookup=self.lookup)
            if len(persons) == 0:
                break
            for person in persons:
                telegram_account = None
                for account in person.pop('telegram_accounts', []):
                    if not account['private']:
                        telegram_account = account['value']
                        break
                person['telegram_account'] = telegram_account

                result[person['uid']] = person
                usernames.add(person['login'])
                department_ids.add(int(person['department_group']['id']))
            last_id = persons[-1]['id']
            yield result, usernames, department_ids

    @staticmethod
    def create_user(uid, auth_user_id, fields):
        if auth_user_id is None:
            user = User.objects.create(
                username=fields['login'],
                first_name=fields.get('first_name', fields['login']),
                last_name=fields.get('last_name', ''),
                email=fields.get('work_email', f'{fields["login"]}@yandex-team.ru'),
                is_active=not fields['is_dismissed'],
            )
            if settings.CROWDTEST:
                permissions = Permission.objects.filter(
                    codename__in=settings.CROWDTEST_DEFAULT_PERMISSIONS,
                )
                user.user_permissions.add(*permissions)
            auth_user_id = user.id
        Staff.objects.create(uid=uid, user_id=auth_user_id, **fields)

    @classmethod
    @transaction.atomic
    def update_batch(cls, staff_persons, staff_usernames, department_ids, delete_duplicates, all_staffs):
        staff_users = {
            user.uid: user
            for user
            in Staff.objects.filter(uid__in=staff_persons.keys())
        }
        auth_users = {
            user.username: user
            for user
            in User.objects.filter(username__in=staff_usernames)
        }
        departments = {
            staff_id: department_id
            for staff_id, department_id
            in Department.objects.filter(staff_id__in=department_ids).values_list('staff_id', 'id')
        }
        staff_id_matches = {
            user.staff_id: user
            for user
            in Staff.objects.filter(staff_id__in=[p['id'] for p in staff_persons.values()])
        }

        for uid, person in staff_persons.items():
            chief_login = person['chief']['login'] if 'chief' in person else None
            fields = {
                'login': person['login'],
                'is_dismissed': person['official']['is_dismissed'],
                'join_at': date_from_str(person['official']['join_at']),
                'quit_at': date_from_str(person['official']['quit_at']),
                'is_robot': person['official']['is_robot'],
                'affiliation': person['official']['affiliation'],
                'first_name': person['name']['first']['ru'],
                'first_name_en': person['name']['first']['en'],
                'last_name': person['name']['last']['ru'],
                'last_name_en': person['name']['last']['en'],
                'gender': GENDER_MAPPING.get(person['personal']['gender'], ''),
                'lang_ui': person['language']['ui'],
                'work_email': person['work_email'],
                # Департамент может не существовать на момент синка, обновим при следующем полном синке
                'department_id': departments.get(int(person['department_group']['id'])),
                'staff_id': person['id'],
                # Если стафф руководителя ещё не засинкали, то обновим при следующем полном синке
                'chief': all_staffs.get(chief_login),
                'telegram_account': person['telegram_account'],
            }
            auth_fields = {
                'is_active': not fields['is_dismissed']
            }
            if uid in staff_users:
                update_model_fields(auth_users[fields['login']], auth_fields)
                update_model_fields(staff_users[uid], fields)
            else:
                if person['id'] in staff_id_matches:
                    if delete_duplicates:
                        staff_id_matches[person['id']].delete()
                    else:
                        raise IntegrityError(message=f'Duplicate staff with staff_id {person["id"]} detected')
                auth_user_id = auth_users.get(fields['login'])
                if auth_user_id:
                    auth_user_id = auth_user_id.id
                cls.create_user(uid=uid, auth_user_id=auth_user_id, fields=fields)


class ServiceStaffImporter(BaseStaffImporter):
    staff_repo = get_staff_repo('group')

    def __init__(self):
        super().__init__()
        self.lookup = {
            '_sort': 'id',
            'type': 'service',
            '_limit': STAFF_OBJECTS_LIMIT,
            '_fields': ','.join((
                'id',
                'service.id',
            )),
        }

    @staticmethod
    def map_fields(data):
        staff_id = data['id']
        service_id = data.get('service', {}).get('id')
        return staff_id, service_id

    @classmethod
    @transaction.atomic
    def update_batch(cls, batch):
        mapped_data = {}
        for data in batch:
            staff_id, service_id = cls.map_fields(data)
            mapped_data[service_id] = staff_id

        services = Service.objects.filter(pk__in=mapped_data.keys())
        for service in services:
            if service.staff_id != mapped_data[service.id]:
                service.staff_id = mapped_data[service.id]
                service.save(update_fields=['staff_id'])


class ServiceScopeStaffImporter(BaseStaffImporter):
    staff_repo = get_staff_repo('group')

    def __init__(self):
        super().__init__()
        self.lookup = {
            '_sort': 'id',
            'type': 'servicerole',
            '_limit': STAFF_OBJECTS_LIMIT,
            '_fields': ','.join((
                'id',
                'parent.service.id',
                'role_scope',
            )),
        }

        self.role_scope_dict = {
            role_scope.slug: role_scope.id
            for role_scope in RoleScope.objects.all()
        }

        self.services_ids = Service.objects.values_list('id', flat=True)

    def map_fields(self, data):
        staff_id = data['id']
        parent = data.pop('parent', None)
        service_id = None
        if parent is not None:
            service_id = parent['service']['id']
        role_scope_slug = data['role_scope']

        if role_scope_slug not in self.role_scope_dict:
            log.info(f'Import services scope with lookup {self.lookup}, {role_scope_slug} RoleScope.DoesNotExist')
            return None, staff_id

        if service_id not in self.services_ids:
            log.info(f'Import services scope with lookup {self.lookup}, {service_id} Service.DoesNotExist')
            return None, staff_id

        role_scope_id = self.role_scope_dict[role_scope_slug]
        return (service_id, role_scope_id), staff_id

    @transaction.atomic
    def update_batch(self, batch):
        mapped_data = {}
        for data in batch:
            service_scope_tuple, staff_id = self.map_fields(data)
            if service_scope_tuple is None:
                continue

            mapped_data[service_scope_tuple] = staff_id

        condition = Q()
        for data in mapped_data.keys():
            condition |= Q(service_id=data[0], role_scope_id=data[1])

        scopes = ServiceScope.objects.filter(condition)
        for scope in scopes:
            service_scope_tuple = (scope.service_id, scope.role_scope_id)
            if service_scope_tuple in mapped_data.keys():
                staff_id = mapped_data.pop(service_scope_tuple)
                if scope.staff_id != staff_id:
                    scope.staff_id = staff_id
                    scope.save(update_fields=['staff_id'])

        # все остальные скоупы, что не проапдейтили - создаём
        # TODO: этот кусок можно будет выпилить после ABC-8394
        # тк больше не должно быть скоупов, которые придут из стафф-апи и о которых мы ничего не знаем
        new_scopes = []
        for service_scope_tuple, staff_id in mapped_data.items():
            fields = ServiceScope(
                staff_id=staff_id,
                service_id=service_scope_tuple[0],
                role_scope_id=service_scope_tuple[1]
            )
            new_scopes.append(fields)

        if new_scopes:
            ServiceScope.objects.bulk_create(new_scopes)


class DepartmentStaffImporter(BaseStaffImporter):
    staff_repo = get_staff_repo('group')

    def __init__(self):
        super().__init__()
        self.lookup = {
            'type': 'department',
            '_limit': STAFF_OBJECTS_LIMIT,
            '_sort': 'id',
            '_fields': ','.join((
                'id',
                'url',
                'parent.id',
                'parent.url',
                'department.name',
                'department.heads.role',
                'department.heads.person.login',
            )),
        }
        if settings.CROWDTEST:
            self.query_template += f' and (ancestors.id == {settings.STAFF_CROWDTEST_SYNC_ROOT}' \
                                   f' or id == {settings.STAFF_CROWDTEST_SYNC_ROOT})'

    @staticmethod
    def department_dict(staff_ids_list):
        return {
            department.staff_id: department
            for department
            in Department.objects.filter(staff_id__in=staff_ids_list).prefetch_related(
                'departmentstaff_set',
                'departmentstaff_set__staff',
            )
        }

    @classmethod
    def prepare_staff_id_to_department(cls, staff_id_set, parent_urls, mapped_data):
        values = cls.department_dict(staff_id_set)
        staff_id_to_create = staff_id_set - values.keys()

        for staff_id in staff_id_to_create:
            data = mapped_data.pop(staff_id, {})
            parent_staff_id = data.pop('parent_id', None)

            if parent_staff_id is not None and parent_staff_id in values:
                data['parent'] = values[parent_staff_id]

            if 'url' not in data:
                data['url'] = parent_urls[staff_id]

            department = Department.objects.create(staff_id=staff_id, **data)
            values[staff_id] = department

            if parent_staff_id in staff_id_to_create:
                # если parent должен досоздаться в этом же цикле, то проапдейтим его после
                # поэтому возвращаем департамент в mapped_data только с parent_id
                mapped_data[staff_id] = {'parent_id': parent_staff_id}

        return values, mapped_data

    @staticmethod
    def map_fields(data):
        staff_id = data['id']
        parent_staff_id = None
        parent = data.get('parent', None)
        if parent is not None:
            parent_staff_id = parent.get('id', None)
        fields = {
            'parent_id': parent_staff_id,
            'name': data['department']['name']['full']['ru'],
            'name_en': data['department']['name']['full']['en'],
            'short_name': data['department']['name']['short']['ru'],
            'short_name_en': data['department']['name']['short']['en'],
            'url': data['url'],
        }
        heads = set()
        staff_heads = data['department'].get('heads', [])
        for head in staff_heads:
            # Пока сохраняем только руководителя и заместителя без фильтрации по признаку уволенности
            if head['role'] not in STAFF_RANKS_TO_DEPARTMENT_ROLES:
                continue
            heads.add((head['person']['login'], STAFF_RANKS_TO_DEPARTMENT_ROLES[head['role']]))
        return staff_id, fields, heads, parent

    @classmethod
    @transaction.atomic
    def update_batch(cls, batch):
        mapped_data = {}
        departments_heads = {}
        parents_staff_id_set = set()
        parents_urls = {}
        for data in batch:
            staff_id, data, heads, parent = cls.map_fields(data)
            mapped_data[staff_id] = data
            departments_heads[staff_id] = heads
            parent_id = data['parent_id']
            if parent_id:
                parents_staff_id_set.add(parent_id)
                parents_urls[parent_id] = parent.get('url')

        staff_id_to_department, mapped_data = cls.prepare_staff_id_to_department(
            parents_staff_id_set, parents_urls, mapped_data
        )
        existing_departments = cls.department_dict(mapped_data.keys())

        for staff_id, data in mapped_data.items():
            try:
                cls.update_department(staff_id, data, staff_id_to_department, existing_departments, departments_heads)
            except Exception:
                log.exception('Error when updating department with staff id %s', staff_id)

    @classmethod
    def update_department(cls, staff_id, data, staff_id_to_department, existing_departments, departments_heads):
        parent_staff_id = data.pop('parent_id')
        if parent_staff_id:
            data['parent'] = staff_id_to_department[parent_staff_id]
        else:
            data['parent'] = None

        data['native_lang'] = 'ru'

        if staff_id in existing_departments:
            department = existing_departments[staff_id]
            update_model_fields(department, data)
        else:
            department = Department.objects.create(staff_id=staff_id, **data)
        cls.sync_department_heads(department, departments_heads[department.staff_id])

    @classmethod
    def sync_department_heads(cls, department, heads):
        to_remove_ids = []
        for head in department.departmentstaff_set.all():
            value = (head.staff.login, head.role)
            if value not in heads:
                to_remove_ids.append(head.id)
            else:
                heads.remove(value)
        if to_remove_ids:
            DepartmentStaff.objects.filter(id__in=to_remove_ids).delete()
        new_objects = []
        for login, role in sorted(heads):
            try:
                # Руководитель департамента меняется не часто, кажется лучше сходить в базу,
                # чем зря выбирать все объекты в память
                user = Staff.objects.get(login=login)
                new_objects.append(DepartmentStaff(staff=user, department=department, role=role))
            except Staff.DoesNotExist:
                log.exception('In heads of deparment %s user with missing login %s', department.id, login)
        DepartmentStaff.objects.bulk_create(new_objects)


@lock_task
def sync_staff_users(force=False, delete_duplicates=False):
    log.info('Importing users from Staff API')
    since_at = None
    if not force:
        since_at = get_last_success_start('sync_staff_users')
    importer = PersonStaffImporter()
    staff_persons = importer.get_objects(since_at)

    all_staffs = {
        s.login: s
        for s in Staff.objects.active()
    }
    for persons, usernames, department_ids in staff_persons:
        importer.update_batch(persons, usernames, department_ids, delete_duplicates, all_staffs)


@lock_task
def sync_services_scope(force=False):
    log.info('Importing services scope from Staff API')
    since_at = None
    if not force:
        since_at = get_last_success_start('sync_services_scope')

    # два синка:
    # 1 синкает серисы (type=service) и кладём id в сервис
    # 2 синкает сервисные скоупы (type=servicerole)  и кладём id в специальную модель
    services_importer = ServiceStaffImporter()
    scope_importer = ServiceScopeStaffImporter()
    services_importer.get_objects(since_at)
    scope_importer.get_objects(since_at)


@lock_task
def sync_departments(force=False):
    since_at = None
    if not force:
        since_at = get_last_success_start('sync_departments')

    importer = DepartmentStaffImporter()
    importer.get_objects(since_at)
