from pymorphy2 import MorphAnalyzer

from django.conf import settings
from django.utils.translation import get_language

from plan.internal_roles.models import InternalRole
from plan.staff.constants import GENDER

forms = {
    'subjective': 'nomn',
    'genitive': 'gent',
    'dative': 'datv',
    'accusative': 'accs',
    'ablative': 'ablt',
    'prepositional': 'loct',
}

genders = {
    GENDER.MALE: 'masc',
    GENDER.FEMALE: 'femn',
}


class StaffInflection(object):
    def __init__(self, staff):
        self.staff = staff
        self.morph = MorphAnalyzer()

    def inflect(self, form_name):
        if form_name in forms:
            return self._inflect(forms[form_name])
        else:
            raise AttributeError('Unknown inflection form')

    def select_best(self, parse):
        gender_tags = [p.tag.gender for p in parse]
        if genders[self.staff.gender] in gender_tags:
            return parse[gender_tags.index(genders[self.staff.gender])]
        else:
            return parse[0]

    def inflect_name(self, name, form):
        best_parse = self.select_best(self.morph.parse(name))
        inflections = best_parse.inflect({form})
        result = name
        if 'LATN' not in best_parse.tag and inflections and len(inflections) > 0:
            result = inflections[0].capitalize()
        return result

    def _inflect(self, form):
        inflected_first_name = self.inflect_name(self.staff.i_first_name, form)
        inflected_last_name = self.inflect_name(self.staff.i_last_name, form)
        return f'{inflected_first_name} {inflected_last_name}'

    def __getattr__(self, attr_name):
        return self.inflect(attr_name)


def get_i_field(model_dict, field, prefix=''):
    en_field = prefix + field + '_en'
    nl_field = prefix + 'native_lang'
    if en_field in model_dict and nl_field in model_dict:
        lang = get_language()
        pos = lang.find('-')
        if pos > 0:
            lang = lang[:pos]
        if lang != model_dict[nl_field]:
            return model_dict[en_field]
    return model_dict[prefix + field]


def calculate_abc_ext(login):
    codes_roles = [settings.FULL_ACCESS, settings.SERVICES_VIEWER, settings.OWN_ONLY_VIEWER]
    access = InternalRole.objects.filter(
        staff__login=login,
        role__in=codes_roles
    ).values_list('role', flat=True)

    abc_ext = []
    if settings.FULL_ACCESS in access or not access:
        return abc_ext

    if settings.SERVICES_VIEWER in access:
        abc_ext.append(settings.HAS_SERVICES_VIEWER)
        return abc_ext

    if settings.OWN_ONLY_VIEWER in access:
        abc_ext.append(settings.HAS_OWN_ONLY_VIEWER)

    return abc_ext
