import django_filters

from plan.services.models import Service
from plan.api.filters import (
    PlanFilterSet,
    CustomModelChoiceFilter,
    CustomModelMultipleChoiceFilter,
)
from plan.suspicion.models import (
    Complaint,
    ServiceAppealIssue,
    ServiceIssue,
    Issue,
)


class ComplaintFilter(PlanFilterSet):
    class Meta:
        model = Complaint
        fields = {
            'service': ['exact', 'in'],
        }


class ServiceAppealIssueFilter(PlanFilterSet):

    DIRECT = ['direct']
    HIERARCHY = ['hierarchy', 'true', 'True']

    service = CustomModelChoiceFilter(
        queryset=Service.objects.active(),
        field_name='service_issue__service',
        distinct=True,
    )
    only_mine = django_filters.CharFilter(method='filter_only_mine')
    kind = django_filters.CharFilter(method='filter_appeal_kind')

    def filter_only_mine(self, queryset, name, value):
        if value in self.DIRECT:
            queryset = queryset.for_services_of_person(self.request.person, directly=True).for_active_services()
        elif value in self.HIERARCHY:
            queryset = queryset.for_services_of_person(self.request.person).for_active_services()
        return queryset

    def filter_appeal_kind(self, queryset, name, value):
        if value == 'active':
            queryset = queryset.active()
        elif value == 'inactive':
            queryset = queryset.dead()
        return queryset

    class Meta:
        model = ServiceAppealIssue
        fields = ['service', 'only_mine', 'state']


class ServiceIssueFilterSet(PlanFilterSet):
    service = CustomModelMultipleChoiceFilter(
        queryset=Service.objects.active(),
        field_name='service',
    )

    service__slug = CustomModelMultipleChoiceFilter(
        field_name='service__slug',
        queryset=Service.objects.active(),
        to_field_name='slug'
    )

    issue = CustomModelMultipleChoiceFilter(
        field_name='issue__code',
        queryset=Issue.objects.all(),
        to_field_name='code',
    )

    class Meta:
        model = ServiceIssue
        fields = {
            'service': ['exact', 'in'],
            'state': ['exact', 'in'],
            'issue': ['exact', 'in'],
        }
