import collections

from django.db.models import Min

from rest_framework import serializers, fields

from plan.cabinet.serializers import StaffSerializer, ServiceSerializer
from plan.api.fields import MappingField, IntegerInterfacedSerializerField
from plan.api.serializers import ModelSerializer, BasePlanModelSerializer
from plan.api.serializers import CompactServiceSerializer
from plan.services import permissions
from plan.suspicion.constants import ISSUE_GROUP_SERIALIZER_META_FIELDS, SERVICE_TRAFFIC_STATUS_SERIALIZER_META_FIELDS
from plan.suspicion.models import (
    Complaint,
    ExecutionStep,
    Issue,
    IssueGroup,
    IssueGroupThreshold,
    ServiceTrafficStatus,
    ServiceIssue,
    ServiceExecutionAction,
    ServiceAppealIssue
)
from plan.services.models import Service


HUNDRED_PERCENT = 100
ROUND_PERCENTS_TO = 2


class ComplaintSerializer(ModelSerializer):
    class Meta:
        model = Complaint
        fields = ['created_at', 'message', 'service']


class IssueGroupSerializer(ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = IssueGroup
        fields = ISSUE_GROUP_SERIALIZER_META_FIELDS


class ServiceTrafficStatusSerializer(ModelSerializer):
    group = IssueGroupSerializer(source='issue_group')

    class Meta:
        model = ServiceTrafficStatus
        fields = SERVICE_TRAFFIC_STATUS_SERIALIZER_META_FIELDS


class ExecutionSerializer(ModelSerializer):
    name = MappingField({'ru': 'execution__name', 'en': 'execution__name_en'})
    is_critical = fields.BooleanField(source='execution.is_critical')
    fields_mapping_ = {
        'name': ('name', 'name_en'),
        'is_critical': ('execution__is_critical',),
    }

    class Meta:
        model = ServiceExecutionAction
        fields = ['name', 'is_critical', 'should_be_applied_at', 'applied_at']


class ServiceIssueSerializer(ModelSerializer):
    name = MappingField({'ru': 'issue__name', 'en': 'issue__name_en'})
    description = MappingField({'ru': 'issue__description', 'en': 'issue__description_en'})
    recommendation = MappingField({'ru': 'issue__recommendation', 'en': 'issue__recommendation_en'})
    can_be_appealed = fields.BooleanField(source='issue.can_be_appealed', required=False)
    is_appealed = fields.SerializerMethodField()
    on_review = fields.SerializerMethodField(read_only=True)
    fields_mapping_ = {
        'name': ('issue__name', 'issue__name_en'),
        'description': ('issue__description', 'issue__description_en'),
        'recommendation': ('issue__recommendation', 'issue__recommendation_en'),
        'can_be_appealed': ('issue__can_be_appealed',),
        'is_appealed': ('state',),
        'on_review': ('state',),
    }

    class Meta:
        model = ServiceIssue
        fields = [
            'id', 'name', 'description', 'recommendation',
            'can_be_appealed', 'is_appealed', 'on_review',
            'context',
        ]

    @staticmethod
    def get_is_appealed(obj):
        return obj.state == ServiceIssue.STATES.APPEALED

    @staticmethod
    def get_on_review(obj):
        return obj.state == ServiceIssue.STATES.REVIEW


class ServiceIssueSerializerWithExecution(ServiceIssueSerializer):
    execution = ExecutionSerializer(source='action')
    code = fields.SerializerMethodField(read_only=True)
    weight = fields.SerializerMethodField(read_only=True)

    class Meta:
        model = ServiceIssueSerializer.Meta.model
        fields = ServiceIssueSerializer.Meta.fields + ['execution', 'code', 'weight']

    def get_code(self, service_issue):
        return service_issue.issue.code

    def get_weight(self, service_issue):
        return service_issue.issue.weight


class MainServiceIssueSerializer(BasePlanModelSerializer):
    execution = ExecutionSerializer(source='action')
    code = fields.SerializerMethodField(read_only=True)

    class Meta:
        model = ServiceIssue
        fields = ['execution', 'code']

    def get_code(self, service_issue):
        if service_issue.issue:
            return service_issue.issue.code
        return service_issue.issue_group.code


class IssueGroupThresholdSerializer(BasePlanModelSerializer):
    weight = fields.SerializerMethodField()  # проценты, при котором активируется
    is_current = fields.SerializerMethodField()  # является текущим
    is_next = fields.SerializerMethodField()  # является следующим по красноте
    execution = fields.SerializerMethodField()

    class Meta:
        model = IssueGroupThreshold
        fields = ['level', 'weight', 'is_current', 'is_next', 'execution']

    def get_weight(self, threshold):
        return round(threshold.threshold * HUNDRED_PERCENT, ROUND_PERCENTS_TO)

    def get_is_next(self, threshold):
        return False

    def get_is_current(self, threshold):
        return False

    def get_execution(self, threshold):
        pass


class IssueGroupWithIssuesSerializer(IssueGroupSerializer):
    issues_count = fields.SerializerMethodField()
    issues = ServiceIssueSerializerWithExecution(source='service_issues', many=True)
    summary = fields.SerializerMethodField()
    description = MappingField({'ru': 'description', 'en': 'description_en'})
    recommendation = MappingField({'ru': 'recommendation', 'en': 'recommendation_en'})
    fields_mapping_ = {
        'description': ('description', 'description_en'),
        'recommendation': ('recommendation', 'recommendation_en'),
    }

    class Meta:
        model = IssueGroup
        fields = ['name', 'issues_count', 'issues', 'summary', 'description', 'recommendation', 'code']

    @staticmethod
    def get_issues_count(obj):
        return len(obj.service_issues)

    def _get_chain_to_first_and_last_actions(self, service_issue_group, chains):
        all_actions = (
            ServiceExecutionAction.objects.filter(
                service_issue=service_issue_group,
                execution_chain_id__in=chains,
            )
            .order_by('should_be_applied_at')
            .select_related('execution')
        )
        first_future_actions = {}
        last_past_actions = {}
        for action in all_actions:
            chain = action.execution_chain_id
            if action.applied_at:
                last_past_actions[chain] = action
            elif chain not in first_future_actions:
                first_future_actions[chain] = action
        return first_future_actions, last_past_actions

    def _get_first_potential_execution_for_chain(self, chain):
        step = (
            ExecutionStep.objects.filter(execution_chain_id=chain)
            .order_by('apply_after')
            .select_related('execution')
            .first()
        )
        if step:
            return step.execution

    def get_summary(self, issue_group):
        result = {}

        service_id = self.context['service_id']
        traffic_status = ServiceTrafficStatus.objects.filter(service=service_id, issue_group=issue_group).first()

        # фронт ждёт информацию в процентах
        percent = traffic_status.current_weight * HUNDRED_PERCENT if traffic_status is not None else 0
        result['current_weight'] = round(percent)

        thresholds = issue_group.thresholds.select_related('chain').order_by('level')
        result['thresholds'] = IssueGroupThresholdSerializer(thresholds, many=True).data
        level_to_chain = {
            threshold.level: threshold.chain_id
            for threshold in thresholds
        }

        last_threshold_before_current = None
        service_issue_group = issue_group.service_issuegroup.active().filter(service=service_id)
        chain_to_first_future_action, chain_to_last_past_action = self._get_chain_to_first_and_last_actions(
            service_issue_group,
            level_to_chain.values()
        )

        for threshold in result['thresholds']:
            if result['current_weight'] > threshold['weight']:
                threshold['is_current'] = True
                # ищем следующее наказание
                chain = level_to_chain[threshold['level']]
                action = chain_to_first_future_action.get(chain) or chain_to_last_past_action.get(chain)
                if action:
                    threshold['execution'] = ExecutionSerializer(action).data
                break

            if result['current_weight'] < threshold['weight']:
                last_threshold_before_current = threshold

        if last_threshold_before_current:
            last_threshold_before_current['is_next'] = True
            chain = level_to_chain[last_threshold_before_current['level']]
            if chain in chain_to_first_future_action:
                next_execution = chain_to_first_future_action.get(chain).execution
            else:
                next_execution = self._get_first_potential_execution_for_chain(chain)
            if next_execution:
                last_threshold_before_current['execution'] = {
                    'name': {
                        'ru': next_execution.name,
                        'en': next_execution.name_en,
                    }
                }

        return result

    def to_representation(self, instance):
        result = super(IssueGroupWithIssuesSerializer, self).to_representation(instance)
        current_issues = {}
        for issue in result['issues']:
            current_issues[issue['code']] = issue

        all_issues = Issue.objects.active().filter(issue_group=instance).order_by('-weight')
        max_weight = sum(issue.weight for issue in all_issues)

        for issue in result['issues']:
            issue['weight'] = round(HUNDRED_PERCENT * issue['weight'] / max_weight, ROUND_PERCENTS_TO)

        for issue in all_issues:
            issue_is_appealed = current_issues.get(issue.code, {}).get('is_appealed', False)
            if issue.code not in current_issues or issue_is_appealed:
                result['issues'].append(
                    {
                        'name': {'ru': issue.name, 'en': issue.name_en},
                        'description': {'ru': issue.description, 'en': issue.description_en},
                        'recommendation': {'ru': issue.recommendation, 'en': issue.recommendation_en},
                        'code': issue.code,
                        'weight': round(HUNDRED_PERCENT * issue.weight / max_weight, ROUND_PERCENTS_TO),
                        'is_appealed': False,
                    }
                )
                if issue_is_appealed:
                    result['issues_count'] -= 1
                    result['issues'].remove(current_issues[issue.code])
                    result['issues'][-1]['is_appealed'] = True

        return result


def main_service_issue(service_id):
    action = (
        ServiceExecutionAction.objects
        .filter(service_issue__in=ServiceIssue.objects.problem().filter(service__id=service_id))
        .order_by('-execution__is_critical', 'should_be_applied_at')
        .select_related('service_issue', 'execution', 'service_issue__issue')
        .first()
    )
    if action is None:
        return None
    return action_to_service_issue(action)


def grouped_service_issues(service_id):
    result = []
    service_execution_action_list = (
        ServiceExecutionAction.objects
        .filter(
            service_issue__service__id=service_id,
            service_issue__issue__isnull=False,
            service_issue__state=ServiceIssue.STATES.ACTIVE
        )
        .order_by('service_issue__issue__issue_group_id', 'service_issue', 'should_be_applied_at')
        .distinct('service_issue__issue__issue_group_id', 'service_issue')
        .values_list('pk', flat=True)
    )

    actions = (
        ServiceExecutionAction.objects
        .filter(pk__in=service_execution_action_list)
        .select_related('execution', 'service_issue')
        .order_by('service_issue__issue__issue_group_id', 'should_be_applied_at')
    )

    # Нас интересует только последний ServiceExecutionAction для каждого ServiceIssue
    service_issue_id_to_action = {action.service_issue.id: action for action in actions}

    service_issues = (
        ServiceIssue.objects
        .alive()
        .filter(issue__isnull=False, service__id=service_id)
        .select_related('issue')
        .annotate(Min('execution_actions__should_be_applied_at'))
        .order_by('issue__issue_group_id', '-issue__weight')
    )

    group_id_to_service_issues = collections.defaultdict(list)
    for service_issue in service_issues:
        service_issue.action = service_issue_id_to_action.get(service_issue.id)
        group_id_to_service_issues[service_issue.issue.issue_group_id].append(service_issue)

    for group in IssueGroup.objects.order_by('code'):
        # service_issues здесь НЕ поле модели
        group.service_issues = group_id_to_service_issues[group.id]
        result.append(group)

    return result


def action_to_service_issue(action):
    issue = action.service_issue
    issue.action = action
    return issue


class ServiceAppealIssueSerializer(ModelSerializer):
    requester = StaffSerializer(read_only=True)
    approvers = serializers.SerializerMethodField(read_only=True)
    user_can_approve = serializers.SerializerMethodField(read_only=True)
    issue = IntegerInterfacedSerializerField(
        queryset=ServiceIssue.objects.all(),
        source='service_issue',
        serializer=ServiceIssueSerializer,
    )
    service = ServiceSerializer(source='service_issue.service', read_only=True)

    class Meta:
        model = ServiceAppealIssue
        fields = [
            'id',
            'requester',
            'message',
            'created_at',
            'approvers',
            'issue',
            'service',
            'user_can_approve',
            'state',
        ]

    @staticmethod
    def get_approvers(obj):
        return StaffSerializer(approvers_for_service(obj.service_issue.service), many=True).data

    def get_user_can_approve(self, obj):
        person = self.context['request'].person
        return permissions.can_approve_appeal(obj.service_issue.service, person)


def approvers_for_service(service):
    return [member.staff for member in service.get_responsible(parents=True, include_self=False).order_by('pk').select_related('staff')]


class IssueSerializer(ModelSerializer):
    issue_group = IssueGroupSerializer(read_only=True)

    class Meta:
        model = Issue
        fields = (
            'id', 'name', 'name_en',
            'code', 'issue_group',
            'description', 'description_en',
            'recommendation', 'recommendation_en',
        )


class ServiceIssueViewSerializer(ModelSerializer):
    issue = IntegerInterfacedSerializerField(
        queryset=Issue.objects.all(),
        serializer=IssueSerializer,
    )
    issue_group = IssueGroupSerializer(read_only=True)
    service = IntegerInterfacedSerializerField(
        queryset=Service.objects.all(),
        serializer=CompactServiceSerializer,
    )

    class Meta:
        model = ServiceIssue
        fields = (
            'id', 'state',
            'context', 'service',
            'issue', 'issue_group',
            'percentage_of_completion',
        )


class IssueBulkSerializer(serializers.Serializer):
    code = fields.CharField()
    percentage_of_completion = fields.FloatField(required=False, default=1)
    context = fields.JSONField(required=False)


class ServiceIssuesBulkSerializer(serializers.Serializer):
    service = fields.CharField()
    group_code = fields.CharField()
    issues = IssueBulkSerializer(many=True, allow_null=True)
