import logging

from collections import defaultdict

from django.db.models import Q
from django_fsm import TransitionNotAllowed

from rest_framework import mixins, status, viewsets
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.viewsets import GenericViewSet
from rest_framework.permissions import IsAuthenticated
from rest_framework.serializers import ValidationError

from plan.api.mixins import (
    DefaultFieldsMixin,
    OrderingMixin,
    SelectOnlyFieldsMixin,
)
from plan.api.exceptions import BadRequest, PermissionDenied
from plan.services.models import ServiceNotification, Service
from plan.suspicion.api.serializers import (
    ComplaintSerializer,
    ServiceAppealIssueSerializer,
    ServiceIssuesBulkSerializer,
    ServiceIssueViewSerializer,
)
from plan.suspicion.api.filters import (
    ComplaintFilter,
    ServiceAppealIssueFilter,
    ServiceIssueFilterSet,
)
from plan.suspicion.models import (
    Complaint, ServiceAppealIssue,
    Issue, ServiceIssue, IssueGroup,
)
from plan.api.base import ABCPagination, ABCCursorPagination
from plan.api.permissions import TvmAuthenticated
from plan.suspicion.constants import ServiceAppealIssueStates

logger = logging.getLogger(__name__)


class ComplaintView(mixins.CreateModelMixin,
                    mixins.ListModelMixin,
                    GenericViewSet):
    queryset = Complaint.objects.all()
    serializer_class = ComplaintSerializer
    filter_class = ComplaintFilter
    pagination_class = ABCPagination
    ordering = '-created_at'

    def perform_create(self, serializer):
        serializer.save(author=self.request.user.staff)


class V4ComplaintView(DefaultFieldsMixin, ComplaintView):
    pagination_class = ABCCursorPagination
    default_fields = ['created_at', 'message', 'service']


class ServiceAppealPermission(IsAuthenticated):
    def has_permission(self, request, view):
        if view.action not in ('reject', 'approve'):
            return True
        appeal = (
            view.get_queryset().select_related('service_issue', 'service_issue__service')
            .get(pk=view.kwargs['pk'])
        )
        return appeal.can_be_approved_by(request.person)


class ServiceAppealIssueView(OrderingMixin,
                             mixins.CreateModelMixin,
                             mixins.ListModelMixin,
                             GenericViewSet):
    queryset = ServiceAppealIssue.objects.all()
    serializer_class = ServiceAppealIssueSerializer
    filter_class = ServiceAppealIssueFilter
    pagination_class = ABCPagination
    permission_classes = [ServiceAppealPermission]

    def perform_create(self, serializer):
        person = self.request.person
        if self.queryset.filter(
                state__in=[
                    ServiceAppealIssueStates.APPROVED,
                    ServiceAppealIssueStates.REQUESTED,
                ],
                service_issue_id=self.request.data['issue'],
                requester=person.staff,
        ).exists():
            raise ValidationError(detail='This issue already has appeal by this user')
        appeal = serializer.save(requester=person.staff)
        if appeal.can_be_approved_by(person):
            appeal.approve(approver=person.staff)
        else:
            appeal.send_mail_async(ServiceNotification.NEW_APPEAL_CREATED)

    @action(methods=['post'], detail=True)
    def reject(self, request, pk=None):
        appeal = self.get_object()
        try:
            appeal.reject(rejecter=self.request.person.staff)
        except TransitionNotAllowed:
            raise BadRequest(message={
                'ru': 'Решение по этой апелляции уже было принято',
                'en': 'This appeal was already processed',
            })

        return Response(status=204)

    @action(methods=['post'], detail=True)
    def approve(self, request, pk=None):
        appeal = self.get_object()
        try:
            appeal.approve(approver=self.request.person.staff)
        except TransitionNotAllowed:
            raise BadRequest(message={
                'ru': 'Решение по этой апелляции уже было принято',
                'en': 'This appeal was already processed',
            })

        return Response(status=204)


class V4ServiceAppealIssueView(DefaultFieldsMixin, ServiceAppealIssueView):
    pagination_class = ABCCursorPagination
    default_fields = [
        'id',
        'state',

        'service.id',
        'service.slug',
    ]


class ServiceIssuePermissions(TvmAuthenticated):
    def has_object_permission(self, request, view, obj):
        if view.action in ('list', 'retrieve'):
            return True
        elif view.action in ('update', 'partial_update', 'destroy'):
            service_ticket = getattr(request, 'tvm_service_ticket', None)
            if service_ticket and int(service_ticket.src) in obj.issue.issue_group.allowed_tvm_ids:
                return True

        return False


class ServiceIssueView(
    DefaultFieldsMixin, SelectOnlyFieldsMixin,
    OrderingMixin, viewsets.ModelViewSet,
):
    serializer_class = ServiceIssueViewSerializer
    filterset_class = ServiceIssueFilterSet
    permission_classes = (ServiceIssuePermissions, )
    queryset = ServiceIssue.objects.problem()
    pagination_class = ABCCursorPagination
    TVM_ALLOWED_METHODS = {'GET', 'POST', 'PATCH', 'DELETE', 'PUT'}
    default_fields = [
        'id',
        'context',
        'percentage_of_completion',
    ]

    def check_group_perm(self, request, issue):
        service_ticket = getattr(request, 'tvm_service_ticket', None)
        if not service_ticket:
            raise PermissionDenied('Only TVM authorization is supported in this view')
        if int(service_ticket.src) not in issue.issue_group.allowed_tvm_ids:
            raise PermissionDenied(
                f'TVM id {request.tvm_service_ticket.src} have no permissions to edit this issue group'
            )

    def create(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        self.check_group_perm(request=request, issue=serializer.validated_data['issue'])
        self.perform_create(serializer)
        headers = self.get_success_headers(serializer.data)
        return Response(serializer.data, status=status.HTTP_201_CREATED, headers=headers)

    def _get_services(self, data: dict) -> dict:
        ids = set()
        slugs = set()
        for service_issues in data:
            service = service_issues['service']
            if service.isdigit():
                ids.add(service)
            else:
                slugs.add(service)
        result = {}
        for service in Service.objects.filter(
            Q(pk__in=ids) | Q(slug__in=slugs)
        ):
            result[service.slug] = service.id
            result[str(service.id)] = service.id

        return result

    def _get_groups(self, data) -> dict:
        return {
            group.code: group for group in
            IssueGroup.objects.filter(code__in={item['group_code'] for item in data})
        }

    def _get_issues(self, data) -> dict:
        return {
            issue.code: issue for issue in
            Issue.objects.filter(
                code__in={
                    item['code'] for service_issues in data
                    for item in service_issues['issues']
               }
            ).select_related('issue_group')
        }

    @action(methods=['put'], url_path='bulk', url_name='bulk', detail=False)
    def bulk_update(self, request):
        serializer = ServiceIssuesBulkSerializer(data=request.data, many=True)
        serializer.is_valid(raise_exception=True)

        services = self._get_services(data=serializer.data)
        issues = self._get_issues(data=serializer.data)
        groups = self._get_groups(data=serializer.data)

        if not getattr(request, 'tvm_service_ticket', None):
            raise PermissionDenied('Only requests with tvm2 service ticket are accepted')

        if any(
            int(request.tvm_service_ticket.src) not in group.allowed_tvm_ids
            for group in groups.values()
        ):
            raise PermissionDenied(
                f'TVM id {request.tvm_service_ticket.src} have no permissions to edit this issue group'
            )

        to_create = []
        to_delete = []
        current_issues = ServiceIssue.objects.filter(
            service_id__in=services.values(),
            issue__issue_group__code__in=groups.keys(),
        ).select_related('issue', 'issue__issue_group')

        current_service_issues = defaultdict(lambda: defaultdict(dict))
        for service_issue in current_issues:
            current_service_issues[service_issue.service_id][service_issue.issue.code] = service_issue

        target_issues = defaultdict(dict)
        for service_issues in serializer.data:
            group_code = service_issues['group_code']
            group = groups.get(group_code)
            if not group:
                raise BadRequest(f'No group with code {group_code} found')

            service_id = services[service_issues['service']]
            target_issues[service_id][group_code] = set()
            for item in service_issues['issues']:
                issue = issues.get(item['code'])
                if not issue:
                    raise BadRequest(f'No issue with code {item["code"]} found')
                if issue.code in target_issues[service_id][group_code]:
                    raise BadRequest(f'Got duplicate issue {item["code"]} for {service_id}')
                target_issues[service_id][group_code].add(issue.code)
                if issue.issue_group_id != groups[group_code].id:
                    raise BadRequest(
                        f'Issue {issue.code} not in expected group, '
                        f'expected {group_code}, actual {issue.issue_group.code}'
                    )

                current_issue = current_service_issues.get(service_id, {}).get(issue.code)
                if current_issue:
                    update_fields = []
                    for field in ('percentage_of_completion', 'context'):
                        value = item.get(field, None)
                        if value is not None and getattr(current_issue, field) != value:
                            setattr(current_issue, field, value)
                            update_fields.append(field)
                    if update_fields:
                        logger.info(f'Updating issue for {service_id}, {issue.id}')
                        current_issue.save(update_fields=update_fields)
                else:
                    logger.info(f'Creating issue for {service_id}, {issue.id}')
                    to_create.append(
                        ServiceIssue(
                            service_id=service_id,
                            issue_id=issue.id,
                            context=item['context'],
                            percentage_of_completion=item['percentage_of_completion'],
                            state=ServiceIssue.STATES.ACTIVE,
                        )
                    )

        for service_issue in current_issues:
            if (
                service_issue.issue.issue_group.code in target_issues[service_issue.service_id] and
                service_issue.issue.code not in target_issues[service_issue.service_id][service_issue.issue.issue_group.code]
            ):
                logger.info(f'Deleting issue for {service_issue.service_id}, {service_issue.issue.id}')
                to_delete.append(service_issue.id)

        if to_create:
            ServiceIssue.objects.bulk_create(to_create)
        if to_delete:
            ServiceIssue.objects.filter(pk__in=to_delete).delete()

        return Response(status=status.HTTP_204_NO_CONTENT)
