from plan import settings
from plan.resources.models import ResourceType, ServiceResource
from plan.roles.models import Role
from plan.services.models import Service


class BaseIssueFinder(object):
    def __init__(self, qs):
        self.qs = qs or Service.objects.active().exclude(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG)

    def __call__(self, *args, **kwargs):
        raise NotImplementedError()


class ResourceIssueFinder(BaseIssueFinder):
    issue_key = None
    resource_category = None

    def get_services_with_resource_managers(self):
        resource_management_roles_pks = Role.objects.get_resource_management_roles_pks(self.resource_category)
        return set(
            Service.objects
            # ToDO: добавить фильтр по is_active после https://st.yandex-team.ru/ABC-7426
            .filter(members__role_id__in=resource_management_roles_pks)
            .values_list('id', flat=True)
        )

    def get_services_with_resources(self):
        resource_suppliers_pks = Service.objects.get_resource_suppliers_pks(self.resource_category)
        resource_types = set(
            ResourceType.objects
            .filter(supplier_id__in=resource_suppliers_pks, is_enabled=True)
            .values_list('id', flat=True)
        )
        return (
            ServiceResource.objects
            .filter(state__in=ServiceResource.ALIVE_STATES, type_id__in=resource_types)
            .values_list('service_id', flat=True)
        )

    def __call__(self, *args, **kwargs):
        filtered_services = (
            self.qs
            .filter(id__in=self.get_services_with_resources())
            .values_list('id', flat=True)
            .order_by()
        )

        services_with_resource_managers = self.get_services_with_resource_managers()

        for service_id in filtered_services:
            if service_id not in services_with_resource_managers:
                yield service_id, {}, self.issue_key, 1
