from django.db.models import Count

from plan.services.models import Service
from plan.suspicion.constants import LEVELS
from plan.suspicion.issue_finders.base import BaseIssueFinder


class IssueFinder(BaseIssueFinder):
    def __call__(self, qs=None, *args, **kwargs):
        # Выберем сервисы, у которых есть хоть один ServiceTrafficStatus в состоянии CRIT
        # И сгруппируем их по parent
        crit_descendants = (
            Service.objects
            .filter(children__traffic_statuses__level=LEVELS.CRIT, children__state__in=Service.states.ACTIVE_STATES)
            .annotate(Count('children', distinct=True))
            .values_list('id', 'children__count')
        )
        crit_descendants_by_parent = {
            parent: children
            for parent, children
            in crit_descendants.iterator()
        }

        children_count = (
            Service.objects
            .active()
            .values('parent_id')
            .annotate(children=Count('parent_id'))
            .values_list('parent_id', 'children')
        )
        children_count_by_parent = {
            parent: children
            for parent, children
            in children_count.iterator()
        }

        for service in self.qs:
            children_count = children_count_by_parent.get(service.id, 0)
            crit_children_count = crit_descendants_by_parent.get(service.id, 0)
            if crit_children_count:
                percentage = crit_children_count / children_count
                yield service.id, {}, 'have_crit_descendants', percentage
