from collections import defaultdict
from django.conf import settings

from plan.services.models import Service
from plan.staff.models import Department, DepartmentStaff
from plan.suspicion.issue_finders.base import BaseIssueFinder


def get_ancestor_departments(department_id, departments_to_parent):
    parent_departments = set()
    while department_id:
        parent_departments.add(department_id)
        if departments_to_parent[department_id]:
            department_id = departments_to_parent[department_id]
        else:
            department_id = None
    return parent_departments


class IssueFinder(BaseIssueFinder):
    def __call__(self, qs=None, *args, **kwargs):
        search_portal_services_ids = (
            Service.objects.get(slug=settings.ABC_SEARCH_PORTAL_SLUG)
            .get_descendants(include_self=True)
            .values_list('id', flat=True)
        )

        services_list = (
            self.qs.exclude(id__in=search_portal_services_ids)
            .order_by('level')
            .values('is_base', 'parent_id', 'id', 'owner_id')
        )

        staff_id_to_departments = self._get_cached_departments()

        departments_of_base_services = dict()
        for service in Service.objects.filter(is_base=True):
            departments_of_base_services[service.id] = Department.objects.filter(
                services=service
            ).values_list('id', flat=True)

        id_to_base = {}
        for service in services_list:
            if service['is_base']:
                id_to_base[service['id']] = service
            else:
                id_to_base[service['id']] = id_to_base.get(service['parent_id'])

        departments_to_parent = dict(Department.objects.select_related('parent').values_list('id', 'parent_id'))

        for service in services_list:
            owner_departments = staff_id_to_departments[service['owner_id']]
            if not service['is_base'] and service['owner_id'] is not None:
                base_parent = id_to_base[service['id']]
                departments_of_service = {
                    ancestor
                    for owner_department in owner_departments
                    for ancestor in get_ancestor_departments(owner_department, departments_to_parent)
                }
                departments_of_base_parent = set()
                if base_parent is not None:
                    departments_of_base_parent = set(departments_of_base_services[base_parent['id']])
                if base_parent is None or (departments_of_base_parent and not departments_of_base_parent & departments_of_service):
                    yield service['id'], {}, 'wrong_base_parent', 1

    def _get_cached_departments(self):
        pairs = (
            DepartmentStaff.objects
            .filter(staff__in=self.qs.values_list('owner_id'))
            .values_list('staff_id', 'department_id')
        )
        result = defaultdict(set)
        for staff_id, department_id in pairs:
            result[staff_id].add(department_id)
        for staff_id, department_id in self.qs.values_list('owner_id', 'owner__department_id'):
            result[staff_id].add(department_id)
        return result
