import logging

from importlib import import_module
from itertools import chain

from django.contrib.postgres.fields import JSONField, ArrayField
from django.db import models
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _
from django_fsm import FSMField, transition
from django.db.models.signals import post_save, post_delete
from django.dispatch import receiver

from plan.services.models import Service, ServiceType, ServiceNotification
from plan.staff.models import Staff
from plan.suspicion.constants import LEVELS
from plan.suspicion.querysets import (
    IssueQueryset,
    ServiceIssueQueryset,
    ServiceAppealIssueQueryset,
    ExecutionStepQuerySet,
)
from plan.suspicion.constants import ServiceIssueStates, ServiceAppealIssueStates
from plan.suspicion.managers import ServiceExecutionActionManager, ServiceTrafficStatusManager

logger = logging.getLogger(__name__)


class Execution(models.Model):
    code = models.SlugField(verbose_name=_('Код'))
    name = models.TextField(verbose_name=_('Название'))
    name_en = models.TextField(verbose_name=_('Название на английском'))
    can_be_applied_to_base_service = models.BooleanField(
        default=False,
        verbose_name=_('Может применяться к базовым сервисам'),
    )
    is_active = models.BooleanField(default=False, verbose_name=_('Активен'))
    is_critical = models.BooleanField(default=False, verbose_name=_('Критичный'))

    class Meta:
        verbose_name = _('Тип воздействия')
        verbose_name_plural = _('Типы воздействий')

    def __str__(self):
        return 'ExecutionType %s' % self.code


class ExecutionStep(models.Model):
    execution = models.ForeignKey('Execution', related_name='steps', verbose_name=_('Тип'))
    apply_after = models.DurationField(verbose_name=_('Через сколько применяется'))
    is_active = models.BooleanField(default=False, verbose_name=_('Активен'))
    execution_chain = models.ForeignKey('ExecutionChain', related_name='steps', verbose_name=_('Цепочка наказаний'))

    objects = ExecutionStepQuerySet.as_manager()

    class Meta:
        order_with_respect_to = 'execution_chain'
        verbose_name = _('Шаг воздействия')
        verbose_name_plural = _('Шаги воздействия')

    def __str__(self):
        return 'Step of %s' % self.execution


class ExecutionChain(models.Model):
    code = models.SlugField(verbose_name=_('Код'))
    name = models.TextField(verbose_name=_('Название'))
    name_en = models.TextField(verbose_name=_('Название на английском'))

    class Meta:
        verbose_name = _('Цепочка наказаний')
        verbose_name_plural = _('Цепочки наказаний')

    def __str__(self):
        return 'ExecutionChain %s' % self.code


class IssueGroupThreshold(models.Model):
    issue_group = models.ForeignKey('IssueGroup', related_name='thresholds', verbose_name=_('Группа проблем'))
    level = models.CharField(max_length=255, verbose_name=_('Уровень'), choices=LEVELS.THRESHOLD_CHOICES)
    threshold = models.FloatField(verbose_name=_('Порог'))
    chain = models.ForeignKey('ExecutionChain', verbose_name=_('Цепочка наказаний'), null=True, blank=True)

    class Meta:
        verbose_name = _('Порог')
        verbose_name_plural = _('Пороги')

    def __str__(self):
        return 'ExecutionThreshold %s of %s' % (self.threshold, self.issue_group)


class Issue(models.Model):
    name = models.TextField(verbose_name=_('Название'))
    name_en = models.TextField(verbose_name=_('Название на английском'))
    issue_group = models.ForeignKey('IssueGroup', related_name='issues')
    code = models.SlugField(verbose_name=_('Код'), unique=True)
    allowed_for_types = models.ManyToManyField(
        ServiceType,
        blank=True,
        verbose_name=_('К каким типам сервисов применима'),
    )
    description = models.TextField(blank=True, default='', verbose_name=_('Описание'))
    description_en = models.TextField(blank=True, default='', verbose_name=_('Описание на английском'))
    recommendation = models.TextField(blank=True, default='', verbose_name=_('Рекомендации'))
    recommendation_en = models.TextField(blank=True, default='', verbose_name=_('Рекомендации на английском'))
    is_active = models.BooleanField(default=False, verbose_name=_('Активна'))
    can_be_appealed = models.BooleanField(default=False, verbose_name=_('Может быть оспорена'))
    weight = models.FloatField(verbose_name=_('Вес'))
    execution_chain = models.ForeignKey('ExecutionChain', null=True, blank=True, verbose_name=_('Цепочка наказаний'))
    objects = IssueQueryset.as_manager()

    def get_issue_finder(self):
        try:
            module = import_module('plan.suspicion.issue_finders.%s' % self.code)
            return module.IssueFinder
        except ModuleNotFoundError:
            logger.warning(f'No such suspicion module {self.code}')

    class Meta:
        verbose_name = _('Проблема')
        verbose_name_plural = _('Проблемы')

    def __str__(self):
        return 'Issue %s' % self.code


class IssueGroup(models.Model):
    name = models.TextField(verbose_name=_('Название'))
    name_en = models.TextField(verbose_name=_('Название на английском'))
    code = models.SlugField(verbose_name=_('Код'), unique=True)
    description = models.TextField(blank=True, default='', verbose_name=_('Описание'))
    description_en = models.TextField(blank=True, default='', verbose_name=_('Описание на английском'))
    recommendation = models.TextField(blank=True, default='', verbose_name=_('Рекомендации'))
    recommendation_en = models.TextField(blank=True, default='', verbose_name=_('Рекомендации на английском'))
    allowed_tvm_ids = ArrayField(models.IntegerField(), blank=True, null=True)
    manual_resolving = models.BooleanField(
        default=False,
        verbose_name=_('Не использовать автоматическую обработку проблем')
    )
    send_suggest = models.BooleanField(
        default=False,
        verbose_name=_('Отправлять письма по этой группе проблем')
    )

    class Meta:
        verbose_name = _('Группа проблем')
        verbose_name_plural = _('Группы проблем')

    def __str__(self):
        return 'IssueGroup %s' % self.code

    def get_thresholds_for_service(self, service):
        traffic_status = ServiceTrafficStatus.objects.get(service=service, issue_group=self)
        thresholds = IssueGroupThreshold.objects.filter(issue_group=self)
        active = None
        inactive = []
        for threshold in thresholds:
            if threshold.level == traffic_status.level:
                active = threshold
            else:
                inactive.append(threshold)
        return active, inactive


class Complaint(models.Model):
    author = models.ForeignKey(Staff, verbose_name=_('Автор жалобы'))
    created_at = models.DateTimeField(auto_now_add=True, verbose_name=_('Время поступления жалобы'), db_index=True)
    service = models.ForeignKey(
        Service,
        verbose_name=_('Сервис, на который поступила жалоба'),
        related_name='complaints',
    )
    message = models.TextField(verbose_name=_('Коментарий'))

    class Meta:
        verbose_name = _('Жалоба на сервис')
        verbose_name_plural = _('Жалобы на сервис')

    def __str__(self):
        return 'Complaint "%s" of user %s to service %s' % (self.message, self.author.login, self.service.name)


class ServiceIssue(models.Model):

    STATES = ServiceIssueStates

    CHOICES = ServiceIssueStates.CHOICES

    context = JSONField(null=True, blank=True, verbose_name=_('Контекст'))
    created_at = models.DateTimeField(auto_now_add=True, verbose_name=_('Время создания'))
    fixed_at = models.DateTimeField(null=True, blank=True, verbose_name=_('Время исправления'))
    state = models.CharField(max_length=255, verbose_name=_('Статус'), choices=CHOICES)
    service = models.ForeignKey(Service, verbose_name=_('Сервис'), related_name='service_issues')
    # Только одно из двух следующих полей должно быть заполнено
    issue = models.ForeignKey('Issue', null=True, blank=True, verbose_name=_('Проблема'))
    issue_group = models.ForeignKey(
        'IssueGroup',
        null=True,
        blank=True,
        verbose_name=_('Группа проблем'),
        related_name='service_issuegroup',
    )
    issue_action_key = models.TextField(null=True)
    expected_action_processing = models.BooleanField(
        default=True,
        verbose_name=_('Надо ли пересчитать связанные ServiceExecutionActions'),
    )
    percentage_of_completion = models.FloatField(default=1, verbose_name=_('Процент выполнения'))

    objects = ServiceIssueQueryset.as_manager()

    class Meta:
        verbose_name = _('Проблема сервиса')
        verbose_name_plural = _('Проблемы сервисов')
        unique_together = (('service', 'issue_group'), )

    def __str__(self):
        return 'ServiceIssue %s of %s' % (self.service, self.issue or self.issue_group)

    def get_issue_action_key(self):
        if self.issue_action_key:
            return self.issue_action_key
        elif self.issue_group:
            prefix = self.issue_group.code
            keys = (
                ServiceIssue.objects.filter(
                    service=self.service,
                    issue__in=self.issue_group.issues.all(),
                    issue_action_key__isnull=False,
                )
                .order_by('issue__code')
                .values_list('issue_action_key', flat=True)
            )
            return '_'.join(chain([prefix], keys))

    def get_weight_and_count(self, max_weight=None, service_issues_for_group=None):
        assert self.issue_group is not None

        if service_issues_for_group is None:
            service_issues_for_group = list(
                ServiceIssue.objects
                .problem()
                .filter(issue__issue_group_id=self.issue_group_id, service_id=self.service_id)
                .select_related('issue')
            )

        weight = sum(
            service_issue.issue.weight * service_issue.percentage_of_completion
            for service_issue
            in service_issues_for_group
        )
        count = len(service_issues_for_group)

        if max_weight is None:
            max_weight = sum(issue.weight for issue in self.issue_group.issues.active())

        return weight / max_weight if weight else weight, count

    def _activate_execution_chain(self, execution_chain):
        if execution_chain is None:
            return
        actions_to_unhold = ServiceExecutionAction.objects.held().filter(
            execution_chain=execution_chain,
            issue_action_key=self.get_issue_action_key(),
            service_issue__service=self.service,
        )
        if actions_to_unhold.exists():
            for action in actions_to_unhold:
                action.unhold(self)
        else:
            ServiceExecutionAction.objects.create_from_service_issue_and_chain(self, execution_chain)

    def _hold_or_delete_all_execution_actions(self):
        actions = self.execution_actions.not_applied()
        for execution_action in actions:
            execution_action.hold_or_delete()

    def _hold_or_delete_chain(self, execution_chain):
        if execution_chain is None:
            return
        actions = self.execution_actions.not_applied().filter(execution_chain=execution_chain)
        for execution_action in actions:
            execution_action.hold_or_delete()

    def _process_group_issue(self):
        active, inactive = self.issue_group.get_thresholds_for_service(self.service)
        if active:
            self._activate_execution_chain(active.chain)
        for threshold in inactive:
            self._hold_or_delete_chain(threshold.chain)

    def process_service_execution_actions(self):
        if self.state == self.STATES.ACTIVE:
            if self.issue:
                self._activate_execution_chain(self.issue.execution_chain)
            else:
                self._process_group_issue()
        else:
            # Тут могут быть только НЕ групповые ServiceIssue
            assert self.issue_group_id is None
            self._hold_or_delete_all_execution_actions()
        self.expected_action_processing = False
        self.save(update_fields=['expected_action_processing'])

    def change_state(self, new_state):
        self.state = new_state
        self.expected_action_processing = True
        if new_state == self.STATES.FIXED:
            self.fixed_at = timezone.now()
            self.save(update_fields=['state', 'expected_action_processing', 'fixed_at'])
        else:
            self.save(update_fields=['state', 'expected_action_processing'])


class ServiceExecutionAction(models.Model):
    service_issue = models.ForeignKey(
        'ServiceIssue',
        null=True,
        blank=True,
        related_name='execution_actions',
        verbose_name=_('Проблема сервиса'),
        on_delete=models.SET_NULL,
    )
    execution = models.ForeignKey('Execution', verbose_name=_('Наказание'), on_delete=models.CASCADE)
    execution_chain = models.ForeignKey(
        ExecutionChain,
        verbose_name=_('По какой цепочке было создано наказание'),
        on_delete=models.SET_NULL,
        null=True,
    )
    created_at = models.DateTimeField(auto_now_add=True)
    should_be_applied_at = models.DateTimeField(verbose_name=_('Время, когда должно быть применено'))
    applied_at = models.DateTimeField(null=True, blank=True, verbose_name=_('Время фактического применения'))
    held_at = models.DateTimeField(null=True, blank=True, verbose_name=_('Время помещения в холодильник'))
    issue_action_key = models.TextField(null=True, verbose_name=_('Ключ для сопоставления с ServiceIssue'))

    objects = ServiceExecutionActionManager()

    class Meta:
        verbose_name = _('Применение наказания')
        verbose_name_plural = _('Применения наказаний')

    def execute(self):
        execution = import_module('plan.suspicion.execution_tasks.' + self.execution.code).Execution()
        execution(self.service_issue.service)
        self.applied_at = timezone.now()
        self.save(update_fields=('applied_at',))

    def hold_or_delete(self):
        if self.issue_action_key is None:
            self.delete()
        elif self.held_at is None:
            self.held_at = timezone.now()
            self.save(update_fields=['held_at'])

    def unhold(self, service_issue):
        if self.held_at is not None:
            now = timezone.now()
            self.should_be_applied_at += now - self.held_at
            self.held_at = None
            # У одиночных service_issue открытие переоткрытие проблемы это создание нового ServiceIssue
            # Поэтому при разморозке нам надо привязать ServiceExecutionAction к новому ServiceIssue
            self.service_issue = service_issue
            self.save(update_fields=['held_at', 'should_be_applied_at', 'service_issue'])

    def __str__(self):
        return 'ServiceExecutionAction issue: %s, execution: %s' % (self.service_issue, self.execution)


@receiver(post_save, sender=ExecutionChain)
def update_service_execution_action_on_save_chain(sender, instance, created, **kwargs):
    ServiceExecutionAction.objects.held().filter(execution_chain=instance).delete()


@receiver(post_delete, sender=ExecutionChain)
def update_service_execution_action_on_delete_chain(sender, instance, **kwargs):
    ServiceExecutionAction.objects.not_applied().filter(execution_chain=instance).delete()


@receiver(post_save, sender=ExecutionStep)
def update_service_execution_action_on_save_step(sender, instance, created, **kwargs):
    ServiceExecutionAction.objects.held().filter(execution_chain=instance.execution_chain).delete()


@receiver(post_save, sender=ExecutionStep)
def update_service_execution_action_on_delete_step(sender, instance, **kwargs):
    ServiceExecutionAction.objects.not_applied().filter(execution_chain=instance.execution_chain).delete()


class ServiceTrafficStatus(models.Model):
    service = models.ForeignKey(Service, related_name='traffic_statuses', verbose_name=_('Сервис'))
    issue_group = models.ForeignKey('IssueGroup', verbose_name=_('Группа проблем'))
    issue_count = models.IntegerField(blank=True, default=0, verbose_name=_('Количество'))
    level = models.CharField(max_length=255, choices=LEVELS.CHOICES, default=LEVELS.OK, verbose_name=_('Уровень'))
    current_weight = models.FloatField(verbose_name=_('Текущий вес'), default=0)

    objects = ServiceTrafficStatusManager()

    class Meta:
        unique_together = ('service', 'issue_group')
        verbose_name = _('Статус группы проблем сервиса')
        verbose_name_plural = _('Статусы групп проблем сервисов')

    def __str__(self):
        return 'ServiceTrafficStatus service: %s, issue_group: %s, level: %s' % (
            self.service, self.issue_group, self.level
        )

    def get_new_level(self, value, issue_group_thresholds=None):
        # ToDo: возможно лучше кешировать в Redis (после его подключения)
        new_level = LEVELS.OK
        if not issue_group_thresholds:
            issue_group_thresholds = (
                IssueGroupThreshold.objects
                .filter(issue_group_id=self.issue_group_id)
                .order_by('threshold')
                .values_list('threshold', flat=True)
            )
        for threshold, level in zip(issue_group_thresholds, (LEVELS.WARN, LEVELS.CRIT)):
            if value > threshold:
                new_level = level

        return new_level

    def update_fields(self, **kwargs):
        update_fields = []
        for field_name, value in kwargs.items():
            if getattr(self, field_name) == value:
                continue
            setattr(self, field_name, value)
            update_fields.append(field_name)
        if update_fields:
            self.save(update_fields=update_fields)


class ServiceAppealIssue(models.Model):
    service_issue = models.ForeignKey('ServiceIssue', related_name='appeals', verbose_name=_('Проблема сервиса'))
    requester = models.ForeignKey(Staff, verbose_name=_('Автор апелляции'))
    approver = models.ForeignKey('staff.Staff', null=True, related_name='approved_appeals', verbose_name=_('Подтвердивший'))
    rejecter = models.ForeignKey(Staff, null=True, related_name='rejected_appeals', verbose_name=_('Отклонивший'))
    created_at = models.DateTimeField(auto_now_add=True, verbose_name=_('Когда запросил'))
    approved_at = models.DateTimeField(null=True, verbose_name=_('Когда подтвердилась'))
    rejected_at = models.DateTimeField(null=True, verbose_name=_('Когда отклонилась'))
    message = models.TextField(blank=True, default='', verbose_name=_('Комментарий'))

    objects = ServiceAppealIssueQueryset.as_manager()

    STATES = ServiceAppealIssueStates

    CHOICES = ServiceAppealIssueStates.CHOICES

    state = FSMField(
        verbose_name=_('Статус'),
        choices=CHOICES,
        default=STATES.REQUESTED,
    )

    def can_be_approved_by(self, person):
        return (
            person.staff.user.is_superuser or
            ServiceAppealIssue.objects.for_services_of_person(person).filter(pk=self.pk).exists()
        )

    def approve(self, approver):
        self.approve_transition(approver)
        self.save(update_fields=['approver', 'approved_at', 'state'])
        if self.requester != self.approver:
            self.send_mail_async(reason=ServiceNotification.APPEAL_APPROVED)

    @transition(field=state, source=(STATES.REQUESTED,),
                target=STATES.APPROVED)
    def approve_transition(self, approver):
        self.approver = approver
        self.approved_at = timezone.now()

    def reject(self, rejecter):
        self.reject_transition(rejecter)
        self.save(update_fields=['rejecter', 'rejected_at', 'state'])
        if self.requester != self.rejecter:
            self.send_mail_async(reason=ServiceNotification.APPEAL_REJECTED)

    @transition(field=state, source=(STATES.REQUESTED,),
                target=STATES.REJECTED)
    def reject_transition(self, rejecter):
        self.rejecter = rejecter
        self.rejected_at = timezone.now()

    def get_mail_context(self, reason):
        context = {
            'appeal': self,
            'service': self.service_issue.service,
            'issue': self.service_issue.issue,
        }
        if reason == ServiceNotification.APPEAL_APPROVED:
            context['decider'] = self.approver
            context['decision'] = _('согласовал')
        elif reason == ServiceNotification.APPEAL_REJECTED:
            context['decider'] = self.rejecter
            context['decision'] = _('отклонил')
        return context

    def send_mail_async(self, reason):
        # Для обхода цикличных импортов
        from plan.suspicion.tasks import send_mail_about_appeal
        send_mail_about_appeal.apply_async(kwargs={'appeal_pk': self.pk, 'reason': reason})
