from django.db import models
from django.utils import timezone
from django.db.models import Q, F


from plan.services.models import Service
from plan.suspicion.constants import ServiceIssueStates, ServiceAppealIssueStates


class IssueQueryset(models.QuerySet):
    def active(self):
        return self.filter(is_active=True)


class ServiceIssueQueryset(models.QuerySet):
    def alive(self):
        return self.filter(state__in=ServiceIssueStates.ALIVE_STATUSES)

    def problem(self):
        return self.filter(state__in=ServiceIssueStates.PROBLEM_STATUSES)

    def active(self):
        return self.filter(state=ServiceIssueStates.ACTIVE)

    def review(self):
        return self.filter(state=ServiceIssueStates.REVIEW)

    def appealed(self):
        return self.filter(state=ServiceIssueStates.APPEALED)

    def mark_fixed(self):
        self.update(
            state=ServiceIssueStates.FIXED,
            expected_action_processing=True,
            fixed_at=timezone.now(),
        )

    def expected_action_processing(self):
        return self.filter(expected_action_processing=True, service__state__in=Service.states.ACTIVE_STATES)

    def set_not_processed(self):
        self.update(expected_action_processing=True)

    def set_processed(self):
        self.update(expected_action_processing=False)

    def mark_appealed(self):
        self.update(
            state=ServiceIssueStates.APPEALED,
            expected_action_processing=True,
        )

    def mark_review(self):
        self.update(
            state=ServiceIssueStates.REVIEW,
            expected_action_processing=True,
        )

    def mark_active(self):
        self.update(
            state=ServiceIssueStates.ACTIVE,
            expected_action_processing=True,
        )

    def without_appeal(self):
        return self.exclude(appeals__state__in=ServiceAppealIssueStates.ACTIVE_STATES)

    def with_active_appeal(self):
        return self.filter(appeals__state=ServiceAppealIssueStates.REQUESTED)

    def with_approved_appeal(self):
        return self.filter(appeals__state=ServiceAppealIssueStates.APPROVED)

    def for_alive_services(self):
        return self.filter(service__state__in=Service.states.ALIVE_STATES)


class ServiceAppealIssueQueryset(models.QuerySet):
    def active(self):
        return self.filter(state=ServiceAppealIssueStates.REQUESTED)

    def dead(self):
        return self.exclude(state=ServiceAppealIssueStates.REQUESTED)

    def for_active_services(self):
        return self.filter(service_issue__service__state__in=Service.states.ACTIVE_STATES)

    def for_services_of_person(self, person, directly=False):
        straight_services_ids = (
            person.get_own_responsible_services().values_list('id', flat=True)
        )
        if directly:
            descendants = Service.objects.filter(parent_id__in=straight_services_ids)
        else:
            descendants = (
                Service._closure_model.objects
                .filter(~Q(parent=F('child')), parent__id__in=straight_services_ids)
                .values_list('child')
            )
        return self.filter(service_issue__service__in=descendants)


class ServiceExecutionActionQuerySet(models.QuerySet):
    def actions_for_execution(self):
        from plan.suspicion.models import ServiceIssue
        return self.filter(
            applied_at__isnull=True,
            should_be_applied_at__lt=timezone.now().replace(hour=0, minute=0, second=0, microsecond=0),
            service_issue__state=ServiceIssue.STATES.ACTIVE,
        )

    def held(self):
        return self.filter(held_at__isnull=False)

    def applied(self):
        return self.filter(applied_at__isnull=False)

    def not_applied(self):
        return self.filter(applied_at__isnull=True)

    def future(self):
        return self.filter(
            applied_at__isnull=True,
            should_be_applied_at__gte=timezone.now()
        )

    def past(self):
        return self.filter(
            applied_at__isnull=False,
            should_be_applied_at__lte=timezone.now()
        )

    def of_chain(self, chain):
        return self.filter(execution_chain=chain)


class ExecutionStepQuerySet(models.QuerySet):

    def active(self):
        return self.filter(is_active=True)

    def of_chain(self, chain):
        return self.filter(execution_chain=chain)
