from collections import defaultdict
from itertools import product
import logging

from django.db.models import Q
from django.utils import timezone

import plan.holidays.calendar
from plan import settings
from plan.celery_app import app
from plan.common.utils.tasks import lock_task
from plan.notify.shortcuts import deliver_email
from plan.services.exceptions import ServicesDontHaveActiveResponsibles
from plan.services.models import Service, ServiceNotification
from plan.services.tasks import send_notification_staffs
from plan.suspicion.constants import ServiceIssueStates, SUSPICION_DIGEST_MAIL_TEMPLATES
from plan.suspicion.models import (
    Issue,
    IssueGroup,
    ServiceAppealIssue,
    ServiceIssue,
    ServiceTrafficStatus,
    ServiceExecutionAction,
)
from plan.suspicion.tasks_helpers import IssueGroupsChecker
from plan.suspicion.constants import LEVELS


log = logging.getLogger(__name__)


@lock_task
def find_issues():
    service_to_issues_with_context = defaultdict(list)
    appealed_service_issues_with_context = defaultdict(list)
    service_to_issues = defaultdict(set)
    for service_issue in ServiceIssue.objects.problem().filter(
        issue__isnull=False,
        issue__issue_group__manual_resolving=False,
    ).select_related('issue'):
        service_to_issues[service_issue.service_id].add(service_issue.issue.code)
        service_to_issues_with_context[service_issue.service_id].append(
            (service_issue.issue.code, service_issue.context))
    for service_issue in ServiceIssue.objects.appealed().select_related('issue'):
        appealed_service_issues_with_context[service_issue.service_id].append(
            (service_issue.issue.code, service_issue.context))

    qs = Service.objects.alive().exclude(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG).order_by('-level')

    # ToDo: врменное решение, пока преобладают дискретные значения percentage (в большинстве случаев 0 или 1)
    #  поменять, если это вдруг изменится
    issue_by_percentage = defaultdict(lambda: defaultdict(list))
    for issue in Issue.objects.active().filter(issue_group__manual_resolving=False):
        issue_code = issue.code
        issue_finder_class = issue.get_issue_finder()
        if not issue_finder_class:
            log.warning(f'Skipping {issue.code}, no finder')
            for issues in service_to_issues.values():
                if issue_code in issues:
                    issues.remove(issue_code)
            continue

        issue_finder = issue_finder_class(qs)
        for service_id, context, issue_action_key, percentage in issue_finder():
            key = (issue_code, context)
            if key in service_to_issues_with_context[service_id]:
                service_to_issues[service_id].remove(issue_code)
                issue_by_percentage[issue_code][percentage].append(service_id)
            elif issue_code in service_to_issues[service_id]:
                ServiceIssue.objects.problem().filter(service_id=service_id, issue__code=issue_code).update(
                    context=context,
                    percentage_of_completion=percentage,
                )
                service_to_issues[service_id].remove(issue_code)
            # Создаем новый ServiceIssue только если такой же не был апеллирован в прошлом
            elif not (key in appealed_service_issues_with_context[service_id]):
                ServiceIssue.objects.create(
                    state=ServiceIssue.STATES.ACTIVE,
                    service_id=service_id,
                    issue=issue,
                    context=context,
                    percentage_of_completion=percentage,
                    issue_action_key=issue_action_key,
                )
    for issue_code, services_by_percentage in issue_by_percentage.items():
        # Обновим процент выполнения критерия
        update_service_issues_percentage(issue_code, services_by_percentage)

    for service_id, codes in service_to_issues.items():
        ServiceIssue.objects.filter(service_id=service_id, issue__code__in=codes).mark_fixed()

    ServiceIssue.objects.active().with_active_appeal().mark_review()
    ServiceIssue.objects.review().without_appeal().mark_active()
    ServiceIssue.objects.problem().with_approved_appeal().mark_appealed()


def update_service_issues_percentage(issue_code, services_by_percentage):
    for percentage, services_ids in services_by_percentage.items():
        need_update_issues = (
            ServiceIssue.objects
            .problem()
            .filter(service_id__in=services_ids, issue__code=issue_code)
            .exclude(percentage_of_completion=percentage)
        )
        need_update_issues.update(percentage_of_completion=percentage)


@lock_task
def check_issue_groups():
    checker = IssueGroupsChecker()
    checker.run()
    now = timezone.now()
    suspicious = Service.objects.filter(traffic_statuses__level__in=LEVELS.NOT_OK)
    not_suspicious = Service.objects.exclude(pk__in=suspicious)
    suspicious.filter(suspicious_date__isnull=True).update(suspicious_date=now)
    not_suspicious.filter(suspicious_date__isnull=False).update(suspicious_date=None)


@lock_task
def add_green_traffic():
    groups = IssueGroup.objects.values_list('id', flat=True)
    services = Service.objects.values_list('id', flat=True)
    group_service_pairs = set(product(groups, services))

    traffics = set(ServiceTrafficStatus.objects.values_list('issue_group', 'service'))

    ServiceTrafficStatus.objects.bulk_create(
        ServiceTrafficStatus(issue_group_id=group_id, service_id=service_id)
        for group_id, service_id in group_service_pairs - traffics
    )


@lock_task
def apply_executions():
    actions = ServiceExecutionAction.objects.actions_for_execution().exclude(
        service_issue__service__slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG
    )
    for action in actions:
        apply_execution.apply_async(args=[action.id])


@app.task
def apply_execution(action_id):
    ServiceExecutionAction.objects.get(id=action_id).execute()


@lock_task
def process_service_execution_actions():
    # Сначала переключаем светофоры помечая нужные ServiceIssue для групп
    check_issue_groups()
    # Удаляем ServiceExecutionAction для удаленных сервисов
    ServiceExecutionAction.objects.filter(
        service_issue__service__state=Service.states.DELETED,
        applied_at__isnull=True
    ).delete()
    # Замораживаем ServiceExecutionAction для закрытых сервисов
    ServiceExecutionAction.objects.filter(
        service_issue__service__state=Service.states.CLOSED,
        applied_at__isnull=True,
        held_at__isnull=True
    ).update(held_at=timezone.now())
    # Создаем/замораживаем/удаляем ServiceExecutionAction для изменившихся ServiceIssue
    for service_issue in ServiceIssue.objects.expected_action_processing():
        service_issue.process_service_execution_actions()
    # Удаляем ServiceExecutionAction, что залежались в холодильнике
    delete_action_threshold = timezone.now() - settings.MAX_SERVICE_EXECUTION_ACTION_HELD_TIME
    actions_to_delete = ServiceExecutionAction.objects.filter(held_at__lt=delete_action_threshold)
    log.info(
        'DELETING ServiceExecutionActions: %s',
        actions_to_delete.values('service_issue__service__slug', 'execution__code', 'service_issue__id')
    )
    actions_to_delete.delete()


APPEAL_MAIL_TEMPLATES_BY_REASON = {
    ServiceNotification.NEW_APPEAL_CREATED: 'notifications.services.suspicious.new_appeal_created',
    ServiceNotification.APPEAL_APPROVED: 'notifications.services.suspicious.decision_on_appeal',
    ServiceNotification.APPEAL_REJECTED: 'notifications.services.suspicious.decision_on_appeal',
}


@app.task(bind=True, max_retries=5)
def send_mail_about_appeal(self, appeal_pk, reason):
    """
    Таска отправки писем о событиях с аппеляцией
    :param appeal_pk: - id аппеляции, про которую отправляем письмо
    :param reason: - причина отправки письма
    """
    appeal = (
        ServiceAppealIssue.objects
        .filter(pk=appeal_pk)
        .select_related('service_issue__service', 'service_issue__issue', 'requester')
        .first()
    )
    if not appeal:
        log.error(
            'Try send email about %s appeal with pk:%s, but appeal with this pk does not exist',
            reason,
            appeal_pk,
        )
        raise self.retry(countdown=30, args=(appeal_pk, reason))  # ретраим через 30 сек
    notification_id = APPEAL_MAIL_TEMPLATES_BY_REASON.get(reason, None)
    if not notification_id:
        log.error(
            'Try send email about %s appeal with pk:%s, but mail template with this reason does not exist',
            reason,
            appeal_pk,
        )
        return
    recipients, cc = get_recipients_and_cc(appeal, reason)
    if not recipients:
        log.error(
            'Try send email about %s appeal with pk:%s, but recipients does not exist',
            reason,
            appeal_pk,
        )
        return
    service_notifications = [
        ServiceNotification(
            service_id=appeal.service_issue.service_id,
            recipient=recipient,
            notification_id=reason,
        )
        for recipient
        in recipients
    ]
    ServiceNotification.objects.bulk_create(service_notifications)
    deliver_email(
        notification_id=notification_id,
        context=appeal.get_mail_context(reason),
        recipients=[recipients],
        cc=cc,
    )
    new_notification_ids = (notification.id for notification in service_notifications)
    ServiceNotification.objects.filter(id__in=new_notification_ids).update(sent_at=timezone.now().date())


def get_recipients_and_cc(appeal, reason):
    recipients = None
    cc = None
    if reason == ServiceNotification.NEW_APPEAL_CREATED:
        try:
            recipients, _ = appeal.service_issue.service.get_responsibles_for_email(include_self=False)
            cc = [appeal.requester]
        except ServicesDontHaveActiveResponsibles:
            pass
    elif reason in {ServiceNotification.APPEAL_REJECTED, ServiceNotification.APPEAL_APPROVED}:
        recipients = [appeal.requester]
    return recipients, cc


def get_suspicious_context(all_notification):
    context = []
    result = {
        'has_traffic_status': False,
        'has_complaints_count': False,
        'context': context,
    }

    for notification in all_notification:
        context.append(
            {
                'service': notification.service,
                'traffic_status': notification.traffic_status,
                'complaints_count': notification.complaints_count,
            }
        )

        if not result['has_traffic_status'] and notification.traffic_status:
            result['has_traffic_status'] = True

        if not result['has_complaints_count'] and notification.complaints_count:
            result['has_complaints_count'] = True

    return result


@app.task
def suspicion_digest():
    today = timezone.now().date()
    # выбираем сервисы, для которых есть активные проблемы или новые жалобы
    seven_days_ago = today - timezone.timedelta(days=7)

    services = set(
        Service.objects
        .active()
        .filter(
            Q(
                service_issues__state__in=ServiceIssueStates.PROBLEM_STATUSES,
                service_issues__issue_group__isnull=True,
                service_issues__issue__issue_group__send_suggest=True,
            ) |
            Q(complaints__created_at__gte=seven_days_ago)
        )
        .without_notifications(ServiceNotification.SUSPICION_DIGEST, today)
        .prefetch_related(
            'service_issues',
            'service_issues__issue',
            'traffic_statuses',
            'traffic_statuses__issue_group'
        )
    )

    # формируем нотификации
    for service in services:
        (
            ServiceNotification.objects
            .create_suspicious_notification_with_recipient(
                service,
                ServiceNotification.SUSPICION_DIGEST,
                send_only_to_direct_responsibles=True
            )
        )

    # группируем нотификации по стаффу
    send_notification_staffs(
        ServiceNotification.SUSPICION_DIGEST,
        SUSPICION_DIGEST_MAIL_TEMPLATES,
        order_by='service__name',
    )


@lock_task
def send_suspicion_digest():
    today = timezone.now().date()
    list_workdays = plan.holidays.calendar.get_list_workdays(today, 1)

    if not plan.holidays.calendar.day_is_workdays(today, list_workdays):
        return

    # если сегодня понедельник и он не выходной/праздник, шлём дайджест
    elif today.weekday() == 0:
        suspicion_digest()

    # иначе, если ближайший понедельник - выходной/праздник, то
    # проверяем следующий рабочий день
    # если он позже, чем следующий понедельник, то отправляем сегодня
    else:
        next_monday = today + timezone.timedelta(days=-today.weekday(), weeks=1)
        if not plan.holidays.calendar.day_is_workdays(next_monday, list_workdays):
            next_workday = plan.holidays.calendar.next_workday(today)
            if next_workday > next_monday:
                suspicion_digest()
